/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.configuration.ox;

import org.apache.commons.logging.Log;
import org.osgi.service.component.ComponentContext;

import com.openexchange.config.ConfigurationService;
import com.openexchange.usm.configuration.ConfigurationManager;
import com.openexchange.usm.configuration.USMInvalidConfigurationException;
import com.openexchange.usm.journal.Journal;

public class ConfigurationManagerService implements ConfigurationManager {
	private Log _journal;
	private ConfigurationService _delegate;

	public String getProperty(String key, String defaultValue, boolean warningIfNotPresent)
			throws USMInvalidConfigurationException {
		return getStringProperty(key, defaultValue, warningIfNotPresent, false);
	}

	public String getPropertyAsString(String key) throws USMInvalidConfigurationException {
		return getStringProperty(key, null, false, true);
	}

	public boolean getProperty(String key, boolean defaultValue, boolean warningIfNotPresent)
			throws USMInvalidConfigurationException {
		return getBooleanProperty(key, defaultValue, warningIfNotPresent, false);
	}

	public boolean getPropertyAsBoolean(String key) throws USMInvalidConfigurationException {
		return getBooleanProperty(key, false, false, true);
	}

	public int getProperty(String key, int defaultValue, boolean warningIfNotPresent)
			throws USMInvalidConfigurationException {
		return getIntProperty(key, defaultValue, warningIfNotPresent, false);
	}

	public int getPropertyAsInt(String key) throws USMInvalidConfigurationException {
		return getIntProperty(key, 0, false, true);
	}

	public double getProperty(String key, double defaultValue, boolean warningIfNotPresent)
			throws USMInvalidConfigurationException {
		return getDoubleProperty(key, defaultValue, warningIfNotPresent, false);
	}

	public double getPropertyAsDouble(String key) throws USMInvalidConfigurationException {
		return getDoubleProperty(key, 0.0, false, true);
	}

	private String getStringProperty(String key, String defaultValue, boolean warningIfNotPresent,
			boolean exceptionIfNotPresent) throws USMInvalidConfigurationException {
		String value = _delegate.getProperty(key);
		if (value != null)
			return value;
		if (exceptionIfNotPresent)
			throw new USMInvalidConfigurationException(
					USMConfigurationOXBridgeErrorCodes.MISSING_STRING_PROPERTY_VALUE,
					"Missing string configuration property " + key);
		if (warningIfNotPresent)
			_journal.warn("Configuration property " + key + " not set, using default " + defaultValue);
		return defaultValue;
	}

	private boolean getBooleanProperty(String key, boolean defaultValue, boolean warningIfNotPresent,
			boolean exceptionIfNotPresent) throws USMInvalidConfigurationException {
		String stringValue = _delegate.getProperty(key);
		if (stringValue != null)
			return Boolean.parseBoolean(stringValue);
		if (exceptionIfNotPresent)
			throw new USMInvalidConfigurationException(
					USMConfigurationOXBridgeErrorCodes.MISSING_BOOLEAN_PROPERTY_VALUE,
					"Missing boolean configuration property " + key);
		if (warningIfNotPresent)
			_journal.warn("Configuration property " + key + " not set, using default " + defaultValue);
		return defaultValue;
	}

	private int getIntProperty(String key, int defaultValue, boolean warningIfNotPresent, boolean exceptionIfNotPresent)
			throws USMInvalidConfigurationException {
		String stringValue = _delegate.getProperty(key);
		if (stringValue == null) {
			if (exceptionIfNotPresent)
				throw new USMInvalidConfigurationException(
						USMConfigurationOXBridgeErrorCodes.MISSING_INTEGER_PROPERTY_VALUE,
						"Missing integer configuration property " + key);
			if (warningIfNotPresent)
				_journal.warn("Configuration property " + key + " not set, using default " + defaultValue);
			return defaultValue;
		}
		try {
			return Integer.parseInt(stringValue);
		} catch (NumberFormatException e) {
			_journal.error(USMConfigurationOXBridgeErrorCodes.INVALID_INT_PROPERTY_VALUE + 
					" Invalid value '" + stringValue + "' for integer configuration property " + key + ". Using default value instead!", e);
			return defaultValue;
		}
	}

	private double getDoubleProperty(String key, double defaultValue, boolean warningIfNotPresent,
			boolean exceptionIfNotPresent) throws USMInvalidConfigurationException {
		String stringValue = _delegate.getProperty(key);
		if (stringValue == null) {
			if (exceptionIfNotPresent)
				throw new USMInvalidConfigurationException(
						USMConfigurationOXBridgeErrorCodes.MISSING_DOUBLE_PROPERTY_VALUE,
						"Missing double configuration property " + key);
			if (warningIfNotPresent)
				_journal.warn("Configuration property " + key + " not set, using default " + defaultValue);
			return defaultValue;
		}
		try {
			return Double.parseDouble(stringValue);
		} catch (NumberFormatException e) {
		    _journal.error(USMConfigurationOXBridgeErrorCodes.INVALID_DOUBLE_PROPERTY_VALUE + " Invalid value '" + stringValue
                            + "' for double configuration property " + key + ". Using default value instead!", e);
			return defaultValue;
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
	 */
	public void activate(ComponentContext context) {
		_journal = ((Journal) context.locateService(Journal.class.getName())).getLog(ConfigurationManager.class);
		_delegate = (ConfigurationService) context.locateService(ConfigurationService.class.getName());
		_journal.info("USM-OX configuration bridge activated");
	}

	/*
	 * (non-Javadoc)
	 * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
	 */
	public void deactivate(ComponentContext context) {
		_delegate = null;
		_journal.info("USM-OX configuration bridge deactivated");
		_journal = null;
	}
}
