/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.sync;

import java.util.concurrent.ConcurrentHashMap;
import com.openexchange.usm.session.impl.SessionID;
import com.openexchange.usm.session.impl.SessionImpl;

/**
 * {@link SynchronizationLock}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public final class SynchronizationLock {

    private static class LockKey {

        private final SessionID _sessionID;

        private final String _id;

        public LockKey(SessionImpl session, String id) {
            _sessionID = session.getCompleteSessionID().getSessionID();
            _id = id;
        }

        @Override
        public int hashCode() {
            return _sessionID.hashCode() * 31 + _id.hashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (!(obj instanceof LockKey))
                return false;
            LockKey other = (LockKey) obj;
            return _id.equals(other._id) && _sessionID.equals(other._sessionID);
        }
    }

    private static final ConcurrentHashMap<LockKey, String> _lockedKeys = new ConcurrentHashMap<LockKey, String>(16, 0.75f, 1);

    /**
     * Tries to acquire a lock for a given ID in a given session. Locks are acquired per session per ID. For different sessions and/or IDs a
     * different lock is used. If acquiring the lock is successful, returns null. If it is not successful, returns the "acquirer" text that
     * was stored when the lock was acquired by someone else.
     * 
     * @param session Session for which to acquire the lock
     * @param id ID for which to acquire the lock
     * @param acquirer Text that will be stored with the lock (anybody else that fails to acquire the lock while it is held will get this
     *            String which may contain some description who/why this lock was acquired)
     * @return null if lock was acquired successfully, String stored as acquirer by the code that currently holds the lock if the lock is
     *         currently not available
     */
    public static String lock(SessionImpl session, String id, String acquirer) {
        return _lockedKeys.putIfAbsent(new LockKey(session, id), acquirer);
    }

    /**
     * Releases a previously acquired lock. No security check is made if the lock is actually currently locked. After the call the
     * session/ID combination is available for locking.
     * 
     * @param session
     * @param id
     */
    public static void unlock(SessionImpl session, String id) {
        _lockedKeys.remove(new LockKey(session, id));
    }
}
