/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.impl;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;
import org.apache.commons.logging.Log;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.management.ManagementService;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;
import com.openexchange.usm.api.USMVersion;
import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.ContentTypeManager;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.FolderContentType;
import com.openexchange.usm.api.database.DatabaseAccess;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMAccessDeniedException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.exceptions.USMStartupException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.SessionInitializer;
import com.openexchange.usm.api.session.SessionManager;
import com.openexchange.usm.configuration.ConfigurationManager;
import com.openexchange.usm.configuration.ConfigurationProperties;
import com.openexchange.usm.configuration.USMInvalidConfigurationException;
import com.openexchange.usm.mapping.ObjectIdMappingService;
import com.openexchange.usm.ox_event.OXEventListener;
import com.openexchange.usm.ox_event.OXEventManager;
import com.openexchange.usm.ox_json.OXJSONAccess;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.FolderImpl;
import com.openexchange.usm.session.sync.IncrementalContentSyncer;
import com.openexchange.usm.session.sync.OXDataCache;
import com.openexchange.usm.session.sync.SlowContentSyncer;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.Toolkit;
import com.openexchange.usm.uuid.OXObjectID;
import com.openexchange.usm.uuid.UUIDMappingService;
import com.openexchange.usm.uuid.exceptions.OXObjectAlreadyMappedException;
import com.openexchange.usm.uuid.exceptions.UUIDAlreadyMappedException;

public class SessionManagerImpl implements SessionManager, OXEventListener {

    // Stores interval after which an automatic access check is performed.
    // LATER we might want to make this constant configurable
    private final static long ACCESS_CHECK_INTERVAL = 300000L;

    //private final static long SESSION_CACHE_CHECK_INTERVAL = 5000L;

    private final static long ONE_DAY_IN_MILLISECONDS = 24L * 60L * 60L * 1000L;

    public static final String LAST_ACCESS_FIELD_NAME = "USM_ACCESS";

    private final static List<String> SPECIAL_FOLDERS = new ArrayList<String>();

    private final static List<String> SPECIAL_FOLDERS_VIEW = Collections.unmodifiableList(SPECIAL_FOLDERS);

    static {
        SPECIAL_FOLDERS.add(DefaultContentTypes.GROUPS_ID);
        SPECIAL_FOLDERS.add(DefaultContentTypes.RESOURCES_ID);
    }

    private OXJSONAccess _oxAjaxAccess;

    private Log _journal;

    private ContentTypeManager _contentTypeManager;

    private FolderContentType _folderContentType;

    private SessionStorage _sessionStorage;

    private OXDataCache _dataCache;

    private OXEventManager _oxEventManager;

    private ConfigurationManager _configurationManager;

    private DatabaseAccess _databaseAccess;

    private ObjectIdMappingService _mappingService;

    private UUIDMappingService _uuidService;

    private ManagementService _managementService;

    private final USMCacheInformation _usmCacheInformation = new USMCacheInformation(this);

    private ObjectName _jmxSessionObjectName;

    private String _usmInterface;

    // A value < 0 indicates percent, >= 0 an absolute value
    private int _waitForChangesEmailPullDelay;

    private int _waitForChangesEmailMinPullDelay = -1;

    private int _maxSyncStatesInDB = ConfigurationProperties.SYNC_MAX_STATES_IN_DB_DEFAULT;

    private boolean _keepLastSentSyncStateInDB = ConfigurationProperties.SYNC_KEEP_LAST_SENT_STATE_IN_DB_DEFAULT;

    private long _sessionStorageTimeLimit = ConfigurationProperties.USM_SESSION_STORAGE_TIME_LIMIT_DEFAULT * ONE_DAY_IN_MILLISECONDS;

    private TimerService _timerService;

    private ScheduledTimerTask _cleanupTask;

    private ScheduledTimerTask _sessionCacheCleanupTask;
    
    private ScheduledTimerTask _sessionCacheCleanupLogInfoTask;

    private int _syncRetryCount;

    private boolean _ignoreClientDataAtSlowSync;

    private boolean _forceNewSyncKeyOnEmptySync;

    private int _sessionCacheTimeout;

    private int _syncStateCacheTimeout;

    public SessionManagerImpl() {
    }

    /**
     * This constructor is used only for tests of the IncrementalContentSyncer and SlowContentSyncer.
     * 
     * @param log
     */
    public SessionManagerImpl(Log log) {
        _journal = log;
        _dataCache = new OXDataCache(0, 0);
    }

    public void activate(Log log, OXJSONAccess ajaxAccess, ContentTypeManager contentTypeManager, FolderContentType folderContentType, OXEventManager oxEventManager, ConfigurationManager configurationManager, DatabaseAccess dbAccess, ObjectIdMappingService mappingService, UUIDMappingService uuidService, TimerService timerService) {
        _journal = log;
        _oxAjaxAccess = ajaxAccess;
        _contentTypeManager = contentTypeManager;
        _folderContentType = folderContentType;
        _oxEventManager = oxEventManager;
        _configurationManager = configurationManager;
        _databaseAccess = dbAccess;
        _mappingService = mappingService;
        _uuidService = uuidService;
        _timerService = timerService;
        try {
            _syncRetryCount = _configurationManager.getProperty(
                ConfigurationProperties.SYNC_CONCURRENT_MODIFICATION_MAX_RETRIES_PROPERTY,
                ConfigurationProperties.SYNC_CONCURRENT_MODIFICATION_MAX_RETRIES_DEFAULT,
                false);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR,
                "Configuration for Sync retries is invalid",
                e);
        }
        _waitForChangesEmailPullDelay = readEmailPullDelayProperty();
        _waitForChangesEmailMinPullDelay = readEmailMinPullDelayProperty();
        try {
            _maxSyncStatesInDB = _configurationManager.getProperty(
                ConfigurationProperties.SYNC_MAX_STATES_IN_DB_PROPERTY,
                ConfigurationProperties.SYNC_MAX_STATES_IN_DB_DEFAULT,
                false);
            if (_maxSyncStatesInDB < 2)
                throw new USMStartupException(
                    USMSessionManagementErrorCodes.CONFIGURATION_ERROR_5,
                    "Configuration for max_states_in_db must be greater or equal to 2");
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_6,
                "Configuration for max_states_in_db is invalid",
                e);
        }
        try {
            _keepLastSentSyncStateInDB = _configurationManager.getProperty(
                ConfigurationProperties.SYNC_KEEP_LAST_SENT_STATE_IN_DB_PROPERTY,
                ConfigurationProperties.SYNC_KEEP_LAST_SENT_STATE_IN_DB_DEFAULT,
                false);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_7,
                "Configuration for keep_last_state_in_db is invalid",
                e);
        }

        try {
            _usmInterface = _configurationManager.getProperty(
                ConfigurationProperties.USM_INTERFACE_CONFIG_PROPERTY,
                ConfigurationProperties.USM_INTERFACE_CONFIG_DEFAULT,
                true);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_8,
                "Configuration for usm interface module is invalid",
                e);
        }

        try {
            _sessionStorageTimeLimit = _configurationManager.getProperty(
                ConfigurationProperties.USM_SESSION_STORAGE_TIME_LIMIT_PROPERTY,
                ConfigurationProperties.USM_SESSION_STORAGE_TIME_LIMIT_DEFAULT,
                false) * ONE_DAY_IN_MILLISECONDS;
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_9,
                "Invalid configuration for " + ConfigurationProperties.USM_SESSION_STORAGE_TIME_LIMIT_PROPERTY,
                e);
        }

        int cacheTimeLimitSyncComplete;
        int cacheTimeLimitSyncIncomplete;
        try {
            cacheTimeLimitSyncComplete = _configurationManager.getProperty(
                ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_COMPLETE_PROPERTY,
                ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_COMPLETE_DEFAULT,
                false);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_10,
                "Invalid configuration for " + ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_COMPLETE_PROPERTY,
                e);
        }
        try {
            cacheTimeLimitSyncIncomplete = _configurationManager.getProperty(
                ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_INCOMPLETE_PROPERTY,
                ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_INCOMPLETE_DEFAULT,
                false);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_11,
                "Invalid configuration for " + ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_INCOMPLETE_PROPERTY,
                e);
        }

        try {
            _ignoreClientDataAtSlowSync = _configurationManager.getProperty(
                ConfigurationProperties.USM_IGNORE_CLIENT_DATA_AT_SLOW_SYNC,
                ConfigurationProperties.USM_IGNORE_CLIENT_DATA_AT_SLOW_SYNC_DEFAULT,
                false);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_12,
                "Invalid configuration for " + ConfigurationProperties.USM_IGNORE_CLIENT_DATA_AT_SLOW_SYNC,
                e);
        }

        try {
            _forceNewSyncKeyOnEmptySync = _configurationManager.getProperty(
                ConfigurationProperties.USM_FORCE_NEW_SYNC_KEY_ON_EMPTY_SYNC,
                ConfigurationProperties.USM_FORCE_NEW_SYNC_KEY_ON_EMPTY_SYNC_DEFAULT,
                false);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_13,
                "Invalid configuration for " + ConfigurationProperties.USM_FORCE_NEW_SYNC_KEY_ON_EMPTY_SYNC,
                e);
        }
        try {
            _sessionCacheTimeout = _configurationManager.getProperty(
                ConfigurationProperties.USM_CACHE_SESSION_TIMEOUT_PROPERTY,
                ConfigurationProperties.USM_CACHE_SESSION_TIMEOUT_DEFAULT,
                false);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_14,
                "Invalid configuration for " + ConfigurationProperties.USM_CACHE_SESSION_TIMEOUT_PROPERTY,
                e);
        }
        try {
            _syncStateCacheTimeout = _configurationManager.getProperty(
                ConfigurationProperties.USM_CACHE_SYNC_STATE_TIMEOUT_PROPERTY,
                ConfigurationProperties.USM_CACHE_SYNC_STATE_TIMEOUT_DEFAULT,
                false);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_15,
                "Invalid configuration for " + ConfigurationProperties.USM_CACHE_SYNC_STATE_TIMEOUT_PROPERTY,
                e);
        }

        _sessionStorage = new SessionStorage(this);
        _dataCache = new OXDataCache(cacheTimeLimitSyncComplete, cacheTimeLimitSyncIncomplete);
        _oxEventManager.addEventListener(this);
        setTimerService(_timerService); // Initialize timers if TimerService was already set

        _journal.info("USM SessionManager activated");
        _journal.info("Open-Xchange USM Version " + USMVersion.VERSION + " (Build " + USMVersion.BUILD_NUMBER + ") successfully started");
    }

    public UUID getContextUUID(int cid) throws DatabaseAccessException, USMSQLException {
        return _uuidService.getContextUUID(cid);
    }

    public void deactivate() {
        deactivateTimers();
        _oxEventManager.removeEventListener(this);
        _sessionStorage.shutdown(_oxAjaxAccess);
        _configurationManager = null;
        _oxEventManager = null;
        _dataCache = null;
        _sessionStorage = null;
        _contentTypeManager = null;
        _oxAjaxAccess = null;
        _databaseAccess = null;
        _mappingService = null;
        _waitForChangesEmailPullDelay = 0;
        _waitForChangesEmailMinPullDelay = -1;
        _journal.info("USM SessionManager deactivated");
        _journal = null;
    }

    public Session getSession(String user, String password, String protocol, String device) throws USMException {
        return getSession(user, password, protocol, device, null, "127.0.0.1", new HashMap<String, String>());
    }

    // FIXME This code must be modified to use a more efficient approach. Iterating over all context ids takes way too long for large installations (e.g. 2 mio contexts)
    // FIXME E.g. first check if USM session exists in memory and determine context id from it. if it doesn't exist in memory, try to login to OX server, and determine
    // FIXME context id after login was successful. Finally, perform database check only on previously determined context id
    public boolean existsSessionInDB(String user, String protocol, String device) throws USMException {
        SessionID id = new SessionID(user, protocol, device);
        SessionImpl session = _sessionStorage.getSession(id);
        if (session != null)
            return session.checkProperDBStorage();
        
        List<Integer> contextIds = _databaseAccess.getAllContextIds();
        for (Integer contextId : contextIds) {
            EncapsulatedConnection con = _databaseAccess.getReadOnly(contextId);
            PreparedStatement statement = null;
            boolean sessionFound = false;
            try {
                statement = con.prepareStatement("select usmSessionId from USMSession where " + PersistentSessionDataSQLStatements.DEVICE_IN_CONTEXT);
                statement.setInt(1, contextId);
                statement.setString(2, device);
                statement.setString(3, protocol);
                ResultSet result = statement.executeQuery();
                sessionFound = result.next();
            } catch (SQLException e) {
                String errorMessage = " SQL error while retrieving session from DB";
                throw new USMSQLException(USMSessionManagementErrorCodes.PERSISTENT_SESSION_DB_ERROR_2, errorMessage, e);
            } finally {
                Toolkit.close(statement);
                Toolkit.close(con);
            }
            if (sessionFound)
                return true;
        }
        return false;
    }

    public Session getSession(String user, String password, String protocol, String device, SessionInitializer initializer, String clientIP, Map<String, String> xHeaders) throws USMException {
        SessionID id = new SessionID(user, protocol, device);
        SessionImpl session = _sessionStorage.getSession(id);
        if (session != null) {
            if (session.checkProperDBStorage()) {
                if (!session.getPassword().equals(password)) {
                    session.setPassword(password);
                    session.setClientIp(clientIP);
                    _oxAjaxAccess.login(session);
                }
                try {
                    recheckUSMAccess(initializer, session);
                    return session;
                } catch (USMAccessDeniedException e) {
                    _sessionStorage.removeSession(session);
                    throw e;
                }
            }
            // Cached Session is no longer present in DB, start from scratch
            _sessionStorage.removeSession(session);
        }
        session = new SessionImpl(this, user, password, protocol, device);
        session.setClientIp(clientIP);
        session.setXRequestHeaders(xHeaders);
        _oxAjaxAccess.login(session);
        JSONObject configuration = _oxAjaxAccess.getConfiguration(session);
        checkUSMAccess(session, configuration);
        session.setLastAccessCheck(System.currentTimeMillis());
        session.initialize(initializer, configuration);
        _sessionStorage.storeSession(session);
        _usmCacheInformation.newSessionCreated();
        session.setPersistentField(LAST_ACCESS_FIELD_NAME, String.valueOf(System.currentTimeMillis()));
        return session;
    }

    private void recheckUSMAccess(SessionInitializer initializer, SessionImpl session) throws USMException {
        long now = System.currentTimeMillis();
        if (session.getLastAccessCheck() < now - ACCESS_CHECK_INTERVAL) {
            checkUSMAccess(session, null);
            initializer.checkAccess(session);
            session.setLastAccessCheck(now);
        }
    }

    private void checkUSMAccess(SessionImpl session, JSONObject configuration) throws USMAccessDeniedException, AuthenticationFailedException, OXCommunicationException {
        if (_usmInterface == null || _usmInterface.length() == 0) {
            JSONObject usmModule = getUSMModule(session, configuration, "com.openexchange.usm");
            if (usmModule != null && usmModule.optBoolean("active"))
                return;
        } else {
            JSONObject oxInterfaces = getUSMModule(session, configuration, "interfaces");
            if (oxInterfaces != null && oxInterfaces.optBoolean(_usmInterface))
                return;
        }
        _oxAjaxAccess.logout(session);
        throw new USMAccessDeniedException(USMSessionManagementErrorCodes.USM_ACCESS_DISABLED, "USM access disabled for user");
    }

    private JSONObject getUSMModule(SessionImpl session, JSONObject configuration, String module) throws AuthenticationFailedException {
        if (configuration != null)
            return JSONToolkit.getJSONObject(configuration, "modules", module);
        try {
            return _oxAjaxAccess.getConfiguration(session, "modules", module);
        } catch (OXCommunicationException e) {
            return null;
        }
    }

    public OXJSONAccess getOXAjaxAccess() {
        return _oxAjaxAccess;
    }

    public Log getJournal() {
        return _journal;
    }

    public ContentTypeManager getContentTypeManager() {
        return _contentTypeManager;
    }

    public FolderContentType getFolderContentType() {
        return _folderContentType;
    }

    public DatabaseAccess getDatabaseAccess() {
        return _databaseAccess;
    }

    public ObjectIdMappingService getMappingService() {
        return _mappingService;
    }

    public int getMaxSyncStatesInDB() {
        return _maxSyncStatesInDB;
    }

    public boolean isKeepLastSentSyncStateInDB() {
        return _keepLastSentSyncStateInDB;
    }

    public void removeSessionFromStorage(SessionImpl session) {
        _sessionStorage.removeSession(session);
    }

    public boolean isIgnoreClientDataAtSlowSync() {
        return _ignoreClientDataAtSlowSync;
    }

    // --- Interface OXEventListener ---

    public void folderChanged(int contextId, int userId, String folderId, long timestamp) {
        SessionID[] sessions = _sessionStorage.getActiveSessions(new UserID(contextId, userId));
        if (sessions != null) {
            if (timestamp == 0)
                timestamp = System.currentTimeMillis();
            for (SessionID session : sessions) {
                SessionImpl s = _sessionStorage.getSession(session);
                if (s != null)
                    s.foldersChanged(timestamp);
            }
        }
    }

    public void folderContentChanged(int contextId, int userId, String folderId, long timestamp) {
        SessionID[] sessions = _sessionStorage.getActiveSessions(new UserID(contextId, userId));
        if (sessions != null) {
            if (timestamp == 0)
                timestamp = System.currentTimeMillis();
            for (SessionID session : sessions) {
                SessionImpl s = _sessionStorage.getSession(session);
                if (s != null) {
                    if (folderId == null)
                        s.folderContentChanged(timestamp);
                    else
                        s.folderContentChanged(timestamp, folderId);
                }
            }
        }
    }

    public boolean isEmailPullEnabled() {
        return _waitForChangesEmailMinPullDelay >= 0;
    }

    public long computePullTime(long start, long end) {
        long result = start + _waitForChangesEmailMinPullDelay;
        // If disabled or minimum wait time greater than interval, return -1 to indicate no email pull
        if (_waitForChangesEmailMinPullDelay < 0 || result > end)
            return -1L;
        // Compute pulltime as percentage of interval or absolute delay from start
        long pullTime = (_waitForChangesEmailPullDelay < 0) ? (start - (end - start) * _waitForChangesEmailPullDelay / 100L) : (start + _waitForChangesEmailPullDelay);
        // return at least minimum wait time, at most normal end time
        return Math.min(Math.max(result, pullTime), end);
    }

    // internal methods

    private int readEmailMinPullDelayProperty() {
        try {
            return _configurationManager.getProperty(
                ConfigurationProperties.SYNC_EMAIL_PULL_MIN_DELAY_PROPERTY,
                ConfigurationProperties.SYNC_EMAIL_PULL_MIN_DELAY_DEFAULT,
                true);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_4,
                "Invalid configuration value for " + ConfigurationProperties.SYNC_EMAIL_PULL_MIN_DELAY_PROPERTY,
                e);
        }
    }

    private int readEmailPullDelayProperty() {
        try {
            String pullValue = _configurationManager.getProperty(
                ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_PROPERTY,
                ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_DEFAULT,
                true);
            if (pullValue.endsWith("%"))
                return -Integer.parseInt(pullValue.substring(0, pullValue.length() - 1));
            return Integer.parseInt(pullValue);
        } catch (NumberFormatException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_2,
                "Invalid configuration value for " + ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_PROPERTY,
                e);
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(
                USMSessionManagementErrorCodes.CONFIGURATION_ERROR_3,
                "Invalid configuration value for " + ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_PROPERTY,
                e);
        }
    }

    public void storeUUID(DataObject o) throws DatabaseAccessException, USMSQLException, UUIDAlreadyMappedException, OXObjectAlreadyMappedException {
        int contentTypeCode = o.getContentType().getCode();
        if (contentTypeCode != DefaultContentTypes.MAIL_CODE) {
            try {
                int objectId = Integer.parseInt(o.getID());
                int folderOwner = o.getParentFolderOwnerID();
                if (folderOwner == o.getSession().getUserIdentifier())
                    folderOwner = 0;
                _uuidService.storeMapping(getModifedUUID(o.getUUID(), folderOwner), new OXObjectID(
                    o.getSession().getContextId(),
                    contentTypeCode,
                    objectId));
            } catch (NumberFormatException ignored) {
                // For all cases in which we have no integer OX object id (e.g. mail folders), only store UUIDs locally in sync states
            }
        }
    }

    public void storeUUID(int cid, ContentType contentType, int objectId, UUID uuid, int folderOwner) throws DatabaseAccessException, USMSQLException, UUIDAlreadyMappedException, OXObjectAlreadyMappedException {
        OXObjectID oxObjectId = new OXObjectID(cid, contentType.getCode(), objectId);
        _uuidService.storeMapping(getModifedUUID(uuid, folderOwner), oxObjectId);
    }

    // public void storeOrRemapUUID(int cid, ContentType contentType, int objectId, UUID uuid)
    // throws DatabaseAccessException, USMSQLException, UUIDAlreadyMappedException, OXObjectAlreadyMappedException {
    // OXObjectID oxObjectId = new OXObjectID(cid, contentType.getCode(), objectId);
    // _uuidService.storeMapping(uuid, oxObjectId); //TODO: implement method in the uuid service
    // }

    public DataObjectSet insertStoredUUID(DataObject o, DataObjectSet oldData, DataObjectSet newestCachedData) throws DatabaseAccessException, USMSQLException {
        boolean useModifiedUUID = o.getSession().getUserIdentifier() != o.getParentFolderOwnerID();
        if (oldData != null) {
            DataObject oldObject = oldData.get(o.getID());
            if (oldObject != null) {
                o.setUUID(useModifiedUUID ? getModifedUUID(oldObject.getUUID(), o.getParentFolderOwnerID()) : oldObject.getUUID());
                return newestCachedData;
            }
        }
        int contentTypeCode = o.getContentType().getCode();
        switch (contentTypeCode) {
        case DefaultContentTypes.FOLDER_CODE:
            if (newestCachedData == null)
                newestCachedData = new DataObjectSet(o.getSession().getCachedFolders());
            DataObject cachedFolder = newestCachedData.get(o.getID());
            if (cachedFolder != null)
                o.setUUID(cachedFolder.getUUID());
            break;
        case DefaultContentTypes.MAIL_CODE:
            if (newestCachedData == null)
                // TODO Optimize so that newest cached data is only retrieved if old data is not already newest cached data ?
                newestCachedData = new DataObjectSet(o.getSession().getCachedFolderElements(o.getParentFolderID(), o.getContentType()));
            DataObject cachedMail = newestCachedData.get(o.getID());
            if (cachedMail != null)
                o.setUUID(useModifiedUUID ? getModifedUUID(cachedMail.getUUID(), o.getParentFolderOwnerID()) : cachedMail.getUUID());
            break;
        default:
            try {
                int objectId = Integer.parseInt(o.getID());
                UUID uuid = _uuidService.getUUID(o.getSession().getContextId(), contentTypeCode, objectId);
                o.setUUID(useModifiedUUID ? getModifedUUID(uuid, o.getParentFolderOwnerID()) : uuid);
            } catch (NumberFormatException ignored) {
            }
            break;
        }
        return newestCachedData;
    }

    public UUID getUUID(Session session, int contentTypeCode, int objectId) throws DatabaseAccessException, USMSQLException {
        return _uuidService.getUUID(session.getContextId(), contentTypeCode, objectId);
    }

    public int getMappedObject(Session session, int contentTypeCode, UUID uuid) throws DatabaseAccessException, USMSQLException {
        OXObjectID oxObjectId = _uuidService.getMappedObject(session.getContextId(), uuid);
        if (oxObjectId != null && oxObjectId.getContentType() == contentTypeCode)
            return oxObjectId.getObjectId();
        else
            return 0;
    }

    public void objectDeleted(int contextID, int objectID, int contentType) {
        OXObjectID oxObjectId = new OXObjectID(contextID, contentType, objectID);
        try {
            _uuidService.removeMapping(contextID, oxObjectId);
        } catch (DatabaseAccessException e) {
            _journal.error("Database could not be accessed to delete UUID mapping for " + oxObjectId);
        } catch (USMSQLException e) {
            _journal.error("SQL error occurred while deleting UUID mapping for " + oxObjectId);
        }
    }

    public synchronized void setManagementService(ManagementService managementService) {
        if (_managementService != null) {
            if (_jmxSessionObjectName != null) {
                try {
                    _managementService.unregisterMBean(_jmxSessionObjectName);
                } catch (final OXException e) {
                    _journal.error(e.getMessage(), e);
                } finally {
                    _jmxSessionObjectName = null;
                    _managementService = null;
                }
            }
        }
        _managementService = managementService;
        if (managementService != null) {
            try {
                _jmxSessionObjectName = getObjectName(USMSessionInformation.class.getName(), USMSessionInformationMBean.SESSION_DOMAIN);
                _managementService.registerMBean(_jmxSessionObjectName, new USMSessionInformation(this, _usmCacheInformation));
            } catch (final MalformedObjectNameException e) {
                _journal.error(e.getMessage(), e);
            } catch (final OXException e) {
                _journal.error(e.getMessage(), e);
            } catch (NotCompliantMBeanException e) {
                _journal.error(e.getMessage(), e);
            }
        }
    }

    /**
     * Creates an appropriate instance of {@link ObjectName} from specified class name and domain name.
     * 
     * @param className The class name to use as object name
     * @param domain The domain name
     * @return An appropriate instance of {@link ObjectName}
     * @throws MalformedObjectNameException If instantiation of {@link ObjectName} fails
     */
    private static ObjectName getObjectName(final String className, final String domain) throws MalformedObjectNameException {
        final int pos = className.lastIndexOf('.');
        return new ObjectName(domain, "name", pos == -1 ? className : className.substring(pos + 1));
    }

    public SessionStorage getSessionStorage() {
        return _sessionStorage;
    }

    public void setWaitForChangesEmailPullDelay(int waitForChangesEmailPullDelay) {
        _waitForChangesEmailPullDelay = waitForChangesEmailPullDelay;
    }

    public int getWaitForChangesEmailPullDelay() {
        return _waitForChangesEmailPullDelay;
    }

    public void setWaitForChangesEmailMinPullDelay(int waitForChangesEmailMinPullDelay) {
        _waitForChangesEmailMinPullDelay = waitForChangesEmailMinPullDelay;
    }

    public int getWaitForChangesEmailMinPullDelay() {
        return _waitForChangesEmailMinPullDelay;
    }

    public void setKeepLastSentSyncStateInDB(boolean set) {
        _keepLastSentSyncStateInDB = set;
    }

    public void setMaxSyncStatesInDB(int states) {
        _maxSyncStatesInDB = states;
    }

    public SlowContentSyncer getSlowSyncer() {
        return new SlowContentSyncer(this, _syncRetryCount);
    }

    public IncrementalContentSyncer getIncrementalSyncer() {
        return new IncrementalContentSyncer(this, _syncRetryCount);
    }

    private void deactivateTimers() {
        if (_cleanupTask != null) {
            _cleanupTask.cancel();
            _cleanupTask = null;
        }
        if (_sessionCacheCleanupTask != null) {
            _sessionCacheCleanupTask.cancel();
            _sessionCacheCleanupTask = null;
        }
        if (_sessionCacheCleanupLogInfoTask != null) {
            _sessionCacheCleanupLogInfoTask.cancel();
            _sessionCacheCleanupLogInfoTask = null;
        }
    }
    
    public void setTimerService(TimerService timerService) {
        deactivateTimers();
        if (_timerService != null && _configurationManager != null) { // Only start new tasks if bundle is activated and TimerService is available
            _cleanupTask = _timerService.scheduleAtFixedRate(
                new USMSessionCleanupTask(this),
                USMSessionCleanupTask.INITIAL_DELAY,
                USMSessionCleanupTask.EXECUTION_INTERVAL);
            
            long sessionCacheCheckInterval = ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_DEFAULT;
            long sessionCacheCheckIntervalLogInfo = ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_LOG_INFO_DEFAULT;
            
            try {
                sessionCacheCheckInterval = _configurationManager.getProperty(
                                ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_PROPERTY, 
                                ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_DEFAULT,
                                true);
                
                sessionCacheCheckIntervalLogInfo = _configurationManager.getProperty(
                                ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_LOG_INFO_PROPERTY, 
                                ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_LOG_INFO_DEFAULT,
                                true);

            } catch (USMInvalidConfigurationException e) {
                _journal.error(e.getStackTrace(), e);
            }
            _sessionCacheCleanupTask = _timerService.scheduleAtFixedRate(
                new USMSessionCacheCleanupTask(this),
                sessionCacheCheckInterval,
                sessionCacheCheckInterval);
            
            _sessionCacheCleanupLogInfoTask = _timerService.scheduleAtFixedRate(
                new USMSessionCacheCleanupLogInfoTask(this),
                sessionCacheCheckIntervalLogInfo,
                sessionCacheCheckIntervalLogInfo);
        }
    }

    public long getSessionStorageTimeLimit() {
        return _sessionStorageTimeLimit;
    }

    /**
     * Creates a modified UUID from a base UUID and a given OX user id. The algorithm uses XOR, so this method can be used both ways, i.e.
     * given a modified UUID and OX user id, it reports back the original unmodified UUID. If either the UUID is null or the userID is 0,
     * this method returns the unmodified UUID.
     * 
     * @param uuid
     * @param userID
     * @return
     */
    public UUID getModifedUUID(UUID uuid, int userID) {
        if (uuid == null || userID == 0)
            return uuid;
        return new UUID(uuid.getMostSignificantBits() ^ userID, uuid.getLeastSignificantBits());
    }

    public OXDataCache getOXDataCache() {
        return _dataCache;
    }

    public Folder getSpecialFolder(SessionImpl session, String id) {
        if (!DefaultContentTypes.GROUPS_ID.equals(id) && !DefaultContentTypes.RESOURCES_ID.equals(id))
            return null;
        ContentType type = getContentTypeManager().getContentType(id);
        if (type == null)
            return null;
        Folder folder = new FolderImpl(session, getFolderContentType());
        folder.setID(id);
        folder.setTimestamp(System.currentTimeMillis());
        folder.setElementsContentType(type);
        return folder;
    }

    public List<String> getSpecialFolders() {
        return SPECIAL_FOLDERS_VIEW;
    }

    public boolean isForceNewSyncKeyOnEmptySync() {
        return _forceNewSyncKeyOnEmptySync;
    }

    public int getSessionCacheTimeout() {
        return _sessionCacheTimeout;
    }

    public void setSessionCacheTimeout(int timeout) {
        _sessionCacheTimeout = timeout;
    }

    public int getSyncStateCacheTimeout() {
        return _syncStateCacheTimeout;
    }

    public void setSyncStateCacheTimeout(int timeout) {
        _syncStateCacheTimeout = timeout;
    }

    public void updateCacheRemovals(int removedSessions, int jvmRemovals, int removedSyncStates) {
        _usmCacheInformation.sessionsRemovedDueToInactivity(removedSessions);
        _usmCacheInformation.sessionsRemovedByJVM(jvmRemovals);
        _usmCacheInformation.syncStatesRemovedDueToInactivity(removedSyncStates);
    }
    
    public USMCacheInformation getUSMCacheInformation() {
        return _usmCacheInformation;
    }

    public void syncStateSavedToDatabase() {
        _usmCacheInformation.syncStateSavedToDatabase();
    }

    public void syncStatesLoadedFromDatabase(int count) {
        _usmCacheInformation.syncStatesLoadedFromDatabase(count);
    }
}
