/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.contenttypes.folder.impl;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.FolderContentType;
import com.openexchange.usm.api.contenttypes.FolderContentTypeTransferHandler;
import com.openexchange.usm.api.contenttypes.UnsupportedContentOperationException;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.FolderNotFoundException;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.OperationDeniedException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMIllegalArgumentException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.contenttypes.util.AbstractTransferHandler;
import com.openexchange.usm.contenttypes.util.OXErrorConstants;
import com.openexchange.usm.contenttypes.util.USMContentTypesUtilErrorCodes;
import com.openexchange.usm.contenttypes.util.UtilConstants;
import com.openexchange.usm.ox_json.JSONResult;
import com.openexchange.usm.ox_json.JSONResultType;
import com.openexchange.usm.ox_json.OXJSONAccess;

/**
 * @author ibr
 */
public class FolderContentTypeTransferHandlerImpl extends AbstractTransferHandler implements FolderContentTypeTransferHandler {

    private static final int FOLDER_ID_LENGTH_LIMIT = 255;

    private static final String ALLOWED_MODULES = "allowed_modules";

    public FolderContentTypeTransferHandlerImpl(FolderContentTypeImpl contentType, Log log, OXJSONAccess ajaxAccess) {
        super(contentType, log, ajaxAccess);
    }

    @Override
    public void readDataObject(DataObject object, BitSet requestedFields) throws USMException {
        readFolder(checkContentType(object), requestedFields);
    }

    @Override
    public Folder[] readFolderContent(Folder folder, BitSet requestedFields) throws USMException {
        Session session = folder.getSession();
        if (_journal.isDebugEnabled())
            _journal.debug(session + " Read all " + _contentType.getID() + " in folder " + folder.getID());
        Map<String, String> parameters = createParametersWithAllowedModules(session, requestedFields);
        parameters.put(UtilConstants.PARENT, folder.getID());
        String errorOnDuplicateNamePersistentField = session.getPersistentField(UtilConstants.ERROR_ON_DUPLICATE_NAME_DB_FIELD);
        if (errorOnDuplicateNamePersistentField != null && errorOnDuplicateNamePersistentField.length() > 0)
            parameters.put(UtilConstants.ERROR_ON_DUPLICATE_NAME, errorOnDuplicateNamePersistentField);
        if(folder.getSession().getObjectsLimit() > 0) {
            addObjectsLimitParameters(folder.getSession().getObjectsLimit(), parameters);
        }
        DataObject[] listOfResults = performActionAndCheckRights(UtilConstants.ACTION_LIST, folder, requestedFields, parameters);
        Folder[] result = tranformToFolders(listOfResults);
        for (Folder f : result) {
            if (f != null)
                f.setParentFolder(folder);
        }
        return result;
    }

    @Override
    public Folder[] readUpdatedFolderContent(Folder folder, BitSet requestedFields, long timestamp) throws USMException {
        Session session = folder.getSession();
        if (_journal.isDebugEnabled())
            _journal.debug(session + " Read updated " + _contentType.getID() + " after " + timestamp + " in folder " + folder.getID());
        Map<String, String> parameters = createParametersWithAllowedModules(session, requestedFields);
        parameters.put(UtilConstants.PARENT, folder.getID()); // object ID of the folder
        parameters.put(UtilConstants.TIMESTAMP, String.valueOf(timestamp));
        return tranformToFolders(getListOfUpdates(
            folder,
            requestedFields,
            _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_UPDATES, session, parameters)));
    }

    @Override
    public void writeDeletedDataObject(DataObject object) throws USMException {
        writeDeletedFolder(checkContentType(object));
    }

    @Override
    public void writeNewDataObject(DataObject object) throws USMException {
        writeNewFolder(checkContentType(object));
    }

    @Override
    public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
        writeUpdatedFolder(checkContentType(object), timestamp);
    }

    public void readFolder(Folder object, BitSet requestedFields) throws USMException {
        Session session = object.getSession();
        if (_journal.isDebugEnabled())
            _journal.debug(session + " Get folder" + object.getID());
        // Build parameter list
        Map<String, String> parameters = createCommonParameters(session, requestedFields);
        parameters.put(UtilConstants.ID, object.getID());
        JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_GET, session, parameters);
        if (result.getResultType() == JSONResultType.Error) {
            checkConflictingChange(result.getJSONObject());
            checkAuthenticationFailed(result.getJSONObject());
            throw new FolderNotFoundException(
                USMContentTypesFolderErrorCodes.FOLDER_NOT_FOUND,
                "Folder not found: " + object.getUUID() + '(' + object.getID() + ')');
        }
        checkResult(result, JSONResultType.JSONObject);
        updateDataObjectFromJSONResult(object, requestedFields, result);
    }

    protected void checkAuthenticationFailed(JSONObject errorObject) throws InternalUSMException, AuthenticationFailedException {
        try {
            int category = errorObject.optInt(UtilConstants.CATEGORY);
            String message = errorObject.getString(UtilConstants.ERROR);
            String code = errorObject.getString(UtilConstants.CODE);
            if (category == OXErrorConstants.CATEGORY_PERMISSION && OXErrorConstants.MSG_1000.equals(code)) {
                throw new AuthenticationFailedException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER15, message, errorObject);
            }
        } catch (JSONException e) {
            throw new InternalUSMException(
                USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER16,
                "OX server sent an invalid  error object " + errorObject);
        }
    }

    private void addTreeParameter(Session session, Map<String, String> parameters) {
        String folderTree = session.getPersistentField(FolderContentType.FOLDER_TREE);
        if (folderTree != null && folderTree.length() > 0)
            parameters.put(UtilConstants.TREE, folderTree);
    }

    public void writeDeletedFolder(Folder object) throws USMException {
        Session session = object.getSession();
        if (_journal.isDebugEnabled())
            _journal.debug(session + " Delete folder " + object.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(UtilConstants.TIMESTAMP, String.valueOf(object.getTimestamp()));
        addTreeParameter(session, parameters);
        JSONArray array = new JSONArray();
        array.put(object.getID());
        JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_DELETE, session, parameters, array);
        if (result.getResultType() == JSONResultType.Error)
            throw new OperationDeniedException(USMContentTypesFolderErrorCodes.FOLDER_DELETION_DENIED_BY_OX, result.getJSONObject());
        // Do not use normal method to check for temporary failed deletions, since also fatal errors are reported for folder deletions that
        // way
        JSONArray notDeletedArray = extractArrayResult(result);
        if (notDeletedArray != null && notDeletedArray.length() > 0)
            throw new USMException(USMContentTypesFolderErrorCodes.FOLDER_DELETION_FAILED_IN_OX, "Folder deletion failed in OX server");
    }

    public void writeNewFolder(Folder object) throws USMException {
        Session session = object.getSession();
        if (_journal.isDebugEnabled())
            _journal.debug(session + " Create folder " + object.getID());
        checkIDLength(object);
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(UtilConstants.PARENTFOLDER_ID, object.getParentFolderID());
        addTreeParameter(session, parameters);
        JSONObject requestBody = createRequestBody(object, false, true);
        JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_NEW, session, parameters, requestBody);
        checkResult(result, JSONResultType.JSONObject);
        JSONObject o = result.getJSONObject();
        try {
            object.setID(o.getString(UtilConstants.RESULT_DATA));
        } catch (JSONException e) {
            throw new OXCommunicationException(
                USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER2,
                "Missing ID on new " + _contentType.getID(),
                e,
                result.toString());
        }
        readOptionalTimestamp(o, object);
    }

    private void checkIDLength(Folder object) throws USMException {
        if (isFolderIDTooLong(object))
            throw new USMException(USMContentTypesFolderErrorCodes.FOLDER_ID_TOO_LONG, "Folder id: " + object.getID() + " is too long");
    }

    private boolean isFolderIDTooLong(DataObject object) {
        String id = object.getID();
        if (id == null)
            return false;
        return id.length() > FOLDER_ID_LENGTH_LIMIT;
    }

    public void writeUpdatedFolder(Folder object, long timestamp) throws USMException {
        checkIDLength(object);
        writeStandardUpdateDataObject(object, timestamp, true);
    }

    public Folder[] readRootFolders(Session session, BitSet requestedFields) throws USMException {
        if (_journal.isDebugEnabled())
            _journal.debug(session + " Get root folders");
        Map<String, String> parameters = createParametersWithAllowedModules(session, requestedFields);
        JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_ROOT, session, parameters);
        checkResult(result, JSONResultType.JSONObject);
        Folder[] folders = tranformToFolders(getListOfResults(
            ((FolderContentType) _contentType).newDataObject(session),
            requestedFields,
            result));
        // Explicitly set parent ID of root folders to "0" so that differences reported by OX (0, null, "") depending on module are ignored
        for (Folder f : folders)
            f.setParentFolderID("0");
        return folders;
    }

    private Map<String, String> createCommonParameters(Session session, BitSet requestedFields) throws UnsupportedContentOperationException {
        Map<String, String> parameters = new HashMap<String, String>();
        addTreeParameter(session, parameters);
        parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
        return parameters;
    }

    private Map<String, String> createParametersWithAllowedModules(Session session, BitSet requestedFields) throws UnsupportedContentOperationException, USMException {
        Map<String, String> parameters = createCommonParameters(session, requestedFields);
        ContentType[] contentTypes = session.getContentTypes();
        if (contentTypes == null)
            return parameters;
        StringBuilder sb = new StringBuilder(128);
        for (ContentType contentType : contentTypes) {
            if (contentType.canBeFolderElementsContentType()) {
                sb.append(contentType.getID());
                sb.append(',');
            }
        }
        parameters.put(ALLOWED_MODULES, sb.toString());
        return parameters;
    }

    private Folder[] tranformToFolders(DataObject[] elements) throws USMException {
        ArrayList<Folder> elementsList = new ArrayList<Folder>();
        try {
            for (DataObject o : elements) {
                if (!isFolderIDTooLong(o))
                    elementsList.add((Folder) o);
            }
        } catch (ClassCastException e) {
            throw new InternalUSMException(USMContentTypesFolderErrorCodes.STRUCTURE_ERROR, "Not all elements are of type Folder", e);
        }
        Folder[] folders = elementsList.toArray(new Folder[elementsList.size()]);
        return folders;
    }

    // Special test method to filter invalid folders from result list
    @SuppressWarnings("unused")
    private Folder[] tranformToFolders2(DataObject[] elements) throws USMException {
        List<Folder> list = new ArrayList<Folder>();
        for (DataObject o : elements) {
            try {
                Folder f = (Folder) o;
                String module = f.getElementsContentTypeID();
                if (module.equals(DefaultContentTypes.TASK_ID) || module.equals(DefaultContentTypes.MAIL_ID) || module.equals(DefaultContentTypes.CALENDAR_ID) || module.equals(DefaultContentTypes.CONTACTS_ID) || module.equals(MODULE_SYSTEM))
                    list.add(f);
            } catch (Exception ignored) {
            }
        }
        return list.toArray(new Folder[list.size()]);
    }

    private Folder checkContentType(DataObject object) {
        if (object instanceof Folder && object.getContentType().getID().equals(DefaultContentTypes.FOLDER_ID))
            return (Folder) object;
        throw new USMIllegalArgumentException(USMContentTypesUtilErrorCodes.INVALID_CONTENT_TYPE_RECEIVED, "DataObject is no folder");
    }

    @Override
    protected String getOXAjaxAccessPath() {
        return UtilConstants.FOLDER_PATH;
    }

    @Override
    protected boolean hasFolderReadRights(Folder folder) throws USMException {
        return true;
    }
}
