/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.contenttypes.calendar.impl;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;

import org.apache.commons.logging.Log;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.usm.api.contenttypes.AppointmentContentType;
import com.openexchange.usm.api.contenttypes.ContentTypeField;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.UnsupportedContentOperationException;
import com.openexchange.usm.api.datatypes.DataTypes;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.contenttypes.util.AbstractTransferHandler;
import com.openexchange.usm.contenttypes.util.USMContentTypesUtilErrorCodes;
import com.openexchange.usm.contenttypes.util.UtilConstants;
import com.openexchange.usm.datatypes.tasks.calendar.CommonCalendarTasksFieldNames;
import com.openexchange.usm.datatypes.tasks.calendar.UserParticipantObject;
import com.openexchange.usm.ox_json.JSONResult;
import com.openexchange.usm.ox_json.JSONResultType;
import com.openexchange.usm.ox_json.OXJSONAccess;

/**
 * Transfer Handler for the module "Calendar"
 * @author ldo
 *
 */
public class AppointmentContentTypeTransferHandler extends AbstractTransferHandler {

	private static final long ONE_DAY_IN_MILLISECONDS = 86400000L;
	private static final long MAXIMUM_EXPECTED_EXCEPTION_MOVEMENT = ONE_DAY_IN_MILLISECONDS * 365L;

	private final BitSet DELETE_EXCEPTION_FIELDS = new BitSet();

	public AppointmentContentTypeTransferHandler(AppointmentContentType appointmentContentType, Log log,
			OXJSONAccess ajaxAccess) {
		super(appointmentContentType, log, ajaxAccess);
		addToDeleteExceptionsBitMask(appointmentContentType.getFields(), UtilConstants.FIELD_ID,
				UtilConstants.FOLDER_ID, CommonCalendarTasksFieldNames.START_DATE,
				CommonCalendarTasksFieldNames.RECURRENCE_POSITION,
				CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION, CalendarConstants.RECURRENCE_ID,
				CalendarConstants.TIMEZONE);
	}

	private void addToDeleteExceptionsBitMask(ContentTypeField[] fields, String... requiredFields) {
		for (String f : requiredFields)
			addToDeleteExceptionsBitMask(f, fields);
	}

	private void addToDeleteExceptionsBitMask(String f, ContentTypeField[] fields) {
		for (int i = 0; i < fields.length; i++) {
			if (f.equals(fields[i].getFieldName())) {
				DELETE_EXCEPTION_FIELDS.set(i);
				return;
			}
		}
		throw new IllegalStateException("Unknown field <" + f + "> in ContentType <" + _contentType + ">");
	}

	@Override
	public DataObject[] readFolderContent(Folder folder, BitSet requestedFields) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(folder.getSession() + " Read all " + _contentType.getID() + " in folder " + folder.getID());
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.FOLDER, folder.getID());
		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		parameters.put(UtilConstants.RECURRENCE_MASTER, Boolean.TRUE.toString());
		Session session = folder.getSession();
		parameters.put(UtilConstants.START, String.valueOf(session.getStartDate()));
		parameters.put(UtilConstants.END, String.valueOf(session.getEndDate()));
		if(folder.getSession().getObjectsLimit() > 0) {
		    addObjectsLimitParameters(folder.getSession().getObjectsLimit(), parameters);
		}
		DataObject[] result = performActionAndCheckRights(UtilConstants.ACTION_ALL, folder, requestedFields, parameters);

		resetRecurrenceTypeBasedFields(result);
		return result;
	}
	

	public DataObject[] getAllAppointments(Session session, BitSet requestedFields) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(session + " Getting new appointments for date range " + session.getStartDate() + " - "
					+ session.getEndDate());
		Map<String, String> parameters = new HashMap<String, String>();

		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		parameters.put(UtilConstants.START, String.valueOf(session.getStartDate()));
		parameters.put(UtilConstants.END, String.valueOf(session.getEndDate()));
		parameters.put(UtilConstants.RECURRENCE_MASTER, Boolean.TRUE.toString());
		
		if(session.getObjectsLimit() > 0) {
            addObjectsLimitParameters(session.getObjectsLimit(), parameters);
        }

		JSONResult jsonResult = _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_ALL, session, parameters);
		DataObject[] result = buildResultArray(session, requestedFields, jsonResult);
		return result;
	}

	public DataObject[] getAllRecurrencesAsAppointments(Session session, BitSet requestedFields, DataObject seriesObject)
			throws USMException {
		return getAllRecurrencesAsAppointments(session, requestedFields, seriesObject, session.getStartDate(),
				session.getEndDate());
	}

	public DataObject[] getAllRecurrencesAsAppointments(Session session, BitSet requestedFields,
			DataObject seriesObject, long startDate, long endDate) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(session + " Getting all appointments as recurrences");
		Map<String, String> parameters = new HashMap<String, String>();

		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		parameters.put(UtilConstants.START, String.valueOf(startDate));
		parameters.put(UtilConstants.END, String.valueOf(endDate));
		String folder = seriesObject != null ? seriesObject.getParentFolderID() : null;
		if (folder != null)
			parameters.put(UtilConstants.FOLDER, folder);
		parameters.put(UtilConstants.RECURRENCE_MASTER, Boolean.FALSE.toString());
		
		if(session.getObjectsLimit() > 0) {
            addObjectsLimitParameters(session.getObjectsLimit(), parameters);
        }

		JSONResult jsonResult = _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_ALL, session, parameters);
		JSONArray jArray = extractArrayResult(jsonResult);
		List<DataObject> resultList = new ArrayList<DataObject>();
		for (int i = 0; i < jArray.length(); i++) {
			JSONArray columnsArray;
			DataObject destination = null;
			try {
				columnsArray = jArray.getJSONArray(i);
				destination = _contentType.newDataObject(session);
				updateDataObjectFromJSONArray(destination, requestedFields, columnsArray);
				readOptionalTimestamp(jsonResult.getJSONObject(), destination);
				if (seriesObject != null) {
					if (destination.getID().equals(seriesObject.getID())
							|| seriesObject.getID().equals(
									String.valueOf(destination.getFieldContent(CalendarConstants.RECURRENCE_ID))))
						resultList.add(destination);
				} else {
					resultList.add(destination);
				}
			} catch (JSONException e) {
				throw generateException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER1, destination,
						"get new appointmets:data not present", e);
			}
		}
		DataObject[] result = resultList.toArray(new DataObject[resultList.size()]);
		resetRecurrenceTypeBasedFields(result);
		return result;
	}
	
    public DataObject[] getAllChangeExceptions(Session session, BitSet requestedFields, DataObject seriesObject) throws USMException {
        if (_journal.isDebugEnabled())
            _journal.debug(session + " Getting all change exceptions for series object: " + seriesObject.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
        parameters.put(UtilConstants.FOLDER, seriesObject.getParentFolderID());
        parameters.put(UtilConstants.ID, seriesObject.getID());

        JSONResult jsonResult = _ajaxAccess.doGet(getOXAjaxAccessPath(), CalendarConstants.GET_CHANGE_EXCEPTIONS_ACTION, session, parameters);
        DataObject[] result = buildResultArray(session, requestedFields, jsonResult);
        return result;
    }
    

    private DataObject[] buildResultArray(Session session, BitSet requestedFields, JSONResult jsonResult) throws OXCommunicationException, InternalUSMException {
        JSONArray jArray = extractArrayResult(jsonResult);
        List<DataObject> resultList = new ArrayList<DataObject>();
        for (int i = 0; i < jArray.length(); i++) {
            JSONArray columnsArray;
            DataObject destination = null;
            try {
                columnsArray = jArray.getJSONArray(i);
                destination = _contentType.newDataObject(session);
                updateDataObjectFromJSONArray(destination, requestedFields, columnsArray);
                resultList.add(destination);
            } catch (JSONException e) {
                throw generateException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER1, destination,
                        "get new appointmets:data not present", e);
            }
        }
        DataObject[] result = resultList.toArray(new DataObject[resultList.size()]);
        resetRecurrenceTypeBasedFields(result);
        return result;
    }

	@Override
	public void writeDeletedDataObject(DataObject object) throws USMException {
		if (Boolean.TRUE
				.equals(object.getSession().getCustomProperty(AppointmentContentType.DECLINE_INSTEAD_OF_DELETE))) {
			if (shouldDeclineInsteadOfDelete(object)) {
				// If yes, decline appointment instead and return
				confirm(object, 2);
				return;
			}
		}
		String recurrencePosition = String.valueOf(object
				.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_POSITION));
		if (recurrencePosition == null || recurrencePosition.equals("0") || recurrencePosition.equals("null"))
			writeExtendedDeleteDataObject(object, UtilConstants.FOLDER, object.getParentFolderID());
		else
			writeExtendedDeleteDataObject(object, CommonCalendarTasksFieldNames.RECURRENCE_POSITION,
					recurrencePosition, UtilConstants.FOLDER, object.getParentFolderID());
	}

	private boolean shouldDeclineInsteadOfDelete(DataObject object) {
		if (object.getParentFolderOwnerID() == 0) {
			// If not in shared folder
			int userID = object.getSession().getUserIdentifier();
			Object o = object.getFieldContent(UtilConstants.CREATED_BY);
			int organizer = 0;
			if (o != null) {
				try {
					organizer = Integer.parseInt(String.valueOf(o));
				} catch (NumberFormatException ignored) {
				}
			}
			if (userID != organizer) {
				// And if not organizer
				o = object.getFieldContent(CommonCalendarTasksFieldNames.USERS);
				if (o instanceof Object[]) {
					Object[] list = (Object[]) o;
					for (Object u : list) {
						if (u instanceof UserParticipantObject) {
							UserParticipantObject participant = (UserParticipantObject) u;
							if (participant.getId() == userID)
								// And if participant -> decline instead of delete
								return true;
						}
					}
				}
			}
		}
		return false;
	}

	@Override
	public void writeNewDataObject(DataObject object) throws USMException {
		//don't call the standard write utility method because conflicts array can be returned from server
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " New " + DefaultContentTypes.CALENDAR_ID);
		checkTypeToWrite(object, DefaultContentTypes.CALENDAR_ID);
		Map<String, String> parameters = new HashMap<String, String>();
		JSONObject requestBody = createRequestBodyWithIgnoreConflicts(object, false);
		//		printTimeValues("New", requestBody);
		JSONResult result = _ajaxAccess.doPut(CalendarConstants.CALENDAR_PATH, UtilConstants.ACTION_NEW,
				object.getSession(), parameters, requestBody);
		checkResult(result, JSONResultType.JSONObject);
		JSONObject o = result.getJSONObject();
		readOptionalTimestamp(o, object);
		try {
			JSONObject resultObject = o.getJSONObject(UtilConstants.RESULT_DATA);
			if (resultObject.has(UtilConstants.RESULT_ID)) {
				object.setID(resultObject.getString(UtilConstants.RESULT_ID));
			} else if (resultObject.has(CalendarConstants.CONFLICTS)) {
				JSONArray arr = resultObject.getJSONArray(CalendarConstants.CONFLICTS);
				throw new OXCommunicationException(USMContentTypesCalendarErrorCodes.CONFLICTS,
						"Appointment could not be created because of conflicts", arr.getJSONObject(0));
			} else {
				throw new OXCommunicationException(USMContentTypesCalendarErrorCodes.OTHER_ERROR,
						"Appointment could not be created because of ox communication error", resultObject.toString());
			}
		} catch (JSONException e) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER1,
					"Invalid Result on new " + DefaultContentTypes.CALENDAR_ID, e, result.toString());
		}
		writeDeletedExceptions(object);
	}

	private void writeDeletedExceptions(DataObject seriesObject) throws USMException {
		Object[] deleteExceptions = (Object[]) seriesObject
				.getFieldContent(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS);
		if (deleteExceptions == null || deleteExceptions.length == 0)
			//|| !seriesObject.isFieldModified(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS))
			return;
		long intervalStart = Long.MAX_VALUE;
		long intervalEnd = 0L;
		Set<Number> deleteExceptionSet = new HashSet<Number>();
		for (Object o : deleteExceptions) {
			if (o instanceof Number) {
				Number n = (Number) o;
				deleteExceptionSet.add(n);
				long l = n.longValue();
				intervalStart = Math.min(intervalStart, l - MAXIMUM_EXPECTED_EXCEPTION_MOVEMENT);
				intervalEnd = Math.max(intervalEnd, l + MAXIMUM_EXPECTED_EXCEPTION_MOVEMENT);
			}
		}
		if (deleteExceptionSet.isEmpty())
			return;
		DataObject[] allOcurrences = getAllRecurrencesAsAppointments(seriesObject.getSession(),
				DELETE_EXCEPTION_FIELDS, seriesObject, intervalStart, intervalEnd);
		long timestamp = 0L;
		for (int j = 0; j < allOcurrences.length; j++) {
			DataObject occ = allOcurrences[j];
			if (occ != null) {
				if (shouldDeleteOccurrence(occ, deleteExceptionSet)) {
					if (timestamp != 0L)
						occ.setTimestamp(timestamp);
					writeDeletedDataObject(occ);
					timestamp = occ.getTimestamp();
				}
			}
		}
		if (timestamp != 0L)
			seriesObject.setTimestamp(timestamp);
	}

	private boolean shouldDeleteOccurrence(DataObject occ, Set<Number> deleteExceptionSet) {
		Object o = occ.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION);
		if (o != null) // the DataObject is a change exception, and therefore recurrence_date_position is set to the original date
			return deleteExceptionSet.contains(o);
		o = occ.getFieldContent(CommonCalendarTasksFieldNames.START_DATE);
		if (!(o instanceof Number))
			return false;
		long start = ((Number) o).longValue();
		long offset = 0L;
		// modify start_date so that it's 00:00 UTC
		o = occ.getFieldContent(CalendarConstants.TIMEZONE);
		if (o instanceof String)
			offset = TimeZone.getTimeZone((String) o).getOffset(start);
		start = (start + offset) / ONE_DAY_IN_MILLISECONDS;
		start *= ONE_DAY_IN_MILLISECONDS;
		return deleteExceptionSet.contains(start);
	}

	private JSONObject createRequestBodyWithIgnoreConflicts(DataObject object, boolean isUpdate)
			throws InternalUSMException {
		JSONObject requestBody = new JSONObject();
		try {
			requestBody.put(CalendarConstants.IGNORE_CONFLICTS, Boolean.TRUE);
		} catch (JSONException e) {
			throwInternalUSMException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER7, "writing", object,
					CalendarConstants.IGNORE_CONFLICTS, DataTypes.BOOLEAN, e);
		}
		ContentTypeField[] fields = object.getContentType().getFields();
		boolean isException = isSeriesException(object);
		boolean isNewException = isException && isNewException(object);
		for (int i = 0; i < fields.length; i++) {
			ContentTypeField field = fields[i];
			if (isUpdate && field.getFieldName().equals(CommonCalendarTasksFieldNames.ALARM)) { // special workaround for "alarm", see bug #17327
				if (object.isFieldModified(CommonCalendarTasksFieldNames.ALARM)
						&& object.getFieldContent(CommonCalendarTasksFieldNames.ALARM) == null) {
					try {
						requestBody.put(CommonCalendarTasksFieldNames.ALARM, -1);
						continue;
					} catch (JSONException e) {
						throwInternalUSMException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER6, "writing",
								object, CommonCalendarTasksFieldNames.ALARM, DataTypes.NUMBER, e);
					}
				}
			}
			boolean isNoSpecialField = isNoSpecialSeriesField(field, !isNewException);
			if (isException) {
				if (isNoSpecialField)
					addFieldToRequestBody(object, isUpdate, !isUpdate, requestBody, field, i);
			} else {
				addFieldToRequestBody(object, isUpdate && isNoSpecialField, !isUpdate, requestBody, field, i);
			}
		}
		requestBody.remove(CalendarConstants.RECURRENCE_ID);
		// XXX Workaround for Bug #14952 (start)
		if (requestBody.optBoolean(CalendarConstants.FULL_TIME)
				&& !requestBody.has(CommonCalendarTasksFieldNames.START_DATE)) {
			for (int i = 0; i < fields.length; i++) {
				if (fields[i].getFieldName().equals(CommonCalendarTasksFieldNames.START_DATE)) {
					addFieldToRequestBody(object, false, true, requestBody, fields[i], i);
					break;
				}
			}
		}
		// XXX Workaround for Bug #14952 (end)
		if (!isException) {
			try {
				requestBody.remove(CommonCalendarTasksFieldNames.RECURRENCE_POSITION);
				requestBody.remove(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION);
				requestBody.remove(CommonCalendarTasksFieldNames.CHANGE_EXCEPTIONS);
				requestBody.remove(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS);
				Number recType = (Number) object.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_TYPE);
				if (recType != null && recType.intValue() > 0) {
					if (object.isFieldModified(CommonCalendarTasksFieldNames.UNTIL)) {
						//this is a special case for changing the until of the series. (iPhone - for all future appointments)
						//All relevant recurrence fields have to be sent
						requestBody.put(CommonCalendarTasksFieldNames.RECURRENCE_TYPE, recType);
						requestBody.put(CommonCalendarTasksFieldNames.INTERVAL,
								object.getFieldContent(CommonCalendarTasksFieldNames.INTERVAL));
						requestBody.put(CommonCalendarTasksFieldNames.UNTIL,
								object.getFieldContent(CommonCalendarTasksFieldNames.UNTIL));
						Object days = object.getFieldContent(CommonCalendarTasksFieldNames.DAYS);
						if (days instanceof Number && ((Number) days).intValue() != 127)
							requestBody.put(CommonCalendarTasksFieldNames.DAYS, days);
						else
							requestBody.remove(CommonCalendarTasksFieldNames.DAYS);
					}
				} else {
					if (object.isFieldModified(CommonCalendarTasksFieldNames.RECURRENCE_TYPE))
						requestBody.put(CommonCalendarTasksFieldNames.RECURRENCE_TYPE, recType);
					else
						requestBody.remove(CommonCalendarTasksFieldNames.RECURRENCE_TYPE);
					requestBody.remove(CommonCalendarTasksFieldNames.INTERVAL);
					requestBody.remove(CommonCalendarTasksFieldNames.UNTIL);
					requestBody.remove(CommonCalendarTasksFieldNames.DAYS);
					requestBody.remove(CommonCalendarTasksFieldNames.OCCURENCES);
				}
			} catch (JSONException e) {
				throwInternalUSMException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER1, "writing", object,
						CalendarConstants.IGNORE_CONFLICTS, DataTypes.BOOLEAN, e);
			}
		}
		return requestBody;
	}

	private boolean isNoSpecialSeriesField(ContentTypeField field, boolean isUpdate) {
		int id = field.getFieldID();
		return id < (isUpdate ? 206 : 209) || (id > 216 && id != 222);
	}

	private boolean isSeriesException(DataObject object) {
		Object o = object.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_POSITION);
		return (o instanceof Number) && ((Number) o).intValue() != 0;
	}

	private boolean isNewException(DataObject object) {
		Object o = object.getFieldContent(CalendarConstants.RECURRENCE_ID);
		return String.valueOf(o).equals(object.getID());
	}

	@Override
	public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Updating " + _contentType.getID());
		checkTypeToWrite(object, _contentType.getID());

		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.ID, object.getID());
		parameters.put(UtilConstants.FOLDER, object.getOriginalParentFolderID());
		parameters.put(UtilConstants.TIMESTAMP, String.valueOf(timestamp));
		JSONObject requestBody = createRequestBodyWithIgnoreConflicts(object, true);
		try {
			if (requestBody.has(CalendarConstants.SEQUENCE))
				requestBody.put("ignoreOutdatedSequence", "true");
		} catch (JSONException e1) {
			throw new USMException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER14,
					"Error on creating request body", e1);
		}

		//		printTimeValues("Change", requestBody);
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_UPDATE, object.getSession(),
				parameters, requestBody);

		checkResult(result, JSONResultType.JSONObject);
		// XXX Shouldn't we use the timestamp returned from the OX server ?
		object.setTimestamp(timestamp);
		JSONObject o = result.getJSONObject();
		try {
			JSONObject resultObject = o.getJSONObject(UtilConstants.RESULT_DATA);
			if (resultObject.has(CalendarConstants.CONFLICTS)) {
				JSONArray arr = resultObject.getJSONArray(CalendarConstants.CONFLICTS);
				throw new OXCommunicationException(USMContentTypesCalendarErrorCodes.CONFLICTS,
						"Appointment could not be changed because of conflicts", arr.getJSONObject(0));
			}
		} catch (JSONException e) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER14,
					"Invalid Result on update " + DefaultContentTypes.CALENDAR_ID, e, result.toString());
		} finally {
			setAppOrTaskConfirmations(object);
			writeDeletedExceptions(object);
		}
	}

	@Override
	public DataObject[] readUpdatedFolderContent(Folder folder, BitSet requestedFields, long timestamp)
			throws USMException {
		Session session = folder.getSession();
		DataObject[] result = readStandardUpdatedFolderContent(folder, timestamp, requestedFields,
				UtilConstants.ACTION_UPDATES, UtilConstants.FOLDER, null, null, String.valueOf(session.getStartDate()),
				String.valueOf(session.getEndDate()));
		resetRecurrenceTypeBasedFields(result);
		return result;
	}

	@Override
	protected String getOXAjaxAccessPath() {
		return CalendarConstants.CALENDAR_PATH;
	}

	public String resolveUID(Session session, String uid) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(session + " Resolving uid " + uid);
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.UID, uid);

		JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_RESOLVEUID, session,
				parameters);

		checkResult(result, JSONResultType.JSONObject);
		JSONObject o = result.getJSONObject();
		try {
			JSONObject resultObject = o.getJSONObject(UtilConstants.RESULT_DATA);
			if (resultObject.has(UtilConstants.ID)) {
				String id = resultObject.getString(UtilConstants.ID);
				return id;
			}
		} catch (JSONException e1) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER7, "Invalid id",
					e1, result.toString());
		}
		return null;
	}

}
