/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.custom.config;

import static com.openexchange.util.custom.base.NullUtil.absent;
import static com.openexchange.util.custom.base.NullUtil.emptyList;
import static com.openexchange.util.custom.base.NullUtil.emptySet;
import static com.openexchange.util.custom.base.NullUtil.immutableCopyOf;
import static com.openexchange.util.custom.base.NullUtil.immutableSetCopyOf;
import static com.openexchange.util.custom.base.NullUtil.optional;
import static com.openexchange.util.custom.base.NullUtil.optionalFromNullable;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.util.Iterator;
import java.util.Objects;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.mail.internet.idn.IDNA;
import com.google.common.base.CharMatcher;
import com.google.common.base.Optional;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterators;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.cascade.ConfigView;
import com.openexchange.config.cascade.ConfigViewFactory;
import com.openexchange.configuration.ConfigurationExceptionCodes;
import com.openexchange.exception.OXException;
import com.openexchange.session.Session;
import com.openexchange.util.custom.base.NullUtil;


/**
 * {@link ConfigViewReader}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.2.0
 */
public class ConfigViewReader {

    public static ConfigViewReader ofUser(final @Nullable ConfigViewFactory configViewFactory, int userId, int contextId) throws OXException {
        return new ConfigViewReader(null, configViewFactory != null ? configViewFactory.getView(userId, contextId) : null);
    }

    public static ConfigViewReader ofUser(final String prefix, final @Nullable ConfigViewFactory configViewFactory, int userId, int contextId) throws OXException {
        return new ConfigViewReader(prefix, configViewFactory != null ? configViewFactory.getView(userId, contextId) : null);
    }

    public static ConfigViewReader ofUser(final String prefix, final @Nullable ConfigViewFactory configViewFactory, final Session session) throws OXException {
        return new ConfigViewReader(prefix, configViewFactory != null ? configViewFactory.getView(session.getUserId(), session.getContextId()) : null);
    }
    
    public static ConfigViewReader of(final @Nullable ConfigView configView) {
        return new ConfigViewReader(null, configView);
    }

    public static ConfigViewReader of(final String prefix, final @Nullable ConfigView configView) {
        return new ConfigViewReader(prefix, configView);
    }
    
    private final @Nullable ConfigView configView;
    private final @Nullable String prefix;
    
    private ConfigViewReader(final @Nullable String prefix, final @Nullable ConfigView configView) {
        this.configView = configView;
        if (prefix == null || prefix.isEmpty()) {
            this.prefix = null;
        } else {
            this.prefix = prefix;
        }
    }
    
    private final String n(final String name) {
        return prefix(prefix, name);
    }
    
    private static final class ConfigPropertyPrefixFunction implements Function<String, String> {
        private final @Nullable String prefix;
        public ConfigPropertyPrefixFunction(final @Nullable String prefix) {
            this.prefix = prefix;
        }
        @Override
        @Nullable
        public String apply(@Nullable String input) {
            return input != null ? prefix(prefix, input) : null;
        }
        @Override
        public boolean equals(@Nullable Object obj) {
            return super.equals(obj);
        }
        @Override
        public int hashCode() {
            return Objects.hash(prefix);
        }
    }
    
    public static final Function<String, String> toPrefixFunction(final @Nullable String prefix) {
        return new ConfigPropertyPrefixFunction(prefix);
    }
    
    @SuppressWarnings("null")
    public static final String prefix(final @Nullable String prefix, final String name) {
        if (prefix == null) {
            return name;
        }
        if (name.startsWith(".") && prefix.endsWith(".")) {
            return new StringBuilder(prefix.length() + name.length() - 1)
                .append(prefix)
                .append(name.substring(1))
                .toString();
        }
        if (! (name.startsWith(".") || prefix.endsWith("."))) {
            return new StringBuilder(prefix.length() + name.length() + 1)
                .append(prefix)
                .append(".")
                .append(name)
                .toString();
        }
        return new StringBuilder(prefix.length() + name.length())
            .append(prefix)
            .append(name)
            .toString();
    }
    
    public final class MandatoryValueReader {
        public String str(final String name) throws OXException {
            final String n = n(name);
            final ConfigView cv = configView;
            if (cv == null) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(ConfigView.class.getSimpleName() + " is null");
            }
            String str = cv.opt(n, String.class, null);
            if (str == null) {
                throw ConfigurationExceptionCodes.PROPERTY_MISSING.create(n);
            }
            str = str.replaceAll("^\\s*", "").replaceAll("\\s*$", "");
            if (str.isEmpty()) {
                throw ConfigurationExceptionCodes.PROPERTY_MISSING.create(n);
            }
            return str;
        }
        
        public String str(final String name, final String defaultValue) throws OXException {
            final String n = n(name);
            final ConfigView cv = configView;
            if (cv == null) {
                return defaultValue;
            }
            String str = cv.opt(n, String.class, null);
            if (str == null) {
                return defaultValue;
            }
            str = str.replaceAll("^\\s*", "").replaceAll("\\s*$", "");
            if (str.isEmpty()) {
                return defaultValue;
            }
            return str;
        }
        
        public long longNum(final String name, final long defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (! str.isPresent()) {
                return defaultValue;
            }
            try {
                return Long.parseLong(str.get());
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }

        public long longNum(final String name) throws OXException {
            final String str = mandatory.str(name);
            try {
                return Long.parseLong(str);
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }
        
        public int intNum(final String name, final int defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (! str.isPresent()) {
                return defaultValue;
            }
            try {
                return Integer.parseInt(str.get());
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }

        public int intNum(final String name) throws OXException {
            final String str = mandatory.str(name);
            try {
                return Integer.parseInt(str);
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }

        public int positiveIntNum(final String name) throws OXException {
            final int i = mandatory.intNum(name);
            if (i < 0) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s must be a positive numeric value", n(name)));
            } else {
                return i;
            }
        }
        
        public boolean bool(final String name, final boolean defaultValue) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return defaultValue;
            }
            final String str = o.get();
            if (str.equalsIgnoreCase("true")) {
                return true;
            }
            if (str.equalsIgnoreCase("false")) {
                return false;
            }
            throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a boolean (true/false)", n(name)));
        }

        public boolean bool(final String name) throws OXException {
            final String str = mandatory.str(name);
            if (str.equalsIgnoreCase("true")) {
                return true;
            }
            if (str.equalsIgnoreCase("false")) {
                return false;
            }
            throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a boolean (true/false)", n(name)));
        }
        
        public String hostname(final String name, final String defaultValue) throws OXException {
            final String str = IDNA.toASCII(mandatory.str(name, defaultValue));
            if (str != null) {
                return str;
            } else {
                return defaultValue;
            }
        }

        public String hostname(final String name) throws OXException {
            final String str = IDNA.toASCII(mandatory.str(name));
            if (str != null) {
                return str;
            } else {
                throw ConfigurationExceptionCodes.PROPERTY_MISSING.create(n(name));
            }
        }
        
        public int port(final String name) throws OXException {
            final int value = mandatory.intNum(name);
            if (value <= 0 || value > 65535) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s does not contain a valid port number", n(name)));
            } else {
                return value;
            }
        }

        public int port(final String name, final int defaultValue) throws OXException {
            final Optional<Integer> o = optional.intNum(name);
            if (! o.isPresent()) {
                return defaultValue;
            }
            final int value = o.get();
            if (value <= 0 || value > 65535) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s does not contain a valid port number", n(name)));
            } else {
                return value;
            }
        }
        
        public Charset charset(final String name, final Charset defaultValue) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return defaultValue;
            } else {
                try {
                    final Charset cs = Charset.forName(o.get());
                    if (cs != null) {
                        return cs;
                    } else {
                        return defaultValue;
                    }
                } catch (final Exception e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("%s does not contain a valid charset: %s",
                        n(name), e.getMessage()));
                }
            }
        }

        public Charset charset(final String name) throws OXException {
            final String str = mandatory.str(name);
            final @Nullable Charset result;
            try {
                result = Charset.forName(str);
            } catch (final Exception e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("%s does not contain a valid charset: %s",
                    n(name), e.getMessage()));
            }
            if (result != null) {
                return result;
            } else {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s does not contain a valid charset", n(name)));
            }
        }
        
        public ImmutableList<String> strList(final String name, final CharMatcher split) throws OXException {
            final String str = mandatory.str(name);
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(str);
            final ImmutableList<String> list = NullUtil.immutableCopyOf(iter);
            if (list.isEmpty()) {
                throw ConfigurationExceptionCodes.PROPERTY_MISSING.create(n(name));
            } else {
                return list;
            }
        }

        public ImmutableList<String> strList(final String name) throws OXException {
            return mandatory.strList(name, DEFAULT_SPLIT_PATTERN);
        }
        
        public ImmutableSet<String> strSet(final String name, final CharMatcher split) throws OXException {
            final String str = mandatory.str(name);
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(str);
            final ImmutableSet<String> set = NullUtil.immutableSetCopyOf(iter);
            if (set.isEmpty()) {
                throw ConfigurationExceptionCodes.PROPERTY_MISSING.create(n(name));
            } else {
                return set;
            }
        }

        public ImmutableSet<String> strSet(final String name) throws OXException {
            return mandatory.strSet(name, DEFAULT_SPLIT_PATTERN);
        }
        
        public URI uri(String name) throws OXException {
            final String str = mandatory.str(name);
            try {
                return new URI(str);
            } catch (final URISyntaxException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid URI", n(name)));
            }
        }
        
        public URI uri(String name, URI defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                try {
                    return new URI(str.get());
                } catch (final URISyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid URI", n(name)));
                }
            } else {
                return defaultValue;
            }
        }
        
        public URI uri(String name, String defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                try {
                    return new URI(str.get());
                } catch (final URISyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid URI", n(name)));
                }
            } else {
                try {
                    return new URI(defaultValue);
                } catch (final URISyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the default value of %s is not a valid URI", n(name)));
                }
            }
        }
        
        public Pattern pattern(String name) throws OXException {
            final String str = mandatory.str(name);
            final Pattern result;
            try {
                result = Pattern.compile(str);
            } catch (PatternSyntaxException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
            }
            if (result != null) {
                return result;
            } else {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
            }
        }
        
        public Pattern pattern(String name, Pattern defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get());
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                return defaultValue;
            }
        }
        
        public Pattern pattern(String name, int flags, Pattern defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get(), flags);
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                return defaultValue;
            }
        }
        
        public Pattern pattern(String name, String defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get());
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                final Pattern result;
                try {
                    result = Pattern.compile(defaultValue);
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the default value for %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the default value for %s is not a valid regular expression (null result)", n(name)));
                }
            }
        }
        
        public Pattern pattern(String name, int flags, String defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get(), flags);
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                final Pattern result;
                try {
                    result = Pattern.compile(defaultValue, flags);
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the default value for %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the default value for %s is not a valid regular expression (null result)", n(name)));
                }
            }
        }
        
        public Pattern pattern(String name, int flags) throws OXException {
            final String str = mandatory.str(name);
            final Pattern result;
            try {
                result = Pattern.compile(str, flags);
            } catch (PatternSyntaxException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
            }
            if (result != null) {
                return result;
            } else {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
            }
        }
        
    };
    public final MandatoryValueReader mandatory = new MandatoryValueReader();
    public final MandatoryValueReader m = mandatory;

    public final class OptionalValueReader {
        public Optional<String> str(final String name) throws OXException {
            final String n = n(name);
            final ConfigView cv = configView;
            if (cv == null) {
                return absent();
            }
            String str = cv.opt(n, String.class, null);
            if (str == null) {
                return absent();
            }
            str = str.replaceAll("^\\s*", "").replaceAll("\\s*$", "");
            if (str.isEmpty()) {
                return absent();
            }
            return optional(str);
        }
        
        public Optional<Long> longNum(final String name) throws OXException {
            final Optional<String> str = optional.str(name);
            if (! str.isPresent()) {
                return absent();
            }
            try {
                return optionalFromNullable(Long.parseLong(str.get()));
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }

        public Optional<Integer> intNum(final String name) throws OXException {
            final Optional<String> str = optional.str(name);
            if (! str.isPresent()) {
                return absent();
            }
            try {
                return optionalFromNullable(Integer.parseInt(str.get()));
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }

        public Optional<Integer> positiveIntNum(final String name) throws OXException {
            final Optional<Integer> o = optional.intNum(name);
            if (o.isPresent()) {
                final int value = o.get();
                if (value < 0) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s must be a positive numeric value", n(name)));
                }
            }
            return o;
        }
        
        public Optional<Boolean> bool(final String name) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return absent();
            }
            final String str = o.get();
            if (str.equalsIgnoreCase("true")) {
                return optional(true);
            }
            if (str.equalsIgnoreCase("false")) {
                return optional(false);
            }
            throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a boolean (true/false)", n(name)));
        }
        
        public Optional<String> hostname(final String name) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return absent();
            }
            return optionalFromNullable(IDNA.toASCII(o.get()));
        }
        
        public Optional<Integer> port(final String name) throws OXException {
            final Optional<Integer> i = optional.intNum(name);
            if (! i.isPresent()) {
                return i;
            }
            final int value = i.get();
            if (value <= 0 || value > 65535) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s does not contain a valid port number", n(name)));
            } else {
                return i;
            }
        }
        
        public Optional<Charset> charset(final String name) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return absent();
            } else {
                try {
                    return optionalFromNullable(Charset.forName(o.get()));
                } catch (final Exception e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("%s does not contain a valid charset: %s",
                        n(name), e.getMessage()));
                }
            }
        }
        
        public ImmutableList<String> strList(final String name, final CharMatcher split, final Iterator<String> defaultValues) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return immutableCopyOf(defaultValues);
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableCopyOf(iter);
        }

        public ImmutableList<String> strList(final String name, final Iterator<String> defaultValues) throws OXException {
            return optional.strList(name, DEFAULT_SPLIT_PATTERN, defaultValues);
        }
        
        @SuppressWarnings("null")
        public ImmutableList<String> strList(final String name, final CharMatcher split, final String... defaultValues) throws OXException {
            return optional.strList(name, split, Iterators.forArray(defaultValues));
        }
        
        @SuppressWarnings("null")
        public ImmutableList<String> strList(final String name, final String... defaultValues) throws OXException {
            return optional.strList(name, DEFAULT_SPLIT_PATTERN, Iterators.forArray(defaultValues));
        }
        
        public ImmutableList<String> strList(final String name, final CharMatcher split) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return emptyList();
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableCopyOf(iter);
        }

        public ImmutableList<String> strList(final String name) throws OXException {
            return optional.strList(name, DEFAULT_SPLIT_PATTERN);
        }
        
        public ImmutableList<String> strList(final String name, final CharMatcher split, final Iterable<String> defaultValues) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return immutableCopyOf(defaultValues);
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableCopyOf(iter);
        }

        public ImmutableList<String> strList(final String name, final Iterable<String> defaultValues) throws OXException {
            return optional.strList(name, DEFAULT_SPLIT_PATTERN, defaultValues);
        }
        
        public ImmutableSet<String> strSet(final String name, final CharMatcher split, final Iterator<String> defaultValues) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return immutableSetCopyOf(defaultValues);
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableSetCopyOf(iter);
        }

        public ImmutableSet<String> strSet(final String name, final Iterator<String> defaultValues) throws OXException {
            return optional.strSet(name, DEFAULT_SPLIT_PATTERN, defaultValues);
        }
        
        @SuppressWarnings("null")
        public ImmutableSet<String> strSet(final String name, final CharMatcher split, final String... defaultValues) throws OXException {
            return optional.strSet(name, split, Iterators.forArray(defaultValues));
        }
        
        @SuppressWarnings("null")
        public ImmutableSet<String> strSet(final String name, final String... defaultValues) throws OXException {
            return optional.strSet(name, DEFAULT_SPLIT_PATTERN, Iterators.forArray(defaultValues));
        }
        
        public ImmutableSet<String> strSet(final String name, final CharMatcher split) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return emptySet();
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableSetCopyOf(iter);
        }

        public ImmutableSet<String> strSet(final String name) throws OXException {
            return optional.strSet(name, DEFAULT_SPLIT_PATTERN);
        }
        
        public ImmutableSet<String> strSet(final String name, final CharMatcher split, final Iterable<String> defaultValues) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return immutableSetCopyOf(defaultValues);
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableSetCopyOf(iter);
        }

        public ImmutableSet<String> strSet(final String name, final Iterable<String> defaultValues) throws OXException {
            return optional.strSet(name, DEFAULT_SPLIT_PATTERN, defaultValues);
        }
        
        public Optional<URI> uri(String name) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return absent();
            }
            try {
                return optional(new URI(o.get()));
            } catch (final URISyntaxException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid URI", n(name)));
            }
        }
        
        public Optional<Pattern> pattern(String name) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get());
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return optional(result);
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                return absent();
            }
        }
        
        public Optional<Pattern> pattern(String name, int flags) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get(), flags);
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return optional(result);
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                return absent();
            }
        }
    };
    public final OptionalValueReader optional = new OptionalValueReader();
    public final OptionalValueReader o = optional;

    @SuppressWarnings("null")
    public static final CharMatcher DEFAULT_SPLIT_PATTERN = CharMatcher.anyOf("\r\n;, ");
    
}
