/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.custom.config;

import java.util.HashSet;
import java.util.Set;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.DefaultInterests;
import com.openexchange.config.Interests;

/**
 * Keeps track of configuration properties and files in order to generate
 * a list of {@link Interests}.
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.2.0
 */
public class ConfigTracker {
    private static final String[] STRING_ARRAY_MARKER = new String[0];
    private final Set<String> properties;
    private final Set<String> files;
    
    /**
     * Initializes a new {@link ConfigTracker} that will use the specified
     * {@link Set}s to collect property names and file names.
     * 
     * @param propertiesSet
     * @param filesSet
     */
    public ConfigTracker(final Set<String> propertiesSet, final Set<String> filesSet) {
        this.properties = propertiesSet;
        this.files = filesSet;
    }

    /**
     * Initializes a new {@link ConfigTracker} that will use the specified
     * {@link Set} to collect property names and its own {@link Set} for
     * file names.
     * 
     * @param propertiesSet
     * @deprecated use {@link ConfigTracker#ConfigTracker(Set, Set)} or {@link ConfigTracker#ConfigTracker()}
     */
    // for backwards compatibility
    @Deprecated
    public ConfigTracker(final Set<String> propertiesSet) {
        this.properties = propertiesSet;
        this.files = new HashSet<>();
    }
    public ConfigTracker() {
        this.properties = new HashSet<>();
        this.files = new HashSet<>();
    }
    public @Nullable Interests interests() {
        final DefaultInterests.Builder b = DefaultInterests.builder();
        b.propertiesOfInterest(properties.toArray(STRING_ARRAY_MARKER));
        b.configFileNames(files.toArray(STRING_ARRAY_MARKER));
        return b.build();
    }
    public ConfigTracker trackProperty(final String propertyName) {
        this.properties.add(propertyName);
        return this;
    }
    public ConfigTracker trackConfigFileName(final String filename) {
        this.files.add(filename);
        return this;
    }
}
