/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.custom.config;

import java.io.Closeable;
import java.io.IOException;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.Interests;
import com.openexchange.config.Reloadables;
import com.openexchange.configuration.ConfigurationExceptionCodes;
import com.openexchange.exception.OXException;


/**
 * {@link ConfigHolder}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.2.0
 */
public class ConfigHolder<C> implements Closeable {
    
    private static interface ConfigOrThrow<E> {
        boolean isValid();
        E get() throws OXException;
    }
    private static final class Config<E> implements ConfigOrThrow<E> {
        private final E config;
        public Config(final E config) {
            this.config = config;
        }
        @Override
        public boolean isValid() {
            return true;
        }
        @Override
        public E get() {
            return config;
        }
        @Override
        public int hashCode() {
            return Objects.hash(config);
        }
        @Override
        public boolean equals(@Nullable Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            @SuppressWarnings("rawtypes") final Config other = (Config) obj;
            return Objects.equals(config, other.config);
        }
        
    }
    private static final class Throw<E> implements ConfigOrThrow<E> {
        private final OXException error;
        public Throw(final OXException error) {
            this.error = error;
        }
        @Override
        public boolean isValid() {
            return false;
        }
        @Override
        public E get() throws OXException {
            throw error;
        }
        @Override
        public int hashCode() {
            return Objects.hash(error);
        }
        @Override
        public boolean equals(@Nullable Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            @SuppressWarnings("rawtypes") final Throw other = (Throw) obj;
            return Objects.equals(error, other.error);
        }
    }
    
    private static final <E> ConfigOrThrow<E> load(final ConfigLoader<E> loader,
        final @Nullable String prefix,
        final @Nullable ConfigurationService configurationService,
        final @Nullable ConfigTracker track) {
        final E config;
        {
            final ConfigReader reader;
            {
                if (prefix != null) {
                    if (track != null) {
                        reader = ConfigReader.of(prefix, configurationService, track);
                    } else {
                        reader = ConfigReader.of(prefix, configurationService);
                    }
                } else {
                    if (track != null) {
                        reader = ConfigReader.of(configurationService, track);
                    } else {
                        reader = ConfigReader.of(configurationService);
                    }
                }
            }
            
            try {
                config = loader.load(reader);
            } catch (final OXException e) {
                return new Throw<E>(e);
            } catch (final Exception e) {
                return new Throw<E>(nn(ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, e.getMessage())));
            }
        }
        return new Config<E>(config);
    }
    
    private static final <E> E nn(final @Nullable E value) {
        if (value == null) {
            throw new IllegalArgumentException("value is null");
        } else {
            return value;
        }
    }
    
    public static final <X> ConfigHolder<X> with(final @Nullable ConfigurationService configurationService,
        final ConfigLoader<X> loader) {
        return new ConfigHolder<X>(configurationService, loader);
    }

    public static final <X> ConfigHolder<X> with(final @Nullable ConfigurationService configurationService,
        final @Nullable String prefix,
        final ConfigLoader<X> loader) {
        return new ConfigHolder<X>(configurationService, prefix, loader);
    }
    
    private final AtomicReference<ConfigOrThrow<C>> ref;
    private final ConfigLoader<C> loader;
    private final @Nullable String prefix;
    private final @Nullable Interests interests;
    
    public ConfigHolder(final @Nullable ConfigurationService configurationService,
        final ConfigLoader<C> loader) {
        final ConfigTracker track = new ConfigTracker();
        this.ref = new AtomicReference<>(load(loader, null, configurationService, track));
        this.interests = track.interests();
        this.loader = loader;
        this.prefix = null;
    }

    public ConfigHolder(final @Nullable ConfigurationService configurationService,
        final @Nullable String prefix,
        final ConfigLoader<C> loader) {
        final ConfigTracker track = new ConfigTracker();
        this.ref = new AtomicReference<>(load(loader, prefix, configurationService, track));
        this.interests = track.interests();
        this.loader = loader;
        this.prefix = prefix;
    }
    
    /**
     * Return the configuration object if it is valid.
     * <p>
     * If the configuration object is not valid, that is, if an exception
     * has been thrown while the attempt to load it, the {@link OXException}
     * that was caught during the load attempt is thrown instead.
     * <p>
     * If the {@link ConfigHolder} has already been {@linkplain #close closed},
     * this method will throw an {@link IllegalStateException}.
     * <p>
     * For a safe implementation, verify whether the configuration
     * object is valid first, using {@link #isValid()}.
     * 
     * @return the configuration object if it is valid
     * @throws OXException the exception that was caught while attempting
     *  to load the configuration object, if applicable, if the object is
     *  not {@linkplain #isValid() valid}
     * @throws IllegalStateException if the {@link ConfigHolder} has been
     *  {@linkplain #close closed}
     */
    public final C get() throws OXException {
        final ConfigOrThrow<C> c = this.ref.get();
        if (c != null) {
            return c.get();
        } else {
            throw new IllegalStateException("configuration has been closed");
        }
    }

    /**
     * Returns whether the configuration is valid (i.e. has been loaded without
     * throwing an exception) or not.
     * 
     * @return whether the configuration is valid or not
     * @throws IllegalStateException if the {@link ConfigHolder} has been
     *  {@linkplain #close closed}
     */
    public final boolean isValid() {
        final ConfigOrThrow<C> c = this.ref.get();
        if (c != null) {
            return c.isValid();
        } else {
            throw new IllegalStateException("configuration has been closed");
        }
    }
    
    /**
     * Reload the configuration.
     * @param configService
     * @since 1.2.0
     */
    public final void reload(final @Nullable ConfigurationService configService) {
        if (configService != null) {
            final ConfigOrThrow<C> newConfig = load(loader, prefix, configService, null);
            if (newConfig.isValid()) {
                final ConfigOrThrow<C> oldConfig = ref.getAndSet(newConfig);
                try {
                    if (! newConfig.equals(oldConfig)) {
                        // this is tricky, but as it is a closed private interface with only those
                        // two implementations, we know that it is a Config and not a Throws
                        // and we can hence cast it to Config as we know that a get() shall not
                        // throw an exception here:
                        onConfigChanged(((Config<C>) newConfig).get());
                    } else {
                        onConfigUnchanged();
                    }
                } finally {
                    // in any case, whether they differ or not, "ref" now contains newConfig
                    // and, hence, we must close oldConfig if it implements Closeable
                    if (oldConfig.isValid()) {
                        try {
                            final C old = oldConfig.get();
                            closeConfig(old);
                        } catch (final Exception e) {
                        }
                    }
                }
            } else {
                onConfigInvalid();
            }
        }
    }
    
    /**
     * Override this method if you want to be signalled when the configuration
     * has been reloaded and has changed.
     * <p>
     * The default implementation does nothing.
     * 
     * @param newConfig the new configuration that is now active
     * @since 1.2.0
     */
    protected void onConfigChanged(final C newConfig) {
    }
    
    /**
     * Override this method if you want to be signalled when the configuration
     * has been reloaded and has not changed.
     * <p>
     * The default implementation does nothing.
     * @since 1.2.0
     */
    protected void onConfigUnchanged() {
    }

    /**
     * Override this method if you want to be signalled when the configuration
     * has been reloaded and the new configuration is not valid.
     * <p>
     * The default implementation does nothing.
     * @since 1.2.0
     */
    protected void onConfigInvalid() {
    }
    
    /**
     * Override this method to customize how the configuration objects
     * ought to be closed.
     * <p>
     * The default implementation calls {@link Closeable#close()} if the
     * configuration object implements the {@link Closeable} interface.
     * 
     * @param config the configuration object to close
     * @throws Exception
     * @since 1.2.0
     */
    protected void closeConfig(final C config) throws Exception {
        if (config instanceof Closeable) {
            ((Closeable) config).close();
        }
    }

    @Override
    public void close() throws IOException {
        final ConfigOrThrow<C> c = ref.getAndSet(null);
        if (c != null) {
            // this is tricky, but as it is a closed private interface with only those
            // two implementations, we know that it is a Config and not a Throws
            // and we can hence cast it to Config as we know that a get() shall not
            // throw an exception here:
            if (c instanceof Config) {
                final C obj = ((Config<C>) c).config;
                try {
                    closeConfig(obj);
                } catch (IOException e) {
                    throw e;
                } catch (RuntimeException e) {
                    throw e;
                } catch (Exception e) {
                    throw new IOException("unexpected exception while closing", e);
                }
            }
        }
    }

    /**
     * Prefix the specified property name strings with the property
     * name prefix that has been passed to the {@link ConfigHolder}
     * constructor or factory method.
     * 
     * @param propertyNames property name strings to prefix with
     *  the {@link ConfigHolder} property name prefix
     * @return the specified property names, prefixed with the
     *  {@link ConfigHolder} property name prefix
     * @since 1.2.0
     */
    @SuppressWarnings("null") // we know Stream.toArray() never returns null
    public String[] prefix(String... propertyNames) {
        return Stream.of(propertyNames)
            .map(s -> s != null ? ConfigReader.prefix(prefix, s) : null)
            .toArray(String[]::new);
    }

    /**
     * Create an {@link Interests} with the specified property names by prefixing
     * them with the {@link ConfigHolder} property name prefix.
     * <p>
     * It is recommended to use the automatic collection of property names
     * and configuration files instead, using {@link #interests()}.
     * 
     * @return {@link Interests} with the specified property names prefixed
     *  with the {@link ConfigHolder} property name prefix
     * @since 1.2.0
     */
    public @Nullable Interests interests(String propertyName, String... morePropertyNames) {
        final Set<String> props;
        {
            props = new HashSet<>(1 + (morePropertyNames.length));
            props.add(propertyName);
            for (final String n : morePropertyNames) {
                if (n != null) {
                    props.add(n);
                }
            }
        }
        return Reloadables.interestsForProperties(props
            .stream()
            .map(s -> s != null ? ConfigReader.prefix(prefix, s) : null)
            .toArray(String[]::new));
    }
    
    /**
     * Returns interest in the properties that have been queried using the {@link ConfigReader} in
     * the {@link ConfigLoader}.
     *  
     * @return interest in the properties that have been queried using the {@link ConfigReader} in
     *  the {@link ConfigLoader}
     * @since 1.2.0
     */
    public @Nullable Interests interests() {
        return interests;
    }
}
