/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.custom.collect;

import static com.openexchange.util.custom.base.NullUtil.build;
import static com.openexchange.util.custom.base.NullUtil.emptyMap;
import static com.openexchange.util.custom.base.NullUtil.immutableCopyOf;
import static com.openexchange.util.custom.base.NullUtil.mapBuilder;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.Map;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterators;
import com.openexchange.annotation.Nullable;


/**
 * {@link ImmutableDictionary}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.1
 */
public final class ImmutableDictionary<K,V> extends Dictionary<K,V> {
    
    public static final <X,Y> ImmutableDictionary<X,Y> of(final @Nullable ImmutableMap<X,Y> map) {
        if (map != null) {
            return new ImmutableDictionary<>(map);
        } else {
            return new ImmutableDictionary<>(emptyMap());
        }
    }

    public static final <X,Y> ImmutableDictionary<X,Y> copyOf(final @Nullable Map<X,Y> map) {
        if (map != null) {
            return new ImmutableDictionary<>(immutableCopyOf(map));
        } else {
            return new ImmutableDictionary<>(emptyMap());
        }
    }
    
    public static final <X,Y> ImmutableDictionary<X,Y> copyOf(final @Nullable Dictionary<X,Y> dictionary) {
        if (dictionary != null) {
            final ImmutableMap.Builder<X, Y> builder = mapBuilder();
            final Enumeration<X> e = dictionary.keys();
            while (e.hasMoreElements()) {
                final X key = e.nextElement();
                if (key != null) {
                    final Y value = dictionary.get(key);
                    if (value != null) {
                        builder.put(key, value);
                    }
                }
            }
            return new ImmutableDictionary<>(build(builder));
        } else {
            return new ImmutableDictionary<>(emptyMap());
        }
    }
    
    private final ImmutableMap<K, V> map;
    
    private ImmutableDictionary(final ImmutableMap<K, V> map) {
        this.map = map;
    }

    @Override
    public int size() {
        return map.size();
    }

    @Override
    public boolean isEmpty() {
        return map.isEmpty();
    }

    @SuppressWarnings("null")
    @Override
    public Enumeration<K> keys() {
        return Iterators.asEnumeration(map.keySet().iterator());
    }

    @SuppressWarnings("null")
    @Override
    public Enumeration<V> elements() {
        return Iterators.asEnumeration(map.values().iterator());
    }

    @Override
    public @Nullable V get(@Nullable Object key) {
        return map.get(key);
    }

    @Override
    public V put(@Nullable K key, @Nullable V value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public V remove(@Nullable Object key) {
        throw new UnsupportedOperationException();
    }

}
