/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import java.util.Dictionary;


import javax.annotation.concurrent.ThreadSafe;
import org.osgi.framework.BundleContext;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;
import com.openexchange.util.activator.ServiceDependencyResolver;
import com.openexchange.util.activator.ServiceRegistrationHandler;

/**
 * This abstract service tracker can wait for some service and register another one if all become available.
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 */
@ThreadSafe
public class ServiceWithDependenciesServiceRegisterer<S, T extends S>
    extends InstanceFactoryServiceAwareDependentServiceRegistererTemplate<S, T> {

    private final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler;        
    private final ServiceWithDependencies<T> service;
    
    public ServiceWithDependenciesServiceRegisterer(final BundleContext context,
        final ActivatorRegistry registry,
        final ServiceWithDependencies<T> service,
        final ImmutableSet<Class<?>> serviceTypes,
        final String serviceInstanceName,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        super(context,
            serviceTypes,
            serviceInstanceName,
            properties,
            service.mandatoryServiceDependencies(),
            service.optionalServiceDependencies(),
            RegistrationTools.serviceSets(context, registry, service.serviceSetClasses()),
            RegistrationTools.serviceListings(context, registry, service.serviceListingClasses()),
            RegistrationTools.collectPropertiesOfInterest(service.mandatoryServiceDependencies(), service.optionalServiceDependencies())
        );
        this.serviceRegistrationHandler = serviceRegistrationHandler;
        this.service = service;
    }
    
    @Override
    protected T createInstance(final ServiceDependencyResolver resolver) throws Exception {
        final T serviceInstance = service.createInstance(resolver);
        if (serviceRegistrationHandler != null) {
            serviceRegistrationHandler.onServiceRegistered(serviceInstance);
        }
        return serviceInstance;
    }
}
