/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.logger;
import static com.openexchange.util.custom.base.NullUtil.notNull;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;


import org.slf4j.Logger;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXActionServiceFactory;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.exception.OXException;
import com.openexchange.tools.servlet.AjaxExceptionCodes;
import com.openexchange.util.activator.AjaxAction;
import com.openexchange.util.activator.DependencyProvider;
import com.openexchange.util.activator.RegistrationException;
import com.openexchange.util.activator.ServiceDependencies;
import com.openexchange.util.activator.ServiceDependencyResolver;

/**
 * {@link AJAXActionServiceFactory} implementation that uses the {@code @}{@link AjaxAction}
 * annotation to discover {@link AJAXActionService} implementations.
 * <p>
 * Note that if a class is not annotated with {@code @}{@link AJAXActionService}, it will
 * attempt to infer its action from the beginning of its class name (e.g. {@code GetFooAction}
 * yields {@code "get"}), but that approach is less deterministic and the preferred method is
 * to use the {@code @}{@link AjaxAction} annotation.
 * <p>
 * See {@link #CLASS_NAME_PREFIX_ACTION} for a list of supported class name prefixes.
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.2
 */
public final class DiscoveringActionFactory implements AJAXActionServiceFactory, DependencyProvider, Closeable {
    
    private static final Logger LOG = logger(DiscoveringActionFactory.class);
    
    @SuppressWarnings("null")
    public static final @NonNull ImmutableSet<String> CLASS_NAME_PREFIX_ACTION = ImmutableSet.of(
        "Default", "Get", "Set", "List", "Delete", "All", "Copy", "New", "Move", "Edit", "Update", "Search"
    );
    
    private final String module;
    private final ImmutableMap<String, Class<? extends AJAXActionService>> actions;
    private final ServiceDependencies needs;
    private final AtomicReference<ImmutableMap<String, AJAXActionService>> instancesRef; 
    
    public DiscoveringActionFactory(final String module, ImmutableSet<Class<? extends AJAXActionService>> ajaxActionServiceClasses) {
        this.module = module;

        {
            final ImmutableMap.Builder<String, Class<? extends AJAXActionService>> actionsBuilder = ImmutableMap.builder();
            final List<ServiceDependencies> deps = new ArrayList<>(ajaxActionServiceClasses.size());
            for (final Class<? extends AJAXActionService> c : ajaxActionServiceClasses) {
                final ImmutableSet<String> actions;
                {
                    final AjaxAction ajaxAction = c.getAnnotation(AjaxAction.class);
                    if (ajaxAction != null) {
                        actions = ImmutableSet.copyOf(notNull(ajaxAction.value()));
                    } else {
                        String a = null;
                        final Iterator<String> iter = CLASS_NAME_PREFIX_ACTION.iterator();
                        while (iter.hasNext() && a == null) {
                            final String prefix = iter.next();
                            if (c.getSimpleName().startsWith(prefix)) {
                                a = prefix.toLowerCase(Locale.US);
                            }
                        }
                        if (a != null) {
                            actions = ImmutableSet.of(a);                            
                        } else {
                            final String msg = f(
                                "%s implementation class '%s' has no @%s annotation and its class name does not start with a well-known action prefix in [%s]",
                                AJAXActionService.class.getSimpleName(),
                                c.getName(),
                                AjaxAction.class.getSimpleName(),
                                Joiner.on(", ").join(CLASS_NAME_PREFIX_ACTION));
                            LOG.error(msg);
                            throw new RegistrationException(msg);
                        }
                    }
                }
                for (final String a : actions) {
                    if (a != null) {
                        actionsBuilder.put(a, c);
                    }
                }
                
                final ServiceWithDependencies<?> d = ConstructorServiceWithDependencies.find(c, AJAXActionService.class);
                deps.add(d);
            }
            
            {
                final Set<MandatoryServiceDependency<?>> mandatoryDependenciesBuilder = new HashSet<>();
                final Set<OptionalServiceDependency<?>> optionalDependenciesBuilder = new HashSet<>();
                final Set<Class<?>> serviceSetsBuilder = new HashSet<>();
                final Set<Class<?>> serviceListingsBuilder = new HashSet<>();
                for (final ServiceDependencies d : deps) {
                    mandatoryDependenciesBuilder.addAll(d.mandatoryServiceDependencies());
                    optionalDependenciesBuilder.addAll(d.optionalServiceDependencies());
                    serviceSetsBuilder.addAll(d.serviceSetClasses());
                    serviceListingsBuilder.addAll(d.serviceListingClasses());
                }
                    
                @SuppressWarnings("null")
                final @NonNull ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies = ImmutableSet.copyOf(mandatoryDependenciesBuilder);
                @SuppressWarnings("null")
                final @NonNull ImmutableSet<OptionalServiceDependency<?>> optionalDependencies = ImmutableSet.copyOf(optionalDependenciesBuilder);
                @SuppressWarnings("null")
                final @NonNull ImmutableSet<Class<?>> serviceSets = ImmutableSet.copyOf(serviceSetsBuilder);
                @SuppressWarnings("null")
                final @NonNull ImmutableSet<Class<?>> serviceListings = ImmutableSet.copyOf(serviceListingsBuilder);
                
                this.needs = new ImmutableServiceDependencies(
                    mandatoryDependencies,
                    optionalDependencies,
                    serviceSets,
                    serviceListings
                );                
            }
            @SuppressWarnings("null")
            final @NonNull ImmutableMap<String, Class<? extends AJAXActionService>> a = actionsBuilder.build();
            this.actions = a;
        }
        this.instancesRef = new AtomicReference<>(ImmutableMap.<String,AJAXActionService>of());
    }

    @Override
    public AJAXActionService createActionService(final @Nullable String action) throws OXException {
        final AJAXActionService service = instancesRef.get().get(action);
        if (service != null) {
            return service;
        } else {
            throw AjaxExceptionCodes.UNKNOWN_ACTION_IN_MODULE.create(action, module);
        }
    }

    @Override
    public ServiceDependencies dependsOn() {
        return needs;
    }
    
    @Override
    public void onDependenciesAvailable(ServiceDependencyResolver resolver) {
        final ImmutableMap.Builder<String, AJAXActionService> b = ImmutableMap.builder();
        for (final Map.Entry<String, Class<? extends AJAXActionService>> e : actions.entrySet()) {
            final String name = e.getKey();
            if (name != null) {
                final Class<? extends AJAXActionService> serviceClass = e.getValue();
                if (serviceClass != null) {
                    final ServiceWithDependencies<?> s = ConstructorServiceWithDependencies.find(serviceClass, AJAXActionService.class);
                    b.put(name, (AJAXActionService) s.createInstance(resolver));
                }
            }
        }
        close(this.instancesRef.getAndSet(b.build()));
    }

    @Override
    public void close() throws IOException {
        final ImmutableMap<String, AJAXActionService> oldValue = instancesRef.getAndSet(ImmutableMap.<String, AJAXActionService>of());
        close(oldValue);
    }
    
    private static void close(final @Nullable ImmutableMap<String, AJAXActionService> map) {
        if (map != null) {
            for (final AJAXActionService s : map.values()) {
                if (s != null) {
                    if (s instanceof Closeable) {
                        try {
                            ((Closeable) s).close();
                        } catch (final Exception e) {
                            LOG.error(f("failed to close %s (suppressed)", className(s), e));
                        }
                    }
                }
            }            
        }
    }

}
