package com.openexchange.usm.util;

import java.util.UUID;
import com.openexchange.usm.api.exceptions.InvalidUUIDException;

/**
 * Utility class for computed UUIDs. 
 * @author ldo
 *
 */
public class UUIDToolkit {
    
    private static final long GENERATED_UUID_FIXED_LSB_BITS = 0xFFFFFF0000000000L;
    
    
    /**
     * Checks if the given uuid is a computed one. Clients are not allowed to use these uuids for own creations. 
     * @param uuid
     * @param contextUUID
     * @return
     */
    public static boolean isComputedUUID(UUID uuid, UUID contextUUID)  {
        return (contextUUID.getMostSignificantBits() == uuid.getMostSignificantBits() && (((contextUUID
                .getLeastSignificantBits() ^ uuid.getLeastSignificantBits()) & GENERATED_UUID_FIXED_LSB_BITS) == 0L));
    }
    
    /**
     * Generates a UUID for server objects based on the current context UUID, the objects content type and id.
     * @param contextUUID
     * @param contentType
     * @param objectId
     * @return
     */
    public static UUID generateUUID(UUID contextUUID, int contentType, int objectId) {
        long lsb = contextUUID.getLeastSignificantBits();
        lsb ^= (objectId | ((long) contentType) << 32);
        return  new UUID(contextUUID.getMostSignificantBits(), lsb);
    }
    
    
    /**
     * Creates a modified UUID from a base UUID and a given OX user id. The algorithm uses XOR, so this method can be used both ways, i.e.
     * given a modified UUID and OX user id, it reports back the original unmodified UUID. If either the UUID is null or the userID is 0,
     * this method returns the unmodified UUID.
     * 
     * @param uuid
     * @param userID
     * @return
     */
    public static UUID getModifedUUID(UUID uuid, int userID) {
        if (uuid == null || userID == 0)
            return uuid;
        return new UUID(uuid.getMostSignificantBits() ^ userID, uuid.getLeastSignificantBits());
    }
    
    /**
     * Creates a UUID based on a given uuid string representation. 
     * @param uuid
     * @return
     * @throws InvalidUUIDException
     */
    public static UUID extractUUIDFromString(String uuid) throws InvalidUUIDException {
        try {
            if (uuid != null)
                return UUID.fromString(uuid);
        } catch (IllegalArgumentException ignored) {
            // Fall through to error handling
        }
        throw new InvalidUUIDException(USMUtilErrorCode.INVALID_UUID, "Invalid uuid: " + uuid);
    }
    
}
