/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.util;

import org.apache.commons.logging.Log;
import com.openexchange.usm.api.session.Session;

/**
 * This class can be used by USM protocol implementations to check for clients that behave badly, i.e. perform too many unnecessary synchronizations.
 * If an unnecessary access is reported, an AccessCounter is created for that session. It will be removed if no unnecessary access was detected
 * within the test interval. If the limit of unnecessary accesses has been reached, blocking for that session will be enabled for the refusal interval.
 * 
 * @author afe
 *
 */
public class AccessLimiter {

	// Constant for the key under which the AccessCounter is stored as custom field in the USM Session 
	private final static String _USM_ACCESS_LIMITER = "usm.AccessLimiter";

	private final Log _journal;

	private final int _testInterval;
	private final int _refusalInterval;
	private final int _maxAccesses;

	public AccessLimiter(Log journal, int testInterval, int maxAccesses, int refusalInterval) {
		_journal = journal;
		_testInterval = testInterval;
		_refusalInterval = refusalInterval;
		_maxAccesses = maxAccesses;
	}

	/* (non-Javadoc)
	 * @see com.openexchange.usm.json.access.AccessLimiter#isUSMAccessDenied(com.openexchange.usm.api.session.Session)
	 */
	public boolean isUSMAccessDenied(Session session) {
		if (!isCheckEnabled())
			return false;
		Object o = session.getCustomProperty(_USM_ACCESS_LIMITER);
		if (!(o instanceof AccessCounter))
			return false;
		return ((AccessCounter) o).isAccessDenied(session);
	}

	public void removeCounter(Session session) {
		session.setCustomProperty(_USM_ACCESS_LIMITER, null);
	}

	/* (non-Javadoc)
	 * @see com.openexchange.usm.json.access.AccessLimiter#unnecessaryAccessEncountered(com.openexchange.usm.api.session.Session)
	 */
	public void badAccessEncountered(Session session) {
		if (isCheckEnabled())
			getAccessCounter(session).onUnnecessaryAccess(session);
	}

	private boolean isCheckEnabled() {
		return _testInterval > 0 && _refusalInterval > 0 && _maxAccesses > 0;
	}

	private AccessCounter getAccessCounter(Session session) {
		Object o = session.getCustomProperty(_USM_ACCESS_LIMITER);
		if (o instanceof AccessCounter)
			return (AccessCounter) o;
		AccessCounter p = new AccessCounter(this);
		session.setCustomProperty(_USM_ACCESS_LIMITER, p);
		return p;
	}

	public int getTestInterval() {
		return _testInterval;
	}

	public int getRefusalInterval() {
		return _refusalInterval;
	}

	public int getMaxAccesses() {
		return _maxAccesses;
	}

	public Log getJournal() {
		return _journal;
	}
}
