/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.tasks;

import org.apache.commons.logging.Log;
import com.openexchange.usm.session.impl.SessionImpl;
import com.openexchange.usm.session.impl.SessionManagerImpl;
import com.openexchange.usm.session.impl.SessionReference;
import com.openexchange.usm.session.storage.SessionStorage;

/**
 * Task that is executed regularly and removes all USM sessions from memory that have not been accessed for the configured limit.
 * {@link USMSessionCacheCleanupTask}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class USMSessionCacheCleanupTask implements Runnable {

    private final SessionManagerImpl _sessionManager;

    public USMSessionCacheCleanupTask(SessionManagerImpl manager) {
        _sessionManager = manager;
    }

    /**
     * Removes all Sessions from memory that have not been accessed within the configured limit for Sessions. Removes all SyncStates that
     * have not been accessed within the configured limit for SyncStates.
     */
    @Override
    public void run() {
        Log log = _sessionManager.getJournal();
        try {
            long now = System.currentTimeMillis();
            int sessionCacheTimeout = _sessionManager.getSessionCacheTimeout();
            int syncStateCacheTimeout = _sessionManager.getSyncStateCacheTimeout();
            long sessionLimit = now - sessionCacheTimeout * 1000L;
            long syncStateLimit = now - syncStateCacheTimeout * 1000L;
            if (log.isDebugEnabled())
                log.debug("Removing USM Sessions/SyncStates from memory cache that have not been accessed for " + sessionCacheTimeout + '/' + syncStateCacheTimeout + " seconds");
            SessionStorage storage = _sessionManager.getSessionStorage();
            int removedSessions = 0;
            for (SessionReference ref : storage.getSessionReferences()) {
                if (ref.getLastAccess() < sessionLimit) {
                    SessionImpl session = ref.get();
                    if (session != null) {
                        storage.removeSession(session);
                        removedSessions++;
                    }
                }
            }
            int removedSyncStates = _sessionManager.getSyncStateCacheProvider().removeUnusedSyncStates(syncStateLimit);
            int jvmRemovals = storage.removeClearedReferences();
            _sessionManager.updateCacheRemovals(removedSessions, jvmRemovals, removedSyncStates);
        } catch (Exception e) {
            log.error("Exception occurred while cleaning up memory caches", e);
        }
    }
}
