/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.*;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.exceptions.TemporaryDownOrBusyException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.ServletConstants;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;

/**
 * Handler for the USM-Retrieve OX Ids command.
 * 
 * @author ldo
 *
 */
public class ResolveIDsHandler extends BaseCommandHandler {

	private static final String NOT_FOUND = "Not Found";
	private static final String[] REQUIRED_PARAMETERS = { USER, DEVICE, PASSWORD, OBJECT_UUIDS };
	private static final String[] OPTIONAL_PARAMETERS = { FOLDER_UUID };

    public ResolveIDsHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {
		try {
			String user = getStringParameter(USER);
			String password = getStringParameter(PASSWORD);
			String device = getStringParameter(DEVICE);
			
			boolean sessionAlreadyInDB = _servlet.getSessionManager().existsSessionInDB(user, password, ServletConstants.PROTOCOL, device);
			
			if (!sessionAlreadyInDB)
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.RESOLVE_IDS_INVALID_DEVICE,
						ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Unknown device ID");

			_session = _servlet.getSession(user, password, device, _currentHttpRequest.getRemoteAddr(), getXHeadersFromCurrentRequest());

			String folderUUID = getStringParameter(FOLDER_UUID);
			JSONArray objectUUIDs = getJSONArray(_parameters, OBJECT_UUIDS);
			JSONObject resultData = new JSONObject();
			JSONObject objectIds = new JSONObject();
			DataObject[] cachedObjects = null;
			if (folderUUID != null) {
				Folder folder = null;
				try {
					folder = getFolderByUUID(folderUUID);
				} catch (USMJSONAPIException ignored) {
					//do nothing
				}
				resultData.put(FOLDER_ID, (folder != null) ? folder.getID() : NOT_FOUND);
				if (folder != null)
					cachedObjects = _session.getCachedFolderElements(folder.getID(), folder.getElementsContentType());
			} else {
				cachedObjects = _session.getCachedFolders();
			}

			DataObjectSet objectSet = new DataObjectSet(cachedObjects);
			int length = objectUUIDs.length();
			for (int i = 0; i < length; i++) {
				String uuid = objectUUIDs.getString(i);
				DataObject object = objectSet.get(UUID.fromString(uuid));
				objectIds.put(uuid, (object != null) ? object.getID() : NOT_FOUND);
			}
			resultData.put(OBJECTIDS, objectIds);
			return new ResponseObject(ResponseStatusCode.SUCCESS, resultData);
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (IllegalArgumentException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.RESOLVE_IDS_INVALID_UUID,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Invalid UUID", e);
        } catch (TemporaryDownOrBusyException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.RESOLVE_IDS_TEMPORARY_DOWN_OR_BUSY, e);
        } catch (USMException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.RESOLVE_IDS_LOGIN_UNSUCCESSFUL,
					ResponseStatusCode.ACCESS_DENIED, "Can not login.", e);
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(
					ConnectorBundleErrorCodes.RESOLVE_IDS_JSON_ERROR_ON_CREATE_RESULT, e);
		}
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}
}
