/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.OBJECTIDS;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.InvalidUUIDException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.UUIDToolkit;

/**
 * Handler for the USM-locateObjects command.
 * 
 * @author afe
 */

public class LocateObjectsHandler extends NormalCommandHandler {

    private static final String[] REQUIRED_PARAMETERS = { SESSIONID, OBJECTIDS };

    private static final String[] OPTIONAL_PARAMETERS = {};

    private static final UUID FOLDER_HIERARCHY_UUID = new UUID(0L, 0L);

    private static class CachedParentData {

        private long _timestamp;

        private UUID _uuid;

        public CachedParentData() {
            _timestamp = 0L;
            _uuid = null;
        }

        public void update(long timestamp, UUID uuid) {
            if (_timestamp < timestamp) {
                _timestamp = timestamp;
                _uuid = uuid;
            }
        }

        public UUID getUUID() {
            return _uuid;
        }
    }

    public LocateObjectsHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        JSONArray objectUUIDs = getJSONArray(_parameters, OBJECTIDS);
        Map<UUID, CachedParentData> parentMap = new HashMap<UUID, CachedParentData>();
        int length = objectUUIDs.length();
        try {
            for (int i = 0; i < length; i++)
                parentMap.put(UUIDToolkit.extractUUIDFromString(getString(objectUUIDs, i)), new CachedParentData());
            Folder[] folders = _session.getCachedFolders();
            if (folders != null) {
                for (Folder f : folders) {
                    updateInMap(parentMap, f, FOLDER_HIERARCHY_UUID);
                    DataObject[] elements = _session.getCachedFolderElements(f.getID(), f.getElementsContentType());
                    if (elements != null) {
                        for (DataObject e : elements)
                            updateInMap(parentMap, e, f.getUUID());
                    }
                }
            }
            JSONObject response = new JSONObject();
            for (Map.Entry<UUID, CachedParentData> e : parentMap.entrySet()) {
                UUID uuid = e.getKey();
                UUID parent = e.getValue().getUUID();
				response.put(uuid.toString(), (parent == null) ? "not found"
						: (parent == FOLDER_HIERARCHY_UUID) ? "folder" : parent.toString());
            }
            return new ResponseObject(ResponseStatusCode.SUCCESS, response);
        } catch (StorageAccessException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.LOCATE_OBJECTS_DB_ERROR, e);
        } catch (USMStorageException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.LOCATE_OBJECTS_SQL_ERROR, e);
        } catch (JSONException e) {
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.LOCATE_OBJECTS_JSON_ERROR, e);
        } catch (InvalidUUIDException e) {
            throw USMJSONAPIException.createInvalidUUIDException(e);
        }
    }

    private void updateInMap(Map<UUID, CachedParentData> parentMap, DataObject cachedObject, UUID parentUUID) {
        CachedParentData data = parentMap.get(cachedObject.getUUID());
        if (data != null)
            data.update(cachedObject.getTimestamp(), parentUUID);
    }

    @Override
    protected String[] getOptionalParameters() {
        return OPTIONAL_PARAMETERS;
    }

    @Override
    protected String[] getRequiredParameters() {
        return REQUIRED_PARAMETERS;
    }

}
