/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.database.ox.update;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashSet;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.update.PerformParameters;
import com.openexchange.groupware.update.UpdateTaskAdapter;
import com.openexchange.usm.database.ox.USMUpdateExceptionCodes;
import com.openexchange.usm.database.ox.USMUpdateTaskProvider;
import com.openexchange.usm.util.Toolkit;

public abstract class USMBaseUpdateTaskV2 extends UpdateTaskAdapter {

    private static final Log LOG = LogFactory.getLog(USMBaseUpdateTaskV2.class);

    protected final USMUpdateTaskProvider _provider;

    public USMBaseUpdateTaskV2(USMUpdateTaskProvider provider) {
        super();
        _provider = provider;
    }

    @Override
    public void perform(PerformParameters params) throws OXException {
        String taskName = getClass().getName();
        int i = taskName.lastIndexOf('.');
        if (i >= 0) {
            taskName = taskName.substring(i + 1);
        }
        LOG.debug("Performing UpdateTask '" + taskName);
        doPerform(params);
        LOG.info("UpdateTask '" + taskName + "' successfully performed!");
    }

    protected void enableAutocommit(Connection con, Statement stmt) {
        Toolkit.close(stmt);
        try {
            if (con != null && !con.isClosed()) {
                con.setAutoCommit(true);
            }
        } catch (final SQLException e) {
            LOG.error(e.getMessage(), e);
        }
    }

    protected void rollback(Connection con) {
        try {
            if (con != null && !con.isClosed()) {
                con.rollback();
            }
        } catch (final SQLException e2) {
            LOG.error(e2.getMessage(), e2);
        }
    }

    /**
     * The object type "TABLE"
     */
    private static final String[] TYPES = { "TABLE" };

    /**
     * Check a table's existence
     *
     * @param tableName The table name to check
     * @param dbmd The database's meta data
     * @return <code>true</code> if table exists; otherwise <code>false</code>
     * @throws SQLException If a SQL error occurs
     */
    public static boolean tableExists(final String tableName, final DatabaseMetaData dbmd) throws SQLException {
        ResultSet resultSet = null;
        try {
            resultSet = dbmd.getTables(null, null, tableName, TYPES);
            if (resultSet == null) {
                return false;
            }
            return resultSet.next();
        } finally {
            Toolkit.close(resultSet);
        }
    }

    public static Set<String> getTableIndexes(String tableName, Connection conn) throws SQLException {
        Set<String> indexes = new HashSet<String>();
        ResultSet indexInfo = null;
        try {
            indexInfo = conn.getMetaData().getIndexInfo(conn.getCatalog(), null, tableName, false, false);
            while(indexInfo.next()) {
                String indexName = indexInfo.getString("INDEX_NAME");
                if(indexName != null) {
                    indexes.add(indexName);
                }
            }
        } finally {
            Toolkit.close(indexInfo);
        }
        return indexes;
    }

    protected void createTable(final String tablename, final String sqlCreate, final Connection con)
        throws OXException {
        Statement stmt = null;
        try {
            if (!tableExists(tablename, con.getMetaData())) {
                stmt = con.createStatement();
                stmt.executeUpdate(sqlCreate);
            }
        } catch (SQLException e) {
            throw USMUpdateExceptionCodes.SQL_ERROR.create(e);
        } finally {
            Toolkit.close(stmt);
        }
    }

    protected abstract void doPerform(PerformParameters params) throws OXException;
}
