/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.contenttypes.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;
import com.openexchange.usm.api.contenttypes.folder.OXFolderContent;
import com.openexchange.usm.api.contenttypes.resource.ResourceInputStream;
import com.openexchange.usm.api.contenttypes.transferhandlers.ContentTypeTransferHandler;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.datatypes.PIMAttachments;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.ConflictingChangeException;
import com.openexchange.usm.api.exceptions.FolderNotFoundException;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.TemporaryDownOrBusyException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMIllegalArgumentException;
import com.openexchange.usm.api.exceptions.UnsupportedContentOperationException;
import com.openexchange.usm.api.ox.json.JSONResult;
import com.openexchange.usm.api.ox.json.JSONResultType;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.contenttypes.attachments.PIMAttachmentTransferHandler;
import com.openexchange.usm.datatypes.tasks.calendar.CommonCalendarTasksFieldNames;
import com.openexchange.usm.datatypes.tasks.calendar.UserParticipantObject;
import com.openexchange.usm.session.dataobject.DataObjectSet;

/**
 * Abstract base class for Transfer handler implementations. Contains utility methods and simple default implementations for standard OX
 * access calls.
 * 
 * @author afe
 */
public abstract class AbstractTransferHandler implements ContentTypeTransferHandler {
    
    protected static final Log LOG = LogFactory.getLog(AbstractTransferHandler.class);

    protected static final String _LAST_MODIFIED_COLUMN = "5";

    protected static final String RIGHT_HAND_LIMIT = "right_hand_limit";

    protected static final String LEFT_HAND_LIMIT = "left_hand_limit";

    private static final String ERROR_PARAMS = "error_params";

    public static final boolean CHECK_ACCESS_RIGHTS_BEFORE_FOLDER_READ = true;

    private static final String OWN_RIGHTS = "own_rights";

    private static final String OCCURRENCE = "occurrence";

    protected static final String _COLUMNS_ID_AND_LAST_MODIFIED = "1,5";

    private static final int _MAX_OBJECTS_PER_LIST = 500;
    
    protected static final OXFolderContent _EMPTY_FOLDER_CONTENT = new OXFolderContent(true);

    protected final ContentType _contentType;

    protected final OXJSONAccess _ajaxAccess;

    private final PIMAttachmentTransferHandler _pimAttachmentsTransferHandler;

    /**
     * Initializes a new {@link AbstractTransferHandler}.
     * @param contentType
     * @param ajaxAccess
     */
    protected AbstractTransferHandler(ContentType contentType, OXJSONAccess ajaxAccess) {
        _contentType = contentType;
        _ajaxAccess = ajaxAccess;
        _pimAttachmentsTransferHandler = new PIMAttachmentTransferHandler(ajaxAccess);
    }

    /**
     * Creates a comma separated list of columns for the given ContentType. Each column is specified by a numeric identifier. This method
     * assumes that the first field is the ID field of the ContentType and makes sure that it is always added to the list.
     * 
     * @param requestedFields
     * @return
     * @throws UnsupportedContentOperationException
     */
    protected String createColumnsParameter(BitSet requestedFields) throws UnsupportedContentOperationException {
        int len = checkAndGetRequestedFieldLength(requestedFields);
        StringBuilder columns = new StringBuilder(100);
        ContentTypeField[] fields = _contentType.getFields();
        for (int i = 0; i < len; i++) {
            if (requestedFields.get(i)) {
                if (fields[i].getFieldID() < 0) {
					throw new UnsupportedContentOperationException(USMContentTypesUtilErrorCodes.NOT_SUPPORTED_FIELD,
                        "Field " + fields[i].getFieldName() + " is not supported");
                }
                if (columns.length() > 0)
                    columns.append(',');
                appendColumns(columns, fields[i]);
            }
        }
        return columns.toString();
    }

    /**
     * Adds columns of a field to the requested fields. Can be overwritten by implementation for special handling of connected data.
     * 
     * @param columns
     * @param field the index
     */
    protected void appendColumns(StringBuilder columns, ContentTypeField field) {
        columns.append(field.getFieldID());
    }

    public void confirm(DataObject object, int response) throws AuthenticationFailedException, OXCommunicationException {
        confirm(object, response, "", (Number)null);
    }

	public void confirm(DataObject object, int response, String confirmMessage, Number occurrencePosition)
	    throws AuthenticationFailedException, OXCommunicationException {

        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Confirming appointment or task " + object.getSession() + object.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.ID, object.getID());
        parameters.put(CommonConstants.FOLDER, object.getParentFolderID());
        parameters.put(CommonConstants.TIMESTAMP, String.valueOf(object.getTimestamp()));
        if (occurrencePosition != null) {
            parameters.put(OCCURRENCE, String.valueOf(occurrencePosition.intValue()));
        }

        JSONObject requestBody = new JSONObject();
        try {
            requestBody.put("confirmation", response);
            if (!confirmMessage.isEmpty()) {
                requestBody.put("confirmmessage", confirmMessage);
            }
            requestBody.put(CommonConstants.ID, computeFolderOwnerID(object));
        } catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER1, object,
					"can not create request body for confirming appoinment or task", e);
        }
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_CONFIRM, object.getSession(),
				parameters, requestBody);

        if (result.getResultType() == JSONResultType.Error) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER2,
					"Could not confirm appointment or task with id " + object.getID(), result.getJSONObject());
        } else if (result.getResultType() != JSONResultType.JSONObject) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER3,
                "Invalid result type returned, expected JSONObject, was " + result.getResultType());
        } else {
            readOptionalTimestamp(result.getJSONObject(), object);
        }
    }

	public void confirm(DataObject object, int response, String confirmMessage, String participantId)
			throws AuthenticationFailedException, OXCommunicationException {
        if (LOG.isDebugEnabled())
			LOG.debug(object.getSession() + " Confirming appointment " + object.getSession() + object.getID() + " for participant: "
					+ participantId);
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.ID, object.getID());
        parameters.put(CommonConstants.FOLDER, object.getParentFolderID());
        parameters.put(CommonConstants.TIMESTAMP, String.valueOf(object.getTimestamp()));

        JSONObject requestBody = new JSONObject();
        try {
            requestBody.put("confirmation", response);
            if (!confirmMessage.isEmpty()) {
                requestBody.put("confirmmessage", confirmMessage);
            }
            requestBody.put("mail", participantId);
            requestBody.put("type", 5);
        } catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER4, object,
					"can not create request body for confirming appoinment for participant", e);
        }
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_CONFIRM, object.getSession(),
				parameters, requestBody);

        if (result.getResultType() == JSONResultType.Error) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER5,
                "Could not confirm appointment " + object.getID() + " for participant: " + participantId,
                result.getJSONObject());
        } else if (result.getResultType() != JSONResultType.JSONObject) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.CONFIRM_ERROR_NUMBER6,
                "Invalid result type returned, expected JSONObject, was " + result.getResultType());
        } else {
            readOptionalTimestamp(result.getJSONObject(), object);
        }
    }

    protected int computeFolderOwnerID(DataObject o) {
        int id = o.getParentFolderOwnerID();
        return (id == 0) ? o.getSession().getUserIdentifier() : id;
    }

    protected int checkAndGetRequestedFieldLength(BitSet requestedFields) {
        int len = requestedFields.length();
        if (len > _contentType.getFields().length)
			throw new USMIllegalArgumentException(USMContentTypesUtilErrorCodes.INVALID_FIELDS_REQUESTED,
					"Invalid fields requested");
        return len;
    }

	protected void updateDataObjectFromJSONResult(DataObject destination, BitSet requestedFields, JSONResult result)
			throws OXCommunicationException, InternalUSMException {
        int len = checkAndGetRequestedFieldLength(requestedFields);
        checkResult(result, JSONResultType.JSONObject);
        JSONObject jObject = result.getJSONObject();

        JSONObject obj;
        try {
            obj = jObject.getJSONObject(CommonConstants.RESULT_DATA);
        } catch (JSONException e1) {
			throw generateException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER5, destination,
					"update object: data not present", e1);
        }
        readOptionalTimestamp(jObject, destination);
        ContentTypeField[] fields = _contentType.getFields();
        for (int i = 0; i < len; i++) {
            if (requestedFields.get(i))
                updateDataObjectFieldFromJSONObject(obj, fields[i], destination);
        }
    }

    protected void readOptionalTimestamp(JSONObject object, DataObject destination) throws OXCommunicationException {
        if (object.has(CommonConstants.RESULT_TIMESTAMP)) {
            try {
                long timestamp = object.getLong(CommonConstants.RESULT_TIMESTAMP);
                destination.setTimestamp(timestamp);
            } catch (JSONException e1) {
				throw generateException(USMContentTypesUtilErrorCodes.INVALID_TIMESTAMP_NUMBER1, destination,
						"update object: invalid timestamp", e1);
            }
        } else {
            // TODO Check if this is the preferred way to handle mail updates
            destination.setTimestamp(System.currentTimeMillis());
        }
    }

	protected void updateDataObjectFromJSONArray(DataObject destination, BitSet requestedFields, JSONArray array)
			throws InternalUSMException {
        int len = checkAndGetRequestedFieldLength(requestedFields);
        int j = 0;
        ContentTypeField[] fields = destination.getContentType().getFields();
        for (int i = 0; i < len; i++) {
            if (requestedFields.get(i)) {
                updateDataObjectFieldFromJSONArray(array, j, fields[i], destination);
                j++;
            }
        }
    }

    /**
	 *
     * @param source
     * @param field
     * @param destination
     * @throws InternalUSMException
     * @throws JSONException
     */
	protected void updateDataObjectFieldFromJSONObject(JSONObject source, ContentTypeField field, DataObject destination)
			throws InternalUSMException {
        String fieldName = field.getFieldName();
        DataType<?> fieldType = field.getFieldType();
        try {
			destination.setFieldContent(fieldName,
					fieldType.extractFromJSONObject(destination.getSession(), source, fieldName));
        } catch (JSONException e) {
			throwInternalUSMException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER5, "reading", destination,
					fieldName, fieldType, e);
        }
    }

    /**
	 *
     * @param source
     * @param index
     * @param field
     * @param destination
     * @throws InternalUSMException
     */
	protected void updateDataObjectFieldFromJSONArray(JSONArray source, int index, ContentTypeField field,
			DataObject destination) throws InternalUSMException {
        DataType<?> fieldType = field.getFieldType();
        String fieldName = field.getFieldName();
        try {
			destination.setFieldContent(fieldName,
					fieldType.extractFromJSONArray(destination.getSession(), source, index));
        } catch (JSONException e) {
			throwInternalUSMException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER4, "reading", destination,
					fieldName, fieldType, e);
        }
    }

	protected void throwInternalUSMException(int errorCode, String action, DataObject destination, String fieldName,
			DataType<?> fieldType, JSONException e) throws InternalUSMException {
		String message = "Error while " + action + ' ' + fieldName + " of type " + fieldType.getName() + " in "
				+ destination.getContentType().getID();
        // _journal.error(destination.getSession() + " " + message, e);
        throw new InternalUSMException(errorCode, message, e);
    }

	protected JSONObject createRequestBody(DataObject object, boolean isOnlyModifiedFields,
			boolean isIdshouldNotBePresent) throws InternalUSMException {
        JSONObject requestBody = new JSONObject();
        ContentTypeField[] fields = object.getContentType().getFields();
        for (int i = 0; i < fields.length; i++) {
            addFieldToRequestBody(object, isOnlyModifiedFields, isIdshouldNotBePresent, requestBody, fields[i], i);
        }
        return requestBody;
    }

	protected void addFieldToRequestBody(DataObject object, boolean isOnlyModifiedFields,
			boolean isIdshouldNotBePresent, JSONObject requestBody, ContentTypeField field, int fieldIndex)
			throws InternalUSMException {

        DataType<?> fieldType = field.getFieldType();
        String fieldName = field.getFieldName();
        // If explicitly forbidden, do not send ID of object
        if (isIdshouldNotBePresent && field.isIDField())
            return;
        // If only modified fields are requested, do not send unmodified fields
        if (isOnlyModifiedFields && !field.isRequiredForTransferToServer() && !object.isFieldModified(fieldIndex))
            return;
        // String fieldType = field.getFieldType();
        Object obj = object.getFieldContent(fieldName);
        if (!isOnlyModifiedFields) {
            // If all fields are requested, do not send those fields that are not present (i.e. null or empty array)
            if (obj == null)
                return;
            if ((obj instanceof Object[]) && ((Object[]) obj).length == 0)
                return;
        }
        try {
            fieldType.storeInJSONObject(object.getSession(), requestBody, fieldName, obj);
        } catch (JSONException e) {
			throwInternalUSMException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER3, "writing", object,
					fieldName, fieldType, e);
        }
    }

    protected void checkForNotDeletedObjects(DataObject object, JSONResult result) throws USMException {
        // An array with object IDs of folders which were modified after the specified timestamp and were therefore not deleted
        JSONArray notDeletedArray = extractArrayResult(result);
        if (notDeletedArray != null && notDeletedArray.length() > 0) {
            int len = notDeletedArray.length();
            for (int i = 0; i < len; i++) {
                if (LOG.isDebugEnabled()) {
                    try {
						LOG.debug(object.getSession() + " " + notDeletedArray.getString(i)
								+ " not deleted, modified after " + object.getTimestamp());
                    } catch (JSONException e) {
						LOG.error(object.getSession() + " Can't read ID of not deleted object " + object.getID()
								+ " from JSONResult", e);
                    }
                }
            }
			throw new ConflictingChangeException(USMContentTypesUtilErrorCodes.DELETE_ERROR, "Deletion failed",
					result.getJSONObject());
        }
    }

	protected DataObject createDataObject(JSONArray columnsArray, Folder folder, BitSet requestedFields, int idIndex)
			throws InternalUSMException, OXCommunicationException {
        String objId;
        try {
            objId = columnsArray.getString(idIndex);
        } catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER3, folder,
					"read/update folder: no id for new " + _contentType.getID() + " in folder " + folder.getID(), e);
        }
        DataObject dataObj = _contentType.newDataObject(folder.getSession());
        dataObj.setID(objId);
        updateDataObjectFromJSONArray(dataObj, requestedFields, columnsArray);
        return dataObj;
    }

    /**
	 * Returns array of DataObjects as returned in the JSON Result.
	 * Updates and deleted objects are handled separately.
     * @param folder
     * @param requestedFields
     * @param result
     * @return
     * @throws InternalUSMException
     * @throws OXCommunicationException
     */
	protected DataObject[] getListOfResults(Folder folder, BitSet requestedFields, JSONResult result)
			throws OXCommunicationException, InternalUSMException {
        JSONArray jsonar = null;
        long timestamp = 0L;
        checkResult(result, JSONResultType.JSONObject);
        JSONObject jsonObject = result.getJSONObject();
        try {
            jsonar = jsonObject.getJSONArray(CommonConstants.RESULT_DATA);
        } catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER4, folder,
					"getListOfResults: data array not present", e);
        }
        int length = jsonar.length();
        if (length == 0)
            return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
        try {
            if (jsonObject.has(CommonConstants.RESULT_TIMESTAMP))
                timestamp = jsonObject.getLong(CommonConstants.RESULT_TIMESTAMP);
        } catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.INVALID_TIMESTAMP_NUMBER2, folder,
					"getListOfResults: invalid timestamp", e);
        }
        DataObject[] dataObjects = new DataObject[length];
        for (int j = 0; j < length; j++) {
            // get array which contains the information specified by the corresponding identifiers in the columns parameter
            try {
                JSONArray columnsArray = jsonar.getJSONArray(j);
                DataObject newDataObject = createDataObject(columnsArray, folder, requestedFields, 0);
                dataObjects[j] = newDataObject;
                newDataObject.setTimestamp(timestamp);
            } catch (JSONException je) {
				throw generateException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER1, folder,
						"getListOfResults: error while reading index " + j + " of folder " + folder.getID(), je);
            }
        }
        return dataObjects;
    }

    /**
	 * Returns array of DataObjects as returned in the JSON Result.
	 * Updates and deleted objects are handled separately.
     * @param folder
     * @param requestedFields
     * @param result
     * @return
     * @throws InternalUSMException
     * @throws OXCommunicationException
     */
	protected DataObject[] getListOfUpdates(Folder folder, BitSet requestedFields, JSONResult result)
			throws OXCommunicationException, InternalUSMException {
        JSONArray jsonar = extractArrayResult(result);
        long timestamp;
        try {
            timestamp = result.getJSONObject().getLong(CommonConstants.RESULT_TIMESTAMP);
        } catch (JSONException e) {
            if (LOG.isDebugEnabled())
                LOG.debug(folder.getSession() + " Folder updates: could not read timestamp from result");
            timestamp = 0;
        }
        List<DataObject> dataObjects = new ArrayList<DataObject>();
        int length = jsonar.length();
        for (int j = 0; j < length; j++) {
            try {
                Object o = jsonar.get(j);
                // case updated object
                if (o instanceof JSONArray) {
                    // new object or modified object
                    // get array which contain the information specified by the corresponding identifiers in the columns parameter
                    JSONArray columnsArray = jsonar.getJSONArray(j);
                    DataObject newDataObject = createDataObject(columnsArray, folder, requestedFields, 0);
                    dataObjects.add(newDataObject);
                    if (timestamp > 0)
                        newDataObject.setTimestamp(timestamp);
                } else if (o instanceof String) {
                    // deleted object
                    String deletedObjectId = jsonar.getString(j);
                    DataObject dataObj = _contentType.newDataObject(folder.getSession());
                    dataObj.setID(deletedObjectId);
                    if (timestamp > 0)
                        dataObj.setTimestamp(timestamp);
                    dataObj.setChangeState(ChangeState.DELETED);
                    dataObjects.add(dataObj);
                } else {
					throw generateException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER2, folder,
							"Folder updates contains elements other than JSONArray or String at index " + j
									+ " for folder " + folder.getID(), null);
                }
            } catch (JSONException je) {
				throw generateException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER3, folder,
						"Folder updates: error while parsing index " + j + " for folder " + folder.getID(), je);
            }
        }
        return dataObjects.toArray(new DataObject[dataObjects.size()]);
    }

    protected void checkTypeToWrite(DataObject object, String contentTypeID) {
        if (!object.getContentType().getID().equals(contentTypeID))
			throw new USMIllegalArgumentException(USMContentTypesUtilErrorCodes.INVALID_CONTENT_TYPE_TO_WRITE,
                "DataObject is no " + contentTypeID);
    }

    protected JSONArray extractArrayResult(JSONResult result) throws OXCommunicationException, InternalUSMException {
        checkResult(result, JSONResultType.JSONObject);
        try {
            return result.getJSONObject().getJSONArray(CommonConstants.RESULT_DATA);
        } catch (JSONException e) {
            LOG.error("OX server did not send data array: " + result.toString());
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER3,
					"OX server did not send data array", e, result.toString());
        }
    }

	protected void checkResult(JSONResult result, JSONResultType expected) throws OXCommunicationException,
			InternalUSMException {
        if (result.getResultType() != expected) {
            switch (result.getResultType()) {
            case Error:
                checkConflictingChange(result.getJSONObject());
					throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER6,
							result.getJSONObject());
            case JSONArray:
					throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER7,
							"OX server sent array, expected " + expected, result.getJSONArray().toString());
            case JSONObject:
					throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER8,
							"OX server sent object, expected " + expected, result.getJSONObject().toString());
            case String:
					throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER9,
							"OX server sent unknown response, expected " + expected, result.getString());
            }
			throw new InternalUSMException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER10,
                "Unhandled JSONResultType " + result.getResultType());
        } else {
            checkIMAPDownWarning(result);
        }
    }

    protected void checkIMAPDownWarning(JSONResult result) throws TemporaryDownOrBusyException {
        if (result.getResultType() == JSONResultType.JSONObject && result.getJSONObject().has("code")) {
            try {
                JSONObject errorObject = result.getJSONObject();
                int category = errorObject.optInt(CommonConstants.CATEGORY);
                String message = errorObject.getString(CommonConstants.ERROR);
                String code = errorObject.getString(CommonConstants.CODE);
                JSONArray errorParams = errorObject.getJSONArray(ERROR_PARAMS);
                JSONObject errorDetails = new JSONObject();
                errorDetails.put(CommonConstants.CATEGORY, category);
                errorDetails.put(CommonConstants.ERROR, message);
                errorDetails.put(CommonConstants.CODE, code);
                errorDetails.put(ERROR_PARAMS, errorParams);
                if (OXErrorConstants.MSG_1016.equals(code) || OXErrorConstants.IMAP_1016.equals(code))
					throw new TemporaryDownOrBusyException(USMContentTypesUtilErrorCodes.IMAP_DOWN, message,
							errorDetails);
            } catch (JSONException ignored) {
                  //should not happen - thrown on object creation
            }
        }
    }

	protected void checkConflictingChange(JSONObject errorObject) throws ConflictingChangeException,
			InternalUSMException {
        try {
            int category = errorObject.optInt(CommonConstants.CATEGORY);
            String message = errorObject.getString(CommonConstants.ERROR);
            String code = errorObject.getString(CommonConstants.CODE);
			if (category == OXErrorConstants.CATEGORY_CONFLICTING_CHANGE
					|| (category == OXErrorConstants.CATEGORY_PERMISSION && OXErrorConstants.CON_0119.equals(code))
					|| (category == OXErrorConstants.CATEGORY_TRY_AGAIN && OXErrorConstants.CON_0240.equals(code))) {
				throw new ConflictingChangeException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER11, message,
						errorObject);
            }
        } catch (JSONException e) {
			throw new InternalUSMException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER12,
                "OX server sent an invalid  error object " + errorObject);
        }
    }

	protected JSONArray checkInvalidCharacters(JSONObject errorObject) throws InternalUSMException {
        try {
            int category = errorObject.optInt(CommonConstants.CATEGORY);
            String message = errorObject.optString(CommonConstants.ERROR);
            String code = errorObject.optString(CommonConstants.CODE);
            if (category == OXErrorConstants.CATEGORY_WRONG_INPUT && OXErrorConstants.APP_0071.equals(code)) {
                LOG.debug(message);
                JSONArray errorParams = errorObject.getJSONArray(ERROR_PARAMS);
                if (errorParams != null) {
                    LOG.debug(errorParams);
                }
                return errorParams;
            }
        } catch (JSONException e) {
            throw new InternalUSMException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER12,
                "OX server sent an invalid  error object " + errorObject);
        }
        return null;
	}
	
	
    @Override
    public void readDataObject(DataObject object, BitSet requestedFields) throws USMException {
        readDataObject(object, requestedFields, new HashMap<String, String>());
    }

	protected void readDataObject(DataObject object, BitSet requestedFields, Map<String, String> parameters)
			throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Read " + _contentType.getID() + " with id " + object.getID());
        checkTypeToWrite(object, _contentType.getID());
        parameters.put(CommonConstants.ID, object.getID());
        if (_contentType.canBeFolderElementsContentType())
            parameters.put(CommonConstants.FOLDER, object.getParentFolderID());
		JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), CommonConstants.ACTION_GET, object.getSession(),
				parameters);
        checkResult(result, JSONResultType.JSONObject);
        updateDataObjectFromJSONResult(object, requestedFields, result);
    }

    @Override
    public DataObject[] readFolderContent(Folder folder) throws USMException {
        return readFolderContent(folder, Integer.MAX_VALUE, null).getObjects();
    }

    @Override
    public OXFolderContent readFolderContent(Folder folder, int limit, DataObjectFilter filter) throws USMException {
        return readOptimizedFolderContent(folder, limit, filter);
    }

    @Override
    public DataObject[] readFolderContent(Folder folder, BitSet requestedFields) throws USMException {
        return readCompleteFolderContent(folder, requestedFields);
    }

    protected DataObject[] readCompleteFolderContent(Folder folder, BitSet requestedFields) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(folder.getSession() + " Read all " + _contentType.getID() + " in folder " + folder.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.FOLDER, folder.getID());
        parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
        addObjectsLimitParameters(folder.getSession(), parameters);
        return performActionAndCheckRights(CommonConstants.ACTION_ALL, folder, requestedFields, parameters);
    }

	protected DataObject[] performActionAndCheckRights(String action, Folder folder, BitSet requestedFields,
			Map<String, String> parameters) throws AuthenticationFailedException, InternalUSMException, USMException,
			OXCommunicationException {
        if (CHECK_ACCESS_RIGHTS_BEFORE_FOLDER_READ && !hasFolderReadRights(folder))
            return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
        try {
            JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), action, folder.getSession(), parameters);
            return getListOfResults(folder, requestedFields, result);
        } catch (OXCommunicationException e) {
            // catch the OXCommunication exception and than get the access rights
            if (hasFolderReadRights(folder))
                throw e;
            return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
        }
    }

    protected boolean hasFolderReadRights(Folder folder) throws USMException {
        BitSet requestedFields = new BitSet();
        int fieldIndex = folder.getFieldIndex(OWN_RIGHTS);
        requestedFields.set(fieldIndex);
        folder.getContentType().getTransferHandler().readFolder(folder, requestedFields);
        int own_rights = ((Number) folder.getFieldContent(fieldIndex)).intValue();
        return (own_rights & 0x3F80) != 0;
    }

    @Override
    public void writeDeletedDataObject(DataObject object) throws USMException {
        writeExtendedDeleteDataObject(object, CommonConstants.FOLDER, object.getParentFolderID());
    }

    protected void writeExtendedDeleteDataObject(DataObject object, String... extraKeyValuePairs) throws USMException {
        if ((extraKeyValuePairs.length & 1) != 0)
			throw new USMIllegalArgumentException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER4,
                "Extra key identifier without following field name");
        if (LOG.isDebugEnabled())
			LOG.debug(object.getSession() + " Delete " + _contentType.getID() + " with id " + object.getID()
					+ Arrays.toString(extraKeyValuePairs));
        checkTypeToWrite(object, _contentType.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.TIMESTAMP, String.valueOf(object.getTimestamp()));
        JSONObject jObj = new JSONObject();
        try {
            jObj.put(CommonConstants.ID, object.getID());
            for (int i = 0; i < extraKeyValuePairs.length; i += 2) {
                jObj.put(extraKeyValuePairs[i], extraKeyValuePairs[i + 1]);
            }
        } catch (JSONException e) {
			throw new USMIllegalArgumentException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER5,
                "Duplicate or null entry in provided key/value pairs");
        }
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_DELETE, object.getSession(),
				parameters, jObj);
        checkForNotDeletedObjects(object, result);
        readOptionalTimestamp(result.getJSONObject(), object);
    }

    @Override
    public void writeNewDataObject(DataObject object) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " New " + _contentType.getID());
        checkTypeToWrite(object, _contentType.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        JSONObject requestBody = createRequestBody(object, false, true);
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_NEW, object.getSession(),
				parameters, requestBody);
        checkResult(result, JSONResultType.JSONObject);
        JSONObject o = result.getJSONObject();
        readOptionalTimestamp(o, object);
        try {
            object.setID(o.getJSONObject(CommonConstants.RESULT_DATA).getString(CommonConstants.RESULT_ID));
        } catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER4, object,
					"Missing result ID on new " + _contentType.getID(), e);
        }
    }

    @Override
    public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
        writeStandardUpdateDataObject(object, timestamp, false);
    }

	protected void writeStandardUpdateDataObject(DataObject object, long timestamp, boolean isFolder)
			throws InternalUSMException, AuthenticationFailedException, OXCommunicationException,
			FolderNotFoundException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Updating " + _contentType.getID());
        checkTypeToWrite(object, _contentType.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.ID, object.getID());
        String folderTree = object.getSession().getPersistentField(FolderContentType.FOLDER_TREE);
        if (!isFolder) {
            parameters.put(CommonConstants.FOLDER, object.getOriginalParentFolderID());
        } else if (folderTree != null  && folderTree.length() > 0) {
            parameters.put(CommonConstants.TREE, folderTree);
        }
        parameters.put(CommonConstants.TIMESTAMP, String.valueOf(timestamp));
        JSONObject requestBody = createRequestBody(object, true, false);
        // printTimeValues("Change", requestBody);
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_UPDATE, object.getSession(),
				parameters, requestBody);
        if (isFolder) {
            if (result.getResultType() == JSONResultType.Error) {
                if (isFolderNotExistError(result.getJSONObject()))
					throw new FolderNotFoundException(USMContentTypesUtilErrorCodes.FOLDER_NOT_FOUND,
                        "Folder not found: " + object.getUUID() + '(' + object.getID() + ')', result.getJSONObject());
            }
            checkResult(result, JSONResultType.JSONObject);
            JSONObject o = result.getJSONObject();
            if (o.has(CommonConstants.RESULT_DATA)) {
                try {
                    object.setID(o.getString(CommonConstants.RESULT_DATA));
                } catch (JSONException e) {
					throw generateException(USMContentTypesUtilErrorCodes.DATA_INVALID, object,
							"update object: invalid folder id", e);
                }
            }
            readOptionalTimestamp(o, object);
        } else {
            checkResult(result, JSONResultType.JSONObject);
            // XXX Shouldn't we use the timestamp returned from the OX server ?
            object.setTimestamp(timestamp);
        }
    }

    private boolean isFolderNotExistError(JSONObject errorObject) throws InternalUSMException {
        try {
            int category = errorObject.optInt(CommonConstants.CATEGORY);
            String code = errorObject.getString(CommonConstants.CODE);
            if (category == OXErrorConstants.CATEGORY_CODE_ERROR && OXErrorConstants.IMAP_1002.equals(code)) {
                return true;
            }
        } catch (JSONException e) {
			throw new InternalUSMException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER13,
                "OX server sent an invalid  error object " + errorObject);
        }
        return false;
    }

    // For debugging purposes only
    // protected void printTimeValues(String action, JSONObject requestBody) {
    // printTimeValue(action, requestBody, "start_date");
    // printTimeValue(action, requestBody, "end_date");
    // }

    // private void printTimeValue(String action, JSONObject requestBody, String key) {
    // Long st = requestBody.optLong(key);
    // if (st != null) {
    // SimpleDateFormat format = new SimpleDateFormat("EEEE, dd.MM.yyyy HH:mm:ss.SSS z '('Z')'");
    // double d = ((double) st) / 86400000.0;
    // System.out.println(action + ": " + st + "(" + d + ")" + " -> " + format.format(new Date(st)));
    // }
    // }

	@Override
    public DataObject[] readUpdatedFolderContent(Folder folder, BitSet requestedFields, long timestamp)
			throws USMException {
		return readStandardUpdatedFolderContent(folder, timestamp, requestedFields, CommonConstants.ACTION_UPDATES,
				CommonConstants.FOLDER);
    }

	protected DataObject[] readStandardUpdatedFolderContent(Folder folder, long timestamp, BitSet requestedFields,
			String action, String folderKey) throws USMException {
		return readStandardUpdatedFolderContent(folder, timestamp, requestedFields, action, folderKey, null, null,
				null, null);
    }

	protected DataObject[] readStandardUpdatedFolderContent(Folder folder, long timestamp, BitSet requestedFields,
			String action, String folderKey, String sort, String order, String start, String end) throws USMException {
		return readStandardUpdatedFolderContent(folder, timestamp, requestedFields, action, folderKey, sort, order,
				start, end, null);
    }

	protected DataObject[] readStandardUpdatedFolderContent(Folder folder, long timestamp, BitSet requestedFields,
			String action, String folderKey, String sort, String order, String start, String end, String folderTree)
			throws USMException {
        if (LOG.isDebugEnabled())
			LOG.debug(folder.getSession() + " Read updated " + _contentType.getID() + " after " + timestamp
					+ " in folder " + folder.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(folderKey, folder.getID()); // object ID of the folder
        parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
        parameters.put(CommonConstants.TIMESTAMP, String.valueOf(timestamp));
        if (sort != null) {
            parameters.put(CommonConstants.SORT, sort);
            if (order != null)
                parameters.put(CommonConstants.ORDER, order);
        }
        if (start != null) {
            parameters.put(CommonConstants.START, start);
            if (end != null)
                parameters.put(CommonConstants.END, end);
        }
        if (folderTree != null) {
            parameters.put(CommonConstants.TREE, folderTree);
        }
        addObjectsLimitParameters(folder.getSession(), parameters);
        addShowPrivateParameter(parameters, folder.getSession());
        JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), action, folder.getSession(), parameters);
        return getListOfUpdates(folder, requestedFields, result);
    }

    protected OXCommunicationException generateException(int errorCode, DataObject obj, String message, Throwable cause) {
        // _journal.error(obj.getSession() + " " + errorCode + ":" + message, cause);
        return new OXCommunicationException(errorCode, message, cause);
    }

    protected void resetRecurrenceTypeBasedFields(DataObject[] results) {
		/**recurrentType
		 * 0 	 none (single event)
		   1 	daily
		   2 	weekly
		   3 	monthly
		   4 	yearly
		 *
         */
        for (DataObject data : results) {
            int recurrenceType = 0;
            if (data.getFieldContent("recurrence_type") != null)
                recurrenceType = ((Number) data.getFieldContent("recurrence_type")).intValue();
            // single event
            if (recurrenceType == 0) {
                data.setFieldContent("until", null);
                data.setFieldContent("interval", null);
                data.setFieldContent("occurrences", null);
            }

            if (recurrenceType <= 1) {
                data.setFieldContent("days", null);
            }

            if (recurrenceType <= 2) {
                data.setFieldContent("day_in_month", null);
            }
            if (recurrenceType != 4) {
                // Do not explicitly set interval to 1 for yearly series since other intervals are now also allowed
                data.setFieldContent("month", null);
            }
        }
    }

	protected void setAppOrTaskConfirmations(DataObject object) throws AuthenticationFailedException,
			OXCommunicationException {
        if (!object.isFieldModified(CommonCalendarTasksFieldNames.USERS)) 
            return;
            
            int folderOwner = computeFolderOwnerID(object);
            Object[] users = (Object[]) object.getFieldContent(CommonCalendarTasksFieldNames.USERS);
            Object[] usersOld = (Object[]) object.getOriginalFieldContent(CommonCalendarTasksFieldNames.USERS);
            for (Object user : users) {
                UserParticipantObject participant = (UserParticipantObject) user;
                int confirmation = participant.getConfirmation();
                int id = participant.getId();
                if (/* confirmation != 0 && */id == folderOwner) {
                    for (Object userOld : usersOld) {
                        int idOld = ((UserParticipantObject) userOld).getId();
                        if (id == idOld) {
                            int confirmationOld = ((UserParticipantObject) userOld).getConfirmation();
                            if (confirmationOld != confirmation) {
                                confirm(object, confirmation);
                                object.commitChanges();
                            }
                            break;
                        }
                    }
                }
            }
 
    }

    protected abstract String getOXAjaxAccessPath();

    @Override
    public long createNewAttachment(DataObject object, PIMAttachment attachment) throws USMException {
        if (!_contentType.supportsPIMAttachments())
            return 0L;
        return _pimAttachmentsTransferHandler.createNewAttachment(object, attachment);
    }

    @Override
    public PIMAttachments getAllAttachments(DataObject object) throws USMException {
        if (!_contentType.supportsPIMAttachments())
            return null;
        return _pimAttachmentsTransferHandler.getAllAttachments(object);
    }

    @Override
    public long deleteAttachments(DataObject object, PIMAttachment... attachmentsToDelete) throws USMException {
        if (!_contentType.supportsPIMAttachments())
            return 0L;
        return _pimAttachmentsTransferHandler.deleteAttachments(object, attachmentsToDelete);
    }

    @Override
    public byte[] getAttachmentData(DataObject object, int attachmentId) throws USMException {
        if (!_contentType.supportsPIMAttachments())
            return null;
        return _pimAttachmentsTransferHandler.getAttachmentData(object, attachmentId);
    }

    @Override
    public ResourceInputStream getAttachmentDataStream(DataObject object, int attachmentId, long offset, long length) throws USMException {
        if (!_contentType.supportsPIMAttachments())
            return null;
        return _pimAttachmentsTransferHandler.getAttachmentDataStream(object, attachmentId, offset, length);
    }

    protected void addObjectsLimitParameters(Session session, Map<String, String> parameters) {
        int limit = session.getObjectsLimit();
        if (limit > 0)
            addObjectsLimitParameters(limit, _LAST_MODIFIED_COLUMN, parameters); // sort by last modified
    }

    protected void addObjectsLimitParameters(int limit, String defaultSortColumn, Map<String, String> parameters) {
        parameters.put(LEFT_HAND_LIMIT, "0");
        parameters.put(RIGHT_HAND_LIMIT, String.valueOf(limit));
        if (!parameters.containsKey(CommonConstants.SORT)) {
            parameters.put(CommonConstants.SORT, defaultSortColumn);
            parameters.put(CommonConstants.ORDER, "desc");
        }
    }

    protected OXFolderContent readOptimizedFolderContent(Folder folder, int limit, DataObjectFilter filter) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(folder.getSession() + " Read all " + _contentType.getID() + " in folder " + folder.getID());
        if (CHECK_ACCESS_RIGHTS_BEFORE_FOLDER_READ && !hasFolderReadRights(folder))
            return _EMPTY_FOLDER_CONTENT;
        try {
            BitSet requestFields = new BitSet();
            requestFields.set(5);
            requestFields.or(folder.getSession().getFieldFilter(folder.getElementsContentType()));
            DataObjectSet cachedObjects = getCachedObjects(folder);
            OXObjectIDList currentTimestamps = getCurrentTimestampsFromOX(folder);

            // Compute objects to read from OX
            List<String> changedObjects = new ArrayList<String>();
            List<String> newObjects = new ArrayList<String>();
            for (int i = 0; i < currentTimestamps.size(); i++) {
                String id = currentTimestamps.getID(i);
                DataObject cachedObject = cachedObjects.get(id);
                if (cachedObject == null) {
                    newObjects.add(id);
                } else {
                    Object o = cachedObject.getFieldContent(CommonConstants.LAST_MODIFIED);
                    if (!(o instanceof Number) || ((Number) o).longValue() != currentTimestamps.getLastModified(i)) {
                        changedObjects.add(id);
                    }
                }
            }
            DataObjectSet currentServerObjects = new DataObjectSet();
            readObjectsFromOX(folder, changedObjects, requestFields, currentServerObjects, changedObjects.size(), null);
            boolean complete = readObjectsFromOX(folder, newObjects, requestFields, currentServerObjects, limit, filter);
            DataObject[] result = mergeObjectsFromOXWithCachedObjects(currentTimestamps, cachedObjects, currentServerObjects);
            updateTimestampsAndOtherData(currentTimestamps, result);
            return new OXFolderContent(complete, result);
        } catch (OXCommunicationException e) {
            // catch the OXCommunication exception and then get the access rights
            if (hasFolderReadRights(folder))
                throw e;
            return _EMPTY_FOLDER_CONTENT;
        }
    }

    protected DataObjectSet getCachedObjects(Folder folder) throws USMException {
        return new DataObjectSet(folder.getSession().getCachedFolderElements(folder.getID(), folder.getElementsContentType()));
    }

    protected OXObjectIDList getCurrentTimestampsFromOX(Folder folder) throws AuthenticationFailedException, OXCommunicationException, InternalUSMException {
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.FOLDER, folder.getID());
        parameters.put(CommonConstants.COLUMNS, _COLUMNS_ID_AND_LAST_MODIFIED);
        addExtraTimestampListParameters(parameters, folder.getSession());
        Session session = folder.getSession();
        addTimeLimitParameters(parameters, session);
        addObjectsLimitParameters(folder.getSession(), parameters);
        long defaultTimestamp = System.currentTimeMillis();
        JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), CommonConstants.ACTION_ALL, folder.getSession(), parameters);
        checkResult(result, JSONResultType.JSONObject);
        JSONObject jsonObject = result.getJSONObject();
        try {
            return new OXObjectIDList(defaultTimestamp, jsonObject);
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.INVALID_DATA_ON_READING_ALL_IDS,
                folder,
                "getCurrentTimestampsFromOX: Error reading result on action=all with id,last_modified",
                e);
        }
    }

    protected void addTimeLimitParameters(Map<String, String> parameters, Session session) {
        parameters.put(CommonConstants.START, String.valueOf(session.getStartDate()));
        parameters.put(CommonConstants.END, String.valueOf(session.getEndDate()));
    }

    protected void addExtraTimestampListParameters(Map<String, String> parameters, Session session) {
    }
    
    protected void addShowPrivateParameter(Map<String, String> parameters, Session session) {
    }

    protected boolean readObjectsFromOX(Folder folder, List<String> objectsToRead, BitSet requestedFields, DataObjectSet result, int limit, DataObjectFilter filter) throws UnsupportedContentOperationException, AuthenticationFailedException, OXCommunicationException, InternalUSMException {
        if (objectsToRead == null || objectsToRead.isEmpty())
            return true;
        int read = 0;
        for (int i = 0; i < objectsToRead.size();) {
            int upperBounds = Math.min(i + _MAX_OBJECTS_PER_LIST, objectsToRead.size());
            List<String> subList = objectsToRead.subList(i, upperBounds);
            DataObject[] someObjects = readSomeObjectsFromOX(folder, subList, requestedFields);
            for(DataObject o : someObjects) {
                if(filter == null || filter.accept(o)) {
                    result.add(o);
                    read++;
                }
            }
            if(read >= limit && upperBounds < objectsToRead.size())
                return false;
            i = upperBounds;
        }
        return true;
    }

    protected DataObject[] readSomeObjectsFromOX(Folder folder, List<String> objectsToRead, BitSet requestedFields) throws UnsupportedContentOperationException, AuthenticationFailedException, OXCommunicationException, InternalUSMException {
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
        JSONArray ids = new JSONArray();
        for (String id : objectsToRead) {
            JSONObject element = new JSONObject();
            try {
                element.put(CommonConstants.FOLDER, folder.getID());
                element.put(CommonConstants.ID, id);
            } catch (JSONException ignored) {
                // Ignored because it can not happen
            }
            ids.put(element);
        }
        addShowPrivateParameter(parameters, folder.getSession());
        JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_LIST, folder.getSession(), parameters, ids);
        return getListOfResults(folder, requestedFields, result);
    }

    protected DataObject[] mergeObjectsFromOXWithCachedObjects(OXObjectIDList currentTimestamps, DataObjectSet cachedObjects, DataObjectSet currentServerObjects) {
        for (int i = 0; i < currentTimestamps.size(); i++) {
            String id = currentTimestamps.getID(i);
            if (!currentServerObjects.contains(id)) {
                DataObject cachedObject = cachedObjects.get(id);
                if (cachedObject != null)
                    currentServerObjects.add(cachedObject);
            }
        }
        return currentServerObjects.toArray();
    }

    protected void updateTimestampsAndOtherData(OXObjectIDList currentTimestamps, DataObject[] result) {
        for (DataObject o : result) {
            o.setTimestamp(currentTimestamps.getTimestamp());
            finishDataObjectFromOptimizedRead(o);
        }
    }

    protected void finishDataObjectFromOptimizedRead(DataObject o) {
        //empty
    }
}
