/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.datatypes.tasks.calendar;

import java.io.Serializable;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Defines the User Participant Object used in the Tasks Module.
 * 
 * @author ldo
 */
public final class UserParticipantObject implements Serializable {

    private static final long serialVersionUID = 1L;

    public static final String CONFIRM_MESSAGE = "confirmmessage";

    public static final String CONFIRMATION = "confirmation"; /* 0=none, 1=accepted, 2=declined, 3=tentative */

    static final String DISPLAY_NAME = "display_name";

    static final String ID = "id";

    /** User ID */
    private final int _id;

    /** Displayable name of the participant */
    private final String _displayName;

    /** Valid values: 0-3 */
    private final int _confirmation;

    /** Confirm Message of the participant */
    private final String _confirmMessage;

    public UserParticipantObject(int id, String name, int confirmation, String message) {
        _id = id;
        _displayName = name;
        _confirmation = confirmation;
        _confirmMessage = message;
    }

    public UserParticipantObject(JSONObject jObject) throws JSONException {
        _id = jObject.getInt(ID);
        _displayName = jObject.has(DISPLAY_NAME) ? jObject.getString(DISPLAY_NAME) : "";
        _confirmation = jObject.hasAndNotNull(CONFIRMATION) ? jObject.getInt(CONFIRMATION) : 0;
        _confirmMessage = jObject.has(CONFIRM_MESSAGE) ? jObject.getString(CONFIRM_MESSAGE) : "";
    }

    public UserParticipantObject() {
        _id = 0;
        _displayName = "";
        _confirmation = 0;
        _confirmMessage = "";
    }

    public int getId() {
        return _id;
    }

    public String getDisplayName() {
        return _displayName;
    }

    public int getConfirmation() {
        return _confirmation;
    }

    public String getConfirmMessage() {
        return _confirmMessage;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + _id;
        result = prime * result + _displayName.hashCode();
        result = prime * result + _confirmation;
        result = prime * result + _confirmMessage.hashCode();
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        UserParticipantObject other = (UserParticipantObject) obj;
        if (_id != other.getId())
            return false;
        if (!_displayName.equals(other.getDisplayName()))
            return false;
        if (_confirmation != other.getConfirmation())
            return false;
        if (!_confirmMessage.equals(other.getConfirmMessage()))
            return false;
        return true;
    }

    @Override
    public String toString() {
        return '{' + ID + ':' + String.valueOf(_id) + ';' + DISPLAY_NAME + ':' + _displayName + ';' + CONFIRMATION + ':' + String.valueOf(_confirmation) + ';' + CONFIRM_MESSAGE + ':' + _confirmMessage + '}';
    }

    public JSONObject toJSONObject() throws JSONException {
        JSONObject result = new JSONObject();
        result.put(ID, _id);
        result.put(DISPLAY_NAME, _displayName);
        result.put(CONFIRMATION, _confirmation);
        if (!_confirmMessage.isEmpty()) {
            result.put(CONFIRM_MESSAGE, _confirmMessage);
        }
        return result;
    }
}
