/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.contenttypes.folder.impl;

import static com.openexchange.usm.api.datatypes.DataTypes.BOOLEAN;
import static com.openexchange.usm.api.datatypes.DataTypes.NUMBER;
import static com.openexchange.usm.api.datatypes.DataTypes.NUMBER_NOT_NULL;
import static com.openexchange.usm.api.datatypes.DataTypes.STRING;
import static com.openexchange.usm.api.datatypes.DataTypes.TIME;
import static com.openexchange.usm.api.datatypes.DataTypes.TIMESTAMP;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.FolderConstants;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;
import com.openexchange.usm.api.contenttypes.transferhandlers.FolderContentTypeTransferHandler;
import com.openexchange.usm.api.datatypes.OwnRightsDataType;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.contenttypes.util.AbstractContentType;
import com.openexchange.usm.datatypes.folder.FolderPermissionDataType;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.session.dataobject.FolderImpl;

/**
 * This class defines all relevant information for content type folder
 * 
 * @author ibr
 */
public class FolderContentTypeImpl extends AbstractContentType implements FolderContentType {

    private static final ContentTypeField[] FIELDS = {
        // Common folder data
        new ContentTypeField(1, CommonConstants.FIELD_ID, STRING, false, true, false, false),// Object ID
        new ContentTypeField(20, CommonConstants.FOLDER_ID, STRING, false, false, true, false), // Object ID of the parent folder.
        new ContentTypeField(2, CommonConstants.CREATED_BY, STRING), // User ID of the user who created this object.
        new ContentTypeField(3, CommonConstants.MODIFIED_BY, STRING), // User ID of the user who last modified this object.
        new ContentTypeField(4, CommonConstants.CREATION_DATE, TIME), // Date and time of creation.
        new ContentTypeField(5, CommonConstants.LAST_MODIFIED, TIME), // Date and time of the last modification.
        new ContentTypeField(6, FolderConstants.LAST_MODIFIED_UTC, TIMESTAMP), // Timestamp of the last modification.
        // Detailed folder data
        new ContentTypeField(300, FolderConstants.TITLE, STRING), // Name of this folder.
        new ContentTypeField(301, FolderConstants.MODULE, STRING), // Name of the module which implements this folder; e.g. "tasks",
                                                                   // "calendar", "contacts", "infostore", or "mail"
        new ContentTypeField(302, FolderConstants.TYPE, NUMBER), // Type of folder: 1 - private, 2 - public, 3 - shared, 5 - system folder
        new ContentTypeField(304, FolderConstants.SUBFOLDERS, BOOLEAN), // true if this folder has subfolders.
        new ContentTypeField(305, FolderConstants.OWN_RIGHTS, new OwnRightsDataType()), // Permissions which apply to the current user, as
                                                                                        // described either in Permission flags or in RFC
                                                                                        // 2086.
        new ContentTypeField(306, FolderConstants.PERMISSIONS, new FolderPermissionDataType(), true), // Each element is an object described
                                                                                                      // in (Folder)Permission object.
        new ContentTypeField(307, FolderConstants.SUMMARY, STRING), // Information about contained objects.
        new ContentTypeField(308, FolderConstants.STANDARD_FOLDER, BOOLEAN), // Indicates whether or not folder is marked as a default
                                                                             // folder (only OX folder)
        // XXX total can be reported as NULL ? Maybe also other fields ? Why ?
        new ContentTypeField(309, FolderConstants.TOTAL, NUMBER), // The number of objects in this Folder.
        new ContentTypeField(310, FolderConstants.NEW, NUMBER), // The number of new objects in this Folder.
        new ContentTypeField(311, FolderConstants.UNREAD, NUMBER), // The number of unread objects in this Folder.
        new ContentTypeField(312, FolderConstants.DELETED, NUMBER), // The number of deleted objects in this Folder.
        new ContentTypeField(313, FolderConstants.CAPABILITIES, NUMBER), // Bit mask containing information about mail folder capabilites,
                                                                         // as described in capabilities.
        new ContentTypeField(314, FolderConstants.SUBSCRIBED, BOOLEAN), // Indicates whether this folder should appear in folder tree or
                                                                        // not.
        new ContentTypeField(315, FolderConstants.SUBSCR_SUBFLDS, BOOLEAN), // Indicates whether subfolders should appear in folder tree or
                                                                            // not.
        new ContentTypeField(316, FolderConstants.STANDARD_FOLDER_TYPE, NUMBER_NOT_NULL), // Special code indicating standard folders with
                                                                                          // special meaning like "Trash"
    };

    private FolderContentTypeTransferHandler _transferHandler;

    @Override
    public ContentTypeField[] getFields() {
        return FIELDS;
    }

    @Override
    public String getID() {
        return DefaultContentTypes.FOLDER_ID;
    }

    @Override
    public Folder newDataObject(Session session) {
        return new FolderImpl(session, this);
    }

    @Override
    public FolderContentTypeTransferHandler getTransferHandler() {
        return _transferHandler;
    }

    public void setTransferHandler(FolderContentTypeTransferHandler transferHandler) {
        _transferHandler = transferHandler;
    }

    @Override
    public int hashCode() {
        return 1931;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!(obj instanceof FolderContentType))
            return false;
        return true;
    }

    @Override
    public int getMatchRating(DataObject client, DataObject server) {
        // TODO Add more heuristic to determine possible match ?
        // Currently Folders are assumed a match if their title matches, the match is more likely if also their parent folder id matches
        if (!DataObjectUtil.isFieldEqual(client, server, FolderConstants.TITLE))
            return -1;
        return DataObjectUtil.isEqual(client.getParentFolderID(), server.getParentFolderID()) ? 2 : 1;
    }

    @Override
    public int getCode() {
        return DefaultContentTypes.FOLDER_CODE;
    }

    @Override
    public boolean supportsPIMAttachments() {
        return false;
    }

    @Override
    public Object getObjectGroupOwner(DataObject object) {
        return object.getID();
    }

    @Override
    public boolean canBeFolderElementsContentType() {
        return false;
    }

    @Override
    public FolderContentType getFolderContentType() {
        return this;
    }
}
