/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.contenttypes.contacts.impl;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.imageio.ImageIO;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.contact.ContactConstants;
import com.openexchange.usm.api.contenttypes.contact.ContactContentType;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.ConflictingChangeException;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.PictureConversionException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.UnsupportedContentOperationException;
import com.openexchange.usm.api.ox.json.JSONResult;
import com.openexchange.usm.api.ox.json.JSONResultType;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.api.ox.json.OXResource;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.contenttypes.util.AbstractTransferHandler;
import com.openexchange.usm.contenttypes.util.OXErrorConstants;
import com.openexchange.usm.contenttypes.util.USMContentTypesUtilErrorCodes;
import com.openexchange.usm.datatypes.contacts.Image;
import com.openexchange.usm.datatypes.contacts.ImageDataType;
import com.openexchange.usm.util.TempFile;
import com.openexchange.usm.util.TempFileStorage;
import com.openexchange.usm.util.Toolkit;

/**
 * @author ibr
 */
public class ContactContentTypeTransferHandler extends AbstractTransferHandler {

    private static final boolean DECODE_IMAGE_URL = false;

    private static final String OR_SEARCH = "orSearch";

    private static final String EMAIL_AUTO_COMPLETE = "emailAutoComplete";

    // The field "image1" is a complex element that is represented by several fields in OX.
    // This String determines which extra columns have to be fetched on a folder read.
    private static final String IMAGE_COLUMNS = "597,606,601";

    private static final String ACTION_SEARCH = "search";

    private final BitSet _image1FieldSet;

    public ContactContentTypeTransferHandler(ContactContentType contentType, OXJSONAccess ajaxAccess) {
        super(contentType, ajaxAccess);
        _image1FieldSet = new BitSet();
        ContentTypeField[] fields = contentType.getFields();
        for (int i = 0; i < fields.length; i++) {
            if (fields[i].getFieldID() == ContactContentTypeImpl.IMAGE1_FIELD_ID) {
                _image1FieldSet.set(i);
                break;
            }
        }
    }

	@Override
	protected String getOXAjaxAccessPath() {
		return CommonConstants.CONTACT_PATH;
	}

	public Map<String, DataObject[]> resolveRecipients(List<String> toList, Session session, BitSet requestedFields)
			throws AuthenticationFailedException, OXCommunicationException, UnsupportedContentOperationException, InternalUSMException {
		Map<String, DataObject[]> result = new HashMap<String, DataObject[]>();
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));

		for (String to : toList) {
	        JSONObject requestBody = new JSONObject();
			try {
				requestBody.put(ContactConstants.LAST_NAME, to);
				requestBody.put(ContactConstants.FIRST_NAME, to);
				requestBody.put(ContactConstants.DISPLAY_NAME, to);
				requestBody.put(ContactConstants.EMAIL1, to);
				requestBody.put(ContactConstants.EMAIL2, to);
				requestBody.put(ContactConstants.EMAIL3, to);
				requestBody.put(ContactConstants.SECOND_NAME, to);
				requestBody.put(ContactConstants.NICKNAME, to);
				requestBody.put(EMAIL_AUTO_COMPLETE, Boolean.TRUE.toString());
				requestBody.put(OR_SEARCH, Boolean.TRUE.toString());
			} catch (JSONException e) {
				LOG.error(session + " Can not create request body for Resolving recipients", e);
			}

			JSONResult jResult = _ajaxAccess.doPut(getOXAjaxAccessPath(), ACTION_SEARCH, session, parameters,
					requestBody);
            if (jResult.getResultType() == JSONResultType.Error && isUnderMinimumSearchCharactersError(jResult.getJSONObject())) {
                result.put(to, new DataObject[0]);
            } else {
            //  fill List
                result.put(to, parseRecipientsResult(jResult, session, requestedFields));
            }
        }
        return result;
    }
    
    private boolean isUnderMinimumSearchCharactersError(JSONObject errorObject) {
        int category = errorObject.optInt(CommonConstants.CATEGORY);
        String code = errorObject.optString(CommonConstants.CODE);
        return (OXErrorConstants.CATEGORY_WRONG_INPUT == category && OXErrorConstants.CON_1000.equals(code));
    }

    private DataObject[] parseRecipientsResult(JSONResult jResult, Session session, BitSet requestedFields) throws OXCommunicationException, InternalUSMException {
        JSONArray jsonar;

        try {
            checkResult(jResult, JSONResultType.JSONObject);
            jsonar = jResult.getJSONObject().getJSONArray(CommonConstants.RESULT_DATA);
        } catch (JSONException e1) {
            return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
        }
        int length = jsonar.length();
        if (length == 0)
            return SyncResult.EMPTY_DATA_OBJECT_ARRAY;

        List<DataObject> dataObjects = new ArrayList<DataObject>();
        for (int j = 0; j < length; j++) {
            // get array which contains the information specified by the corresponding identifiers in the columns parameter
            try {
                JSONArray columnsArray = jsonar.getJSONArray(j);
                DataObject newDataObject = _contentType.newDataObject(session);
                updateDataObjectFromJSONArray(newDataObject, requestedFields, columnsArray);
                dataObjects.add(newDataObject);

                String emailAddress2 = (String) newDataObject.getFieldContent(ContactConstants.EMAIL2);
                if (emailAddress2 != null && !"".equals(emailAddress2)) {
                    DataObject objCopy = newDataObject.createCopy(true);
                    objCopy.setFieldContent(ContactConstants.EMAIL1, null);
                    objCopy.setFieldContent(ContactConstants.EMAIL3, null);
                    dataObjects.add(objCopy);
                }

                String emailAddress3 = (String) newDataObject.getFieldContent(ContactConstants.EMAIL3);
                if (emailAddress3 != null && !"".equals(emailAddress3)) {
                    DataObject objCopy = newDataObject.createCopy(true);
                    objCopy.setFieldContent(ContactConstants.EMAIL1, null);
                    objCopy.setFieldContent(ContactConstants.EMAIL2, null);
                    dataObjects.add(objCopy);
                }
                // remove additional email addresses from the original object
                newDataObject.setFieldContent(ContactConstants.EMAIL2, null);
                newDataObject.setFieldContent(ContactConstants.EMAIL3, null);

            } catch (JSONException je) {
                LOG.error(session + " Can not read the result data for resolving recipients", je);
            } catch (InternalUSMException e) {
                LOG.error(session + " Internal error resolving recipients", e);
            }
        }
        return dataObjects.toArray(new DataObject[dataObjects.size()]);

    }

    private String getImageUrl(DataObject dataObject) {
        try {
            Image image = (Image) dataObject.getFieldContent(ImageDataType.IMAGE1);
            if (image == null)
                return null;
            String url = image.getUrl();
            if (!Toolkit.isNullOrEmpty(url))
                return url;
            readDataObject(dataObject, _image1FieldSet);
            image = (Image) dataObject.getFieldContent(ImageDataType.IMAGE1);
            return (image == null) ? null : image.getUrl();
        } catch (USMException e) {
            return null;
        }
    }

    public byte[] getPictureData(DataObject dataObject, String pictureFormat, int maxSize) throws AuthenticationFailedException, OXCommunicationException, PictureConversionException {
        String url = getImageUrl(dataObject);
        if (Toolkit.isNullOrEmpty(url))
            return null;
        if (DECODE_IMAGE_URL) {
            try {
                url = URLDecoder.decode(url, "UTF-8");
            } catch (UnsupportedEncodingException e1) {
                return null;
            }
        }
        // String imageContentType = (String) dataObject.getFieldContent("image1_content_type");
        OXResource sourceData = _ajaxAccess.getResource(dataObject.getSession(), url, null);
        int sourceSize = sourceData.getData().length;
        if (sourceSize == 0)
            return null;
        if (sourceData.getContentType().equalsIgnoreCase("image/" + pictureFormat) && sourceSize <= maxSize)
            return sourceData.getData();
        // Conversion required because picture is in the wrong format or too big
        // The scale factor is just an estimate, since we have to scale to some target byte size in the output
        // The picture is either too big or in the wrong format, in either case we assume that if we scale to
        // about 90% of the limit divided by its current size (if the picture is bigger than that), our result
        // will be within the limit. We have a fall back algorithm, that halves the size up to 2 times if our
        // estimate should be too big. After that we return null (assuming that the provided limit is too small
        // for the image format).
        float sizeLimit = maxSize * 0.9f;
        float scaleFactor = (sourceSize > sizeLimit) ? sizeLimit / sourceSize : 1.0f;
        BufferedImage sourceImage;
        try {
            sourceImage = ImageIO.read(new ByteArrayInputStream(sourceData.getData()));
        } catch (IOException e) {
            throw new OXCommunicationException(
                USMContentTypesContactsErrorCodes.IMAGE_ERROR_NUMBER1,
                "Picture received from OX server could not be read into a BufferedImage",
                e);
        }
        if (sourceImage == null) {
            LOG.warn(dataObject.getSession() + " Cannot read picture for contact " + dataObject.getID() + ", no appropriate ImageReader found.");
            return null;
        }
        for (int i = 0; i < 6; i++) {
            int scaledWidth = Math.round(sourceImage.getWidth() * scaleFactor);
            int scaledHeight = Math.round(sourceImage.getHeight() * scaleFactor);
            BufferedImage scaledImage = new BufferedImage(scaledWidth, scaledHeight, BufferedImage.TYPE_INT_RGB);
            Graphics2D g = scaledImage.createGraphics();
            g.setColor(Color.WHITE);
            g.fillRect(0, 0, scaledWidth, scaledHeight);
            g.drawImage(sourceImage, 0, 0, scaledWidth, scaledHeight, null);
            g.dispose();
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            try {
                ImageIO.write(scaledImage, pictureFormat, os);
            } catch (IOException e) {
                throw new PictureConversionException(
                    USMContentTypesContactsErrorCodes.IMAGE_ERROR_NUMBER2,
                    "Can't write picture in type " + pictureFormat,
                    e);
            }
            int size = os.size();
            if (size == 0)
                return null;
            if (size <= maxSize)
                return os.toByteArray();
            scaleFactor *= 0.7f; // estimate was too big, reduce the size and try again
        }
        return null; // one quarter of the estimate is still too big, maybe the limit is too small, return "no image" instead
    }

    @Override
    protected void appendColumns(StringBuilder columns, ContentTypeField field) {
        if (field.getFieldID() == ContactContentTypeImpl.IMAGE1_FIELD_ID)
            columns.append(IMAGE_COLUMNS);
        else
            super.appendColumns(columns, field);
    }

    @Override
    protected void updateDataObjectFromJSONArray(DataObject destination, BitSet requestedFields, JSONArray array) throws InternalUSMException {
        int len = checkAndGetRequestedFieldLength(requestedFields);
        int j = 0;
        ContentTypeField[] fields = destination.getContentType().getFields();
        for (int i = 0; i < len; i++) {
            if (requestedFields.get(i)) {
                String fieldName = fields[i].getFieldName();
                updateDataObjectFieldFromJSONArray(array, j, fields[i], destination);

                if (fieldName.equals(ContactConstants.EMAIL1) || fieldName.equals(ContactConstants.EMAIL2) || fieldName.equals(ContactConstants.EMAIL3)) {
                    Object fieldValue = destination.getFieldContent(fieldName);
                    // Bug 42844: filter email addresses consisting of whitespaces only
                    if (fieldValue != null && fieldValue instanceof String && ((String) fieldValue).trim().isEmpty()) {
                        destination.setFieldContent(fieldName, null);
                    }
                }
                if (fields[i].getFieldID() == ContactContentTypeImpl.IMAGE1_FIELD_ID)
                    j += 3;
                else
                    j++;
            }
        }
    }

    private boolean containsImage(Image i) {
        return (i != null) && i.getContentType() != null && i.getContentType().length() > 0 && (i.getData() != null || i.getTempId() != null);
    }

    @Override
    public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
        Image i = (Image) object.getFieldContent(ImageDataType.IMAGE1);
        if (containsImage(i)) {
            if (LOG.isDebugEnabled())
                LOG.debug(object.getSession() + " Updating contact with picture");
            checkTypeToWrite(object, _contentType.getID());
            Map<String, String> parameters = new HashMap<String, String>();
            parameters.put(CommonConstants.ID, object.getID());
            parameters.put(CommonConstants.FOLDER, object.getOriginalParentFolderID());
            parameters.put(CommonConstants.TIMESTAMP, String.valueOf(timestamp));
            JSONObject requestBody = createRequestBody(object, true, false);

            if (i.getData() != null) {
                JSONResult result = _ajaxAccess.storeResource(
                    getOXAjaxAccessPath(),
                    CommonConstants.ACTION_UPDATE,
                    object.getSession(),
                    parameters,
                    requestBody,
                    i.getData(),
                    i.getContentType());
                checkResult(result, JSONResultType.JSONObject);
            } else {
                JSONResult result = storeImageFromTempFile(CommonConstants.ACTION_UPDATE, object, i, parameters, requestBody);
                checkResult(result, JSONResultType.JSONObject);
            }
            // XXX Shouldn't we use the timestamp returned from the OX server ?
            // object.setFieldContent(ImageDataType.IMAGE1, new Image(timestamp, null, null));
            object.setTimestamp(timestamp);
        } else
            super.writeUpdatedDataObject(object, timestamp);
    }

    @Override
    public void writeNewDataObject(DataObject object) throws USMException {
        Image i = (Image) object.getFieldContent(ImageDataType.IMAGE1);
        checkTypeToWrite(object, _contentType.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        JSONObject requestBody = createRequestBody(object, false, true);
        JSONResult result;

        if (containsImage(i)) {
            if (i.getData() != null) {
                if (LOG.isDebugEnabled())
                    LOG.debug(object.getSession() + " New contact with picture data");
                result = _ajaxAccess.storeResource(
                    getOXAjaxAccessPath(),
                    CommonConstants.ACTION_NEW,
                    object.getSession(),
                    parameters,
                    requestBody,
                    i.getData(),
                    i.getContentType());
            } else {
                if (LOG.isDebugEnabled())
                    LOG.debug(object.getSession() + " New contact with picture tempid " + i.getTempId());
                result = storeImageFromTempFile(CommonConstants.ACTION_NEW, object, i, parameters, requestBody);
            }
        } else {
            if (LOG.isDebugEnabled())
                LOG.debug(object.getSession() + " New " + _contentType.getID());

            if (requestBody.has(ImageDataType.IMAGE1) && requestBody.isNull(ImageDataType.IMAGE1))
                requestBody.remove(ImageDataType.IMAGE1);

            result = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_NEW, object.getSession(), parameters, requestBody);
        }

        checkResult(result, JSONResultType.JSONObject);
        JSONObject o = result.getJSONObject();
        readOptionalTimestamp(o, object);
        try {
            object.setID(o.getJSONObject(CommonConstants.RESULT_DATA).getString(CommonConstants.RESULT_ID));
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER1,
                object,
                "Missing result ID on new " + _contentType.getID(),
                e);
        }
    }

    public JSONResult storeImageFromTempFile(String action, DataObject object, Image i, Map<String, String> parameters, JSONObject requestBody) throws AuthenticationFailedException, OXCommunicationException, USMException {
        TempFile tempFile = null;
        try {
            tempFile = TempFileStorage.getTempFileForRead(object.getSession(), i.getTempId());
            return _ajaxAccess.storeResourceFromStream(
                getOXAjaxAccessPath(),
                action,
                object.getSession(),
                parameters,
                requestBody,
                i.getContentType(),
                tempFile.getSize(),
                tempFile);
        } catch (IOException e) {
            throw new USMException(
                USMContentTypesContactsErrorCodes.CAN_NOT_ACCESS_TEMP_FILE_FOR_IMAGE_DATA,
                "Can't access temp file for creating contact image with tempid '" + i.getTempId() + '\'',
                e);
        } finally {
            Toolkit.close(tempFile);
            if (tempFile != null) {
                try {
                    TempFileStorage.deleteTempFile(object.getSession(), i.getTempId());
                } catch (IOException e) {
                    if (LOG.isDebugEnabled())
                        LOG.debug("Can't delete temp file of contact image " + e);
                }
            }
        }
    }

    @Override
    protected void addFieldToRequestBody(DataObject object, boolean isOnlyModifiedFields, boolean isIdshouldNotBePresent, JSONObject requestBody, ContentTypeField field, int fieldIndex) throws InternalUSMException {
        if (!ContactConstants.NUMBER_OF_ATTACHMENTS.equals(field.getFieldName()))
            super.addFieldToRequestBody(object, isOnlyModifiedFields, isIdshouldNotBePresent, requestBody, field, fieldIndex);
    }

    public DataObject[] getAllUsers(Session session, BitSet requestedFields) throws AuthenticationFailedException, OXCommunicationException, UnsupportedContentOperationException, InternalUSMException {
        if (LOG.isDebugEnabled())
            LOG.debug(session + " Read all users ");
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
        JSONResult result = _ajaxAccess.doGet(CommonConstants.USERS_PATH, CommonConstants.ACTION_ALL, session, parameters);
        return getListOfResults(session, requestedFields, result);
    }

    private DataObject[] getListOfResults(Session session, BitSet requestedFields, JSONResult result) throws OXCommunicationException, InternalUSMException {
        JSONArray jsonar = null;
        long timestamp = 0L;
        checkResult(result, JSONResultType.JSONObject);
        JSONObject jsonObject = result.getJSONObject();
        try {
            jsonar = jsonObject.getJSONArray(CommonConstants.RESULT_DATA);
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER7,
                null,
                "getListOfResults: data array not present",
                e);
        }
        int length = jsonar.length();
        if (length == 0)
            return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
        try {
            if (jsonObject.has(CommonConstants.RESULT_TIMESTAMP))
                timestamp = jsonObject.getLong(CommonConstants.RESULT_TIMESTAMP);
        } catch (JSONException e) {
            throw generateException(USMContentTypesUtilErrorCodes.INVALID_TIMESTAMP_NUMBER3, null, "getListOfResults: invalid timestamp", e);
        }
        DataObject[] dataObjects = new DataObject[length];
        for (int j = 0; j < length; j++) {
            // get array which contains the information specified by the corresponding identifiers in the columns parameter
            try {
                JSONArray columnsArray = jsonar.getJSONArray(j);
                DataObject newDataObject = createDataObject(columnsArray, session, requestedFields, 0);
                dataObjects[j] = newDataObject;
                newDataObject.setTimestamp(timestamp);
            } catch (JSONException je) {
                throw generateException(
                    USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER6,
                    null,
                    "getListOfResults: error while reading index " + j + " of users ",
                    je);
            }
        }
        return dataObjects;
    }

    private DataObject createDataObject(JSONArray columnsArray, Session session, BitSet requestedFields, int idIndex) throws InternalUSMException, OXCommunicationException {
        String objId;
        try {
            objId = columnsArray.getString(idIndex);
        } catch (JSONException e) {
            throw generateException(USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER6, null, "read users: no id for new object", e);
        }
        DataObject dataObj = _contentType.newDataObject(session);
        dataObj.setID(objId);
        updateDataObjectFromJSONArray(dataObj, requestedFields, columnsArray);
        return dataObj;
    }

    public void getUser(DataObject object, BitSet requestedFields) throws AuthenticationFailedException, OXCommunicationException, UnsupportedContentOperationException, InternalUSMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Read user with id " + object.getID());
        checkTypeToWrite(object, _contentType.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.ID, object.getID());
        JSONResult result = _ajaxAccess.doGet(CommonConstants.USERS_PATH, CommonConstants.ACTION_GET, object.getSession(), parameters);
        checkResult(result, JSONResultType.JSONObject);
        updateDataObjectFromJSONResult(object, requestedFields, result);
    }

    @Override
    public void writeDeletedDataObject(DataObject object) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Delete contact with id " + object.getID() + " in folder " + object.getParentFolderID());
        checkTypeToWrite(object, _contentType.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.TIMESTAMP, String.valueOf(object.getTimestamp()));
        JSONObject jObj = new JSONObject();
        try {
            jObj.put(CommonConstants.ID, object.getID());
            jObj.put(CommonConstants.FOLDER, object.getParentFolderID());
        } catch (JSONException ignored) {
            // can not happen with our fixed keys
        }
        JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_DELETE, object.getSession(), parameters, jObj);
        checkResult(result, JSONResultType.JSONObject);
        try {
            JSONObject data = result.getJSONObject().getJSONObject(CommonConstants.RESULT_DATA);
            if (data.length() > 0)
                throw new ConflictingChangeException(
                    USMContentTypesContactsErrorCodes.DELETE_FAILED,
                    "Delete of contact '" + object.getID() + "' failed",
                    data);
            readOptionalTimestamp(result.getJSONObject(), object);
        } catch (JSONException e) {
            LOG.error("OX server did not send data object: " + result.toString());
            throw new OXCommunicationException(
                USMContentTypesContactsErrorCodes.DELETE_DATA_NOT_PRESENT,
                "OX server did not send data object",
                e,
                result.toString());
        }
    }

    /**
     * Special behavior of contacts regarding sorting: If no limit and no sorting is specified, sort explicitly by ID to avoid complex
     * automatic sorting within OX.
     */
    @Override
    protected void addObjectsLimitParameters(Session session, Map<String, String> parameters) {
        int limit = session.getObjectsLimit();
        if (limit > 0)
            addObjectsLimitParameters(limit, _LAST_MODIFIED_COLUMN, parameters); // sort by last modified
        else if (!parameters.containsKey(CommonConstants.SORT))
            parameters.put(CommonConstants.SORT, "1");
    }
}
