/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.configuration.ox;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.config.ConfigurationService;
import com.openexchange.usm.api.configuration.ConfigurationManager;
import com.openexchange.usm.api.exceptions.USMInvalidConfigurationException;

public class ConfigurationManagerAdapter implements ConfigurationManager {

    private static final Log LOG = LogFactory.getLog(ConfigurationManagerAdapter.class);

    private ConfigurationService _delegate;

    /**
     * Initializes a new {@link ConfigurationManagerAdapter}.
     * 
     * @param log
     * @param cs
     */
    public ConfigurationManagerAdapter(ConfigurationService cs) {
        _delegate = cs;
        LOG.info("USM-OX configuration bridge activated");
    }

    /**
     * Deactivate
     */
    public void deactivate() {
        _delegate = null;
        LOG.info("USM-OX configuration bridge deactivated");
    }

    @Override
    public String getProperty(String key, String defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException {
        return getStringProperty(key, defaultValue, warningIfNotPresent, false);
    }

    @Override
    public String getPropertyAsString(String key) throws USMInvalidConfigurationException {
        return getStringProperty(key, null, false, true);
    }

    @Override
    public boolean getProperty(String key, boolean defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException {
        return getBooleanProperty(key, defaultValue, warningIfNotPresent, false);
    }

    @Override
    public boolean getPropertyAsBoolean(String key) throws USMInvalidConfigurationException {
        return getBooleanProperty(key, false, false, true);
    }

    @Override
    public int getProperty(String key, int defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException {
        return getIntProperty(key, defaultValue, warningIfNotPresent, false);
    }

    @Override
    public int getPropertyAsInt(String key) throws USMInvalidConfigurationException {
        return getIntProperty(key, 0, false, true);
    }

    @Override
    public double getProperty(String key, double defaultValue, boolean warningIfNotPresent) throws USMInvalidConfigurationException {
        return getDoubleProperty(key, defaultValue, warningIfNotPresent, false);
    }

    @Override
    public double getPropertyAsDouble(String key) throws USMInvalidConfigurationException {
        return getDoubleProperty(key, 0.0, false, true);
    }

    private String getStringProperty(String key, String defaultValue, boolean warningIfNotPresent, boolean exceptionIfNotPresent) throws USMInvalidConfigurationException {
        String value = _delegate.getProperty(key);
        if (value != null)
            return value;
        if (exceptionIfNotPresent)
            throw new USMInvalidConfigurationException(
                USMConfigurationOXBridgeErrorCodes.MISSING_STRING_PROPERTY_VALUE,
                "Missing string configuration property " + key);
        if (warningIfNotPresent)
            LOG.warn("Configuration property " + key + " not set, using default " + defaultValue);
        return defaultValue;
    }

    private boolean getBooleanProperty(String key, boolean defaultValue, boolean warningIfNotPresent, boolean exceptionIfNotPresent) throws USMInvalidConfigurationException {
        String stringValue = _delegate.getProperty(key);
        if (stringValue != null)
            return Boolean.parseBoolean(stringValue);
        if (exceptionIfNotPresent)
            throw new USMInvalidConfigurationException(
                USMConfigurationOXBridgeErrorCodes.MISSING_BOOLEAN_PROPERTY_VALUE,
                "Missing boolean configuration property " + key);
        if (warningIfNotPresent)
            LOG.warn("Configuration property " + key + " not set, using default " + defaultValue);
        return defaultValue;
    }

    private int getIntProperty(String key, int defaultValue, boolean warningIfNotPresent, boolean exceptionIfNotPresent) throws USMInvalidConfigurationException {
        String stringValue = _delegate.getProperty(key);
        if (stringValue == null) {
            if (exceptionIfNotPresent)
                throw new USMInvalidConfigurationException(
                    USMConfigurationOXBridgeErrorCodes.MISSING_INTEGER_PROPERTY_VALUE,
                    "Missing integer configuration property " + key);
            if (warningIfNotPresent)
                LOG.warn("Configuration property " + key + " not set, using default " + defaultValue);
            return defaultValue;
        }
        try {
            return Integer.parseInt(stringValue);
        } catch (NumberFormatException e) {
            LOG.error(
                USMConfigurationOXBridgeErrorCodes.INVALID_INT_PROPERTY_VALUE + " Invalid value '" + stringValue + "' for integer configuration property " + key + ". Using default value instead!",
                e);
            return defaultValue;
        }
    }

    private double getDoubleProperty(String key, double defaultValue, boolean warningIfNotPresent, boolean exceptionIfNotPresent) throws USMInvalidConfigurationException {
        String stringValue = _delegate.getProperty(key);
        if (stringValue == null) {
            if (exceptionIfNotPresent)
                throw new USMInvalidConfigurationException(
                    USMConfigurationOXBridgeErrorCodes.MISSING_DOUBLE_PROPERTY_VALUE,
                    "Missing double configuration property " + key);
            if (warningIfNotPresent)
                LOG.warn("Configuration property " + key + " not set, using default " + defaultValue);
            return defaultValue;
        }
        try {
            return Double.parseDouble(stringValue);
        } catch (NumberFormatException e) {
            LOG.error(
                USMConfigurationOXBridgeErrorCodes.INVALID_DOUBLE_PROPERTY_VALUE + " Invalid value '" + stringValue + "' for double configuration property " + key + ". Using default value instead!",
                e);
            return defaultValue;
        }
    }
}
