/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.exceptions;

import org.apache.commons.httpclient.Header;
import org.json.JSONObject;

/**
 * RuntimeException that is generated if a severe error occurred during communication
 * with the OX server. Severe errors are:<ul>
 * <li>Server responses with http status code other than OK (200)
 * <li>An HttpException was caught during communication
 * <li>An IOExcaption was caught during communication
 * </ul>
 * The http status code can be retrieved using getHttpStatusCode(), it will
 * return -1 if no status code is available. The originating exception (if
 * present) can be retrieved using the getCause() method.
 * Additional headers may be set to add it to the EAS http response.<br>
 * Since this can happen on every synchronization call, it is implemented as a
 * RuntimeException to avoid adding it to each method signature.
 * 
 * @author afe
 *
 */
public class OXCommunicationException extends USMException {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private final int _httpStatusCode;

	private final JSONObject _jsonError;

	private final String _serverResponse;
	
	private final Header[] _headers;

	public OXCommunicationException(int errorCode, int statusCode) {
		super(errorCode, "http status code " + statusCode + " received");
		_httpStatusCode = statusCode;
		_jsonError = null;
		_serverResponse = null;
		_headers = null;
	}

    public OXCommunicationException(int errorCode, int statusCode, Header[] headers) {
        super(errorCode, "http status code " + statusCode + " received");
        _httpStatusCode = statusCode;
        _jsonError = null;
        _serverResponse = null;
        _headers = headers;
    }

	public OXCommunicationException(int errorCode, Throwable cause) {
		super(errorCode, "Exception occurred during communication with OX server", cause);
		_httpStatusCode = -1;
		_jsonError = null;
		_serverResponse = null;
        _headers = null;
	}

	public OXCommunicationException(int errorCode, JSONObject errorDetails) {
		super(errorCode, "OX server returned error");
		_httpStatusCode = -1;
		_jsonError = errorDetails;
		_serverResponse = null;
        _headers = null;
	}

	public OXCommunicationException(int errorCode, String message, JSONObject errorDetails) {
		super(errorCode, message);
		_httpStatusCode = -1;
		_jsonError = errorDetails;
		_serverResponse = null;
        _headers = null;
	}

	public OXCommunicationException(int errorCode, String message, Throwable cause) {
		super(errorCode, message, cause);
		_httpStatusCode = -1;
		_jsonError = null;
		_serverResponse = null;
        _headers = null;
	}
	
	public OXCommunicationException(int errorCode, String message, Throwable cause, String serverResponse) {
		super(errorCode, message, cause);
		_httpStatusCode = -1;
		_jsonError = null;
		_serverResponse = serverResponse;
        _headers = null;
	}

	public OXCommunicationException(int errorCode, String message, String serverResponse) {
		super(errorCode, message);
		_httpStatusCode = -1;
		_jsonError = null;
		_serverResponse = serverResponse;
        _headers = null;
	}

	public OXCommunicationException(int errorCode, String serverResponse) {
		super(errorCode);
		_httpStatusCode = -1;
		_jsonError = null;
		_serverResponse = serverResponse;
        _headers = null;
	}

	public boolean isHttpStatusCodeSet() {
		return _httpStatusCode != -1;
	}

	public int getHttpStatusCode() {
		return _httpStatusCode;
	}

	public boolean isJSONErrorSet() {
		return _jsonError != null;
	}

	public JSONObject getJSONError() {
		return _jsonError;
	}

	public boolean isServerResponseSet() {
		return _serverResponse != null;
	}

	public String getServerResponse() {
		return _serverResponse;
	}

	@Override
	public JSONObject getOxErrorForJSONResponse() {
		return _jsonError;
	}

	@Override
	public String getErrorMessage() {
		String message = super.getMessage();
		if (isServerResponseSet()) {
		    if (message == null) {
		        message = _serverResponse;
		    }
		    else {
		        message = super.getMessage() + " server-response: " + _serverResponse;
		    }
		}
		return message;
	}
	
    @Override
    public String getMessage() {
        return getErrorMessage();
    }

	@Override
	protected void addStacktrace(Throwable t, JSONObject errorDetailsObject) {
		if (t != this)
			super.addStacktrace(t, errorDetailsObject);
	}

    public Header[] getHeaders() {
        return _headers;
    }

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(super.toString());
		if (_httpStatusCode != -1)
			sb.append(", http-status-code=").append(_httpStatusCode);
		if (_jsonError != null)
			sb.append(", json-error=").append(_jsonError);
		if (_serverResponse != null)
			sb.append(", server-response=").append(_serverResponse);
        if (_headers != null) {
            sb.append(", server-specific-headers=");
                for (Header header: _headers) {
                    sb.append(header.getName() + ":" + header.getValue() + ",");
            }
        }
		return sb.toString();
	}
}
