/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.contenttypes.calendar;

import java.util.BitSet;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Session;

/**
 * Provides all relevant information about the Appointment (Calendar) content type.
 * 
 * @author ldo
 */
public interface AppointmentContentType extends ContentType {

    /**
     * This constant defines the name of a custom Session property that can be set for USM sessions. If it stores a value for which
     * Boolean.TRUE.equals() results in true (i.e. a Boolean object with a value of true), the behavior for deletions is modified for the
     * AppointmentContentType: A deletion of an appointment in which the owner of the folder is a participant, but for which the owner of
     * the folder is not the organizer will not result in an actual deletion, but instead the confirmation status of the owner of the folder
     * will be set to declined. Note that this only changes the current operation, any client protocol using this option has to make sure
     * (e.g. by defining an appropriate DataObjectFilter) that appointments that have been "deleted" in this way will not be reported back
     * to the client as new creations in later synchronizations.
     */
    String DECLINE_INSTEAD_OF_DELETE = "ContentType.Appointments.DeclineInsteadOfDelete";

    /**
     * Get a list of all appointments.
     * 
     * @param session
     * @param requestedFields
     * @return a list of unconfirmed appointments
     * @throws USMException
     */
    DataObject[] getAllAppointments(Session session, BitSet requestedFields) throws USMException;

    /**
     * Confirms the appointment. The confirmMessage will be set to empty String.
     * 
     * @param object the appointment to confirm
     * @param response the user response
     * @throws AuthenticationFailedException
     * @throws OXCommunicationException
     */
    void confirm(DataObject object, int response) throws AuthenticationFailedException, OXCommunicationException;

    /**
     * Confirms the appointment.
     * 
     * @param object the appointment to confirm
     * @param response the user response
     * @param confirmMessage optional message
     * @throws AuthenticationFailedException
     * @throws OXCommunicationException
     */
    void confirm(DataObject object, int response, String confirmMessage) throws AuthenticationFailedException, OXCommunicationException;

    /**
     * Confirms an appointment for a specified occurrence position
     * 
     * @param object
     * @param response
     * @param confirmMessage
     * @param participantId
     * @throws AuthenticationFailedException
     * @throws OXCommunicationException
     */
    void confirm(DataObject object, int response, Number occurrencePosition) throws AuthenticationFailedException, OXCommunicationException;

    /**
     * Confirms an appointment for a specified participant id (email address). Works only for external participants!
     * 
     * @param object
     * @param response
     * @param confirmMessage
     * @param participantId
     * @throws AuthenticationFailedException
     * @throws OXCommunicationException
     */
    void confirm(DataObject object, int response, String confirmMessage, String participantId) throws AuthenticationFailedException, OXCommunicationException;

    /**
     * Returns all series occurrences as single appointment objects. (Usage is not recommended! The result can contain large amount of
     * objects!)
     * 
     * @param session
     * @param requestedFields
     * @param seriesObject
     * @return
     * @throws USMException
     */
    DataObject[] getAllRecurrencesAsAppointments(Session session, BitSet requestedFields, DataObject seriesObject) throws USMException;

    /**
     * Returns the OX-ID based on the given UID of the appointment.
     * 
     * @param session
     * @param uid
     * @return
     * @throws USMException
     */
    String resolveUID(Session session, String uid) throws USMException;

    /**
     * Returns all change exceptions of the series appointment as objects.
     * 
     * @param session
     * @param requiredFields
     * @param seriesObject
     * @return
     * @throws USMException
     */
    DataObject[] getAllChangeExceptions(Session session, BitSet requestedFields, DataObject seriesObject) throws USMException;
}
