/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.office.dcs.registry;

import static org.apache.commons.lang3.StringUtils.isEmpty;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.office.tools.database.DatabaseException;
import com.openexchange.office.tools.database.DocumentsDatabase;


/**
 * {@link DCSDatabase}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.1
 */
@Service
@RegisteredService
public class DCSDatabase extends DocumentsDatabase implements InitializingBean {

    private static final Logger log = LoggerFactory.getLogger(DCSDatabase.class);

    @Autowired
    private DcsDatabaseConfiguration connectionData;

    @Override
	public void afterPropertiesSet() throws Exception {
    	log.trace("afterPropertiesSet for DCSDatabase called");
        if (isEmpty(this.connectionData.getConnectionURL()) ) {
            throw new DatabaseException("Property readUrl must not be empty (Example Url: jdbc:mysql://host:port/databasename)");
        }
        setReadConnectionData(connectionData);
        implCreateConnectionPools(true, false); 
	}

    /**
     * @return
     * @throws DatabaseException
     */
    public List<DCSRegistryItem> getRegisteredDCSItems() throws DatabaseException {
        List<DCSRegistryItem> items = new ArrayList<>();

        try (final Connection con = getReadConnection()) {
            final String queryItemsSql = "SELECT ID, Host, Interface, JMSPort, UseSSL FROM DCS";

            try (final PreparedStatement dcsStmt = con.prepareStatement(queryItemsSql);
                 final ResultSet resultSet = dcsStmt.executeQuery()) {

                if (null != resultSet) {
                   while (resultSet.next()) {
                       final DCSRegistryItem newItem = new DCSRegistryItem();
                       int index = 0;

                       newItem.setID(resultSet.getString(++index));
                       newItem.setHost(resultSet.getString(++index));
                       newItem.setInterface(resultSet.getString(++index));
                       newItem.setJMSPort(Integer.valueOf(resultSet.getInt(++index)));
                       newItem.setUseSSL(Boolean.valueOf(resultSet.getBoolean(++index)));

                       items.add(newItem);
                   }
                }
            } finally {
                // commit even pure query statements, since they could produce transactions (e.g. for AWS RDB)
                con.commit();
            }
        } catch (Exception e) {
            throw new DatabaseException("Exception caught when querying all registered DCSRegistryItems", e);
        }

        return items;
    }    
}
