package org.docx4j.openpackaging.parts;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import javax.xml.bind.JAXBException;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import org.docx4j.openpackaging.parts.relationships.Namespaces;

@XmlAccessorType(XmlAccessType.NONE)
public abstract class DocumentSerialization implements IDocumentSerialization {

    final protected Map<String, String> prefixToUri = new HashMap<String, String>();
    final protected Map<String, String> uriToPrefix = new HashMap<String, String>();
    final protected QName rootElement;
    protected String version = "1.0";
    private Boolean isStandalone = Boolean.TRUE;
    private String characterEncodingScheme = null;
    protected String[] namespaces;
    protected String ignorable = null;

    public DocumentSerialization(QName rootElement, String[] namespaces, String ignorable) {
        this.rootElement = rootElement;
        this.namespaces = namespaces;
        this.ignorable = ignorable;
    }

    /**
     * Gets the value of the ignorable property.
     *
     * @return
     *     possible object is
     *     {@link String }
     *
     */
    @XmlAttribute(name = "Ignorable", namespace = "http://schemas.openxmlformats.org/markup-compatibility/2006")
    public String getIgnorable() {
        return ignorable;
    }

    /**
     * Sets the value of the ignorable property.
     *
     * @param value
     *     allowed object is
     *     {@link String }
     *
     */
    public void setIgnorable(String value) {
        this.ignorable = value;
    }

    /* reads the root element of the document. Used prefixes and uris are saved and
     * will be written 1:1 when calling writeObject, this solves the problem we
     * had with prefixes used in the ignorable attribute and the corresponding uris. */

    @Override
    public void readObject(XMLStreamReader reader, DocumentPart<?> documentPart)
        throws JAXBException, XMLStreamException {

        final String v = reader.getVersion();
        if(v!=null) {
            version = v;
        }
        if(reader.standaloneSet()) {
            isStandalone = Boolean.valueOf(reader.isStandalone());
        }
        characterEncodingScheme = reader.getCharacterEncodingScheme();

        int tag = reader.nextTag();
        int namespaceCount = reader.getNamespaceCount();
        for(int i=0; i<namespaceCount; i++) {
            String prefix = reader.getNamespacePrefix(i);
            if(prefix==null) {
                prefix = "xmlns";
            }
            prefixToUri.put(prefix, reader.getNamespaceURI(i));
            uriToPrefix.put(reader.getNamespaceURI(i), prefix);
        }
        ignorable = mergeIgnorables(reader.getAttributeValue(Namespaces.MARKUP_COMPATIBILITY, "Ignorable"), ignorable);
    }

    static final public String[] Standard_DOCX_Namespaces = {
        "r", "http://schemas.openxmlformats.org/officeDocument/2006/relationships",
        "v", "urn:schemas-microsoft-com:vml",
        "w10", "urn:schemas-microsoft-com:office:word",
        "w", "http://schemas.openxmlformats.org/wordprocessingml/2006/main",
        "sl", "http://schemas.openxmlformats.org/schemaLibrary/2006/main",
        "w15", "http://schemas.microsoft.com/office/word/2012/wordml",
        "w16cid", "http://schemas.microsoft.com/office/word/2016/wordml/cid",
        "w16se", "http://schemas.microsoft.com/office/word/2015/wordml/symex",
        "w14", "http://schemas.microsoft.com/office/word/2010/wordml",
        "wp14", "http://schemas.microsoft.com/office/word/2010/wordprocessingDrawing",
        "m", "http://schemas.openxmlformats.org/officeDocument/2006/math",
        "o", "urn:schemas-microsoft-com:office:office",
        "xsi", "http://www.w3.org/2001/XMLSchema-instance",
        "dsp", "http://schemas.microsoft.com/office/drawing/2008/diagram",
        "pic", "http://schemas.openxmlformats.org/drawingml/2006/picture",
        "dgm", "http://schemas.openxmlformats.org/drawingml/2006/diagram",
        "xdr", "http://schemas.openxmlformats.org/drawingml/2006/spreadsheetDrawing",
        "c", "http://schemas.openxmlformats.org/drawingml/2006/chart",
        "a", "http://schemas.openxmlformats.org/drawingml/2006/main",
        "wne", "http://schemas.microsoft.com/office/word/2006/wordml",
        "wp", "http://schemas.openxmlformats.org/drawingml/2006/wordprocessingDrawing",
        "wps", "http://schemas.microsoft.com/office/word/2010/wordprocessingShape",
        "cx1", "http://schemas.microsoft.com/office/drawing/2014/chartex",
        "cx2", "http://schemas.microsoft.com/office/drawing/2015/10/21/chartex",
        "cx4", "http://schemas.microsoft.com/office/drawing/2016/5/10/chartex",
        "wpc", "http://schemas.microsoft.com/office/word/2010/wordprocessingCanvas",
        "wpg", "http://schemas.microsoft.com/office/word/2010/wordprocessingGroup",
        "", "http://schemas.openxmlformats.org/drawingml/2006/lockedCanvas",
        "", "http://schemas.openxmlformats.org/drawingml/2006/compatibility",
        "", "http://schemas.openxmlformats.org/officeDocument/2006/bibliography",
        "", "http://schemas.microsoft.com/office/2006/coverPageProps",
        "", "http://schemas.microsoft.com/office/drawing/2012/chartStyle",
        "", "http://schemas.openxmlformats.org/drawingml/2006/chartDrawing",
        "", "http://schemas.microsoft.com/office/drawing/2010/main",
        "", "urn:schemas-microsoft-com:office:excel",
        "", "urn:schemas-microsoft-com:office:powerpoint"
    };

    static final public String Standard_DOCX_Ignorables = "w14 w15 w16se w16cid wp14";

    @Override
    public void writeObject(XMLStreamWriter writer, DocumentPart<?> documentPart)
        throws XMLStreamException, JAXBException {

        writer.writeStartDocument("UTF-8", version);
        writer.writeStartElement(rootElement.getPrefix(), rootElement.getLocalPart(), rootElement.getNamespaceURI());

        final Iterator<Entry<String, String>> namespaceIter = prefixToUri.entrySet().iterator();
        while(namespaceIter.hasNext()) {
            final Entry<String, String> namespace = namespaceIter.next();
            final String u = namespace.getValue();
            final String p = namespace.getKey();
            if(p.equals("xmlns")) {
                writer.writeDefaultNamespace(u);
            }
            else {
                writer.writeNamespace(p, u);
            }
        }
        if(!prefixToUri.containsKey("xmlns")) {
            writer.writeDefaultNamespace("");
        }

        // ensure to use namespaces which are defined in package infos,
        // so the fragment writer will not write namespaces each time
        int nsPrefixNumber = 1;
        for(int i=0;i<namespaces.length;i++) {
            final String prefix = namespaces[i++];
            final String namespace = namespaces[i];
            if(!uriToPrefix.containsKey(namespace)) {
                if(!prefix.isEmpty()&&!prefixToUri.containsKey(prefix)) {
                    writer.writeNamespace(prefix, namespace);
                }
                else {
                    while(true) {
                        final String nsPrefix = "ns" + Integer.valueOf(nsPrefixNumber++).toString();
                        if(!prefixToUri.containsKey(nsPrefix)) {
                            writer.writeNamespace(nsPrefix, namespace);
                            break;
                        }
                    }
                }
            }
        }
        if(writer.getPrefix(Namespaces.MARKUP_COMPATIBILITY)==null) {
            writer.writeNamespace("mc", Namespaces.MARKUP_COMPATIBILITY);
        }
        if(ignorable!=null&&!ignorable.isEmpty()) {
            writer.writeAttribute(Namespaces.MARKUP_COMPATIBILITY, "Ignorable", ignorable);
        }
    }

    private String mergeIgnorables(String a, String b) {
        final Set<String> set = new HashSet<String>(8);
        stringArrayToSet(set, a);
        stringArrayToSet(set, b);
        return setToStringArray(set);
    }

    private void stringArrayToSet(Set<String> set, String array) {
        if(array!=null) {
            for(String a : array.split(" ", -1)) {
                set.add(a);
            }
        }
    }

    private String setToStringArray(Set<String> set) {
        if(set.isEmpty()) {
            return null;
        }
        final StringBuilder builder = new StringBuilder(32);
        Iterator<String> iter = set.iterator();
        while(iter.hasNext()) {
            if(builder.length()!=0) {
                builder.append(' ');
            }
            builder.append(iter.next());
        }
        return builder.toString();
    }
}
