/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.fileitem.osgi;

import java.io.File;
import java.net.URI;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;
import org.apache.commons.collections.BidiMap;
import org.apache.commons.collections.bidimap.DualHashBidiMap;
import org.apache.commons.lang.StringUtils;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.ConfigurationService;
import com.openexchange.database.DatabaseService;
import com.openexchange.database.Databases;
import com.openexchange.database.migration.DBMigrationMonitorService;
import com.openexchange.exception.OXException;
import com.openexchange.fileitem.impl.ConnectionWrapper;
import com.openexchange.fileitem.impl.FileItemDatabase;
import com.openexchange.fileitem.impl.FileItemService;
import com.openexchange.fileitem.impl.FileItemUtils;
import com.openexchange.fileitem.impl.Services;
import com.openexchange.filestore.DatabaseAccess;
import com.openexchange.filestore.DatabaseAccessProvider;
import com.openexchange.filestore.DatabaseTable;
import com.openexchange.filestore.FileStorage;
import com.openexchange.filestore.FileStorageCodes;
import com.openexchange.filestore.FileStorageService;
import com.openexchange.filestore.FileStorages;
import com.openexchange.folder.FolderService;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.osgi.HousekeepingActivator;

/**
 * {@link FileItemServiceActivator}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItemServiceActivator extends HousekeepingActivator {

    final private static String SERVICE_NAME = "FileItemService";
    final private static String FILEITEM_STORE_PREFIX = "fis";

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] {
            ConfigurationService.class,
            FileStorageService.class,
            FolderService.class,
            DatabaseService.class,
            DBMigrationMonitorService.class
        };
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.HousekeepingActivator#start(org.osgi.framework.BundleContext)
     */
    @Override
    public void start(BundleContext ctx) throws Exception {
        super.start(ctx);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#startBundle()
     */
    @Override
    protected void startBundle() throws Exception {
        FileItemUtils.logInfo("starting bundle: " + SERVICE_NAME);

        Services.setServiceLookup(this);

        m_dbMigrationMonitorTracker = new ServiceTracker<>(context, DBMigrationMonitorService.class, new DBMigrationMonitorCustomizer(context));
        m_dbMigrationMonitorTracker.open();

        openTrackers();

        FileItemUtils.logInfo("successfully started bundle: " + SERVICE_NAME);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.HousekeepingActivator#stopBundle()
     */
    @Override
    protected void stopBundle() throws Exception {
        FileItemUtils.logInfo("stopping bundle: " + SERVICE_NAME);

        if (null != m_fileItemService) {
            m_fileItemService.shutdown();
            m_fileItemService = null;
        }

        closeTrackers();
        unregisterServices();

        Services.setServiceLookup(null);

        FileItemUtils.logInfo("successfully stopped bundle: " + SERVICE_NAME);
    }

    /**
     *
     */
    protected void implRegisterService() {
        final ConfigurationService configService = getService(ConfigurationService.class);

        FileItemUtils.logInfo("Registering FileItemService");

        if (null != configService) {
            // set spool dir
            FileItemUtils.setSpoolDir(new File(configService.getProperty("com.openexchange.fileItem.spoolPath", "/tmp")));

            // try to get configured filestore
            final Set<Integer> storeIdSet = implGetConfiguredStoreIds(configService);
            final Optional<FileItemDatabase> fileItemDatabaseOpt = Optional.ofNullable(
                implCreateFileItemDatabase(configService, storeIdSet));
            final Optional<BidiMap> bidiMapOpt = Optional.ofNullable(
                implGetFileStores(configService, storeIdSet));

            registerService(IFileItemService.class, m_fileItemService = new FileItemService(bidiMapOpt, fileItemDatabaseOpt));
        }

        if (null == m_fileItemService) {
            FileItemUtils.logError("Could not register FileItemService");
        } else {
            FileItemUtils.logInfo("Successfully registered FileItemService");
        }
    }

    /**
     * @param configService
     * @return
     */
    private @Nullable BidiMap implGetFileStores(@NonNull final ConfigurationService configService, @NonNull Set<Integer> storeIdSet) {
        final FileStorageService fileStoreService = getService(FileStorageService.class);
        BidiMap ret = new DualHashBidiMap();

        if (null != fileStoreService) {
            int storeIndex = 0;

            // retrieve file stores for the given ids
            for (final Integer curStoreId : storeIdSet) {
                try {
                    final URI fileStorageURI = FileStorages.getFullyQualifyingUriFor(curStoreId.intValue(), FILEITEM_STORE_PREFIX + ++storeIndex);
                    final FileStorage fileStorage = (null != fileStorageURI) ? fileStoreService.getFileStorage(fileStorageURI) : null;

                    if (null != fileStorage) {
                        ret.put(fileStorage, curStoreId);
                    } else {
                        storeIdSet.remove(curStoreId);
                    }
                } catch (IllegalStateException e) {
                    // this exception is thrown, if the FileStorageInfoService has been
                    // started, but the internal members have not been initialized
                    // accordingly right now => return null to indicate this situation
                    // in order to try again later in time
                    FileItemUtils.logTrace("FileStorages#getFullyQualifyingUriFor is not yet able to successfully finish: ", Throwables.getRootCause(e));

                    return null;
                } catch (OXException e) {
                    FileItemUtils.logError(new StringBuilder(256).append("Could not retrieve FileStore for given config id: ").append(curStoreId.intValue()).append("(Reason: ").append(Throwables.getRootCause(e).getMessage()).append(')').toString());
                }
            }

            if (ret.size() < 1) {
                FileItemUtils.logError("FileStoreService is available, but no valid FileStore " + "could be found at all. Please check, if a FileStore has been correctly " + "registered via 'registerfilestore' and at least one FileStore registration " + "number has been correctly applied to the 'com.openexchange.fileItem.fileStoreIds' property");
            }
        } else {
            FileItemUtils.logError("Could not retrieve FileStoreService interface");
        }

        return ret;
    }

    /**
     * @param configService
     * @param storeIdSet
     * @return
     */
    private @Nullable FileItemDatabase implCreateFileItemDatabase(@NonNull final ConfigurationService configService, @NonNull final Set<Integer> storeIdSet) {
        FileItemDatabase ret = null;

        try {
            final FileItemDatabase fileItemDatabase = (ret = new FileItemDatabase(configService));

            if (null != fileItemDatabase) {
                final DatabaseService dbService = getService(DatabaseService.class);

                if (null != dbService) {
                    registerService(DatabaseAccessProvider.class, new DatabaseAccessProvider() {

                        @Override
                        public DatabaseAccess getAccessFor(int fileStorageId, String prefix) throws OXException {
                            return (StringUtils.startsWith(prefix, FILEITEM_STORE_PREFIX) && storeIdSet.contains(Integer.valueOf(fileStorageId))) ?
                                new FileItemDatabaseAccess(fileItemDatabase) :
                                    null;
                        }
                    });
                }
            }
        } catch (Exception e) {
            FileItemUtils.logExcp(e);
        }

        return ret;
    }

    /**
     * @param configService
     * @return
     */
    private Set<Integer> implGetConfiguredStoreIds(@NonNull final ConfigurationService configService) {
        final Set<Integer> ret = new LinkedHashSet<>();

        // retrieve valid file store ids from config
        for (String curStoreIdStr : configService.getProperty("com.openexchange.fileItem.fileStoreIds", "-1").split(",")) {
            if (StringUtils.isNotBlank(curStoreIdStr)) {
                try {
                    final Integer curStoreId = Integer.valueOf(StringUtils.trim(curStoreIdStr));

                    if (curStoreId.intValue() > 0) {
                        ret.add(curStoreId);
                    }
                } catch (@SuppressWarnings("unused") NumberFormatException e) {
                    FileItemUtils.logError("Could not parse filestore id: " + curStoreIdStr);
                }
            }
        }

        return ret;
    }

    // - Members ---------------------------------------------------------------

    private ServiceTracker<DBMigrationMonitorService, DBMigrationMonitorService> m_dbMigrationMonitorTracker = null;

    private FileItemService m_fileItemService = null;


    // - Inner classes ---------------------------------------------------------

    /**
     * {@link FileItemDatabaseAccess}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.3

     */
    private static class FileItemDatabaseAccess implements DatabaseAccess {

        /**
         * {@link ConnectionWrappperCreator}
         *
         * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
         * @since v7.10.3

         * @param <R>
         */
        private interface ConnectionWrappperCreator {
            ConnectionWrapper create() throws OXException;
        }

        /**
         * Initializes a new {@link FileItemDatabaseAccess}.
         * @param fileItemDatabase
         */
        FileItemDatabaseAccess(@NonNull final FileItemDatabase fileItemDatabase) {
            super();
            m_fileItemDatabase = fileItemDatabase;
        }

        /**
         *
         */
        @Override
        public void createIfAbsent(DatabaseTable... tables) throws OXException {
            final Connection con = acquireWritable();

            if (null == con) {
                return;
            }

            int rollback = 0;

            try {
                Databases.startTransaction(con);
                rollback = 1;

                for (DatabaseTable curTable : tables) {
                    if (!Databases.tableExists(con, curTable.getTableName())) {
                        try (final PreparedStatement stmt = con.prepareStatement(curTable.getCreateTableStatement())) {
                            stmt.executeUpdate();
                        }
                    }
                }

                rollback = 2;
            } catch (Exception e) {
                throw FileStorageCodes.SQL_PROBLEM.create(e, e.getMessage());
            } finally {
                if (rollback > 0) {
                    if (rollback == 1) {
                        Databases.rollback(con);
                    }
                }

                releaseWritable(con, false);
            }
        }

        /**
         *
         */
        @Override
        public Connection acquireReadOnly() throws OXException {
            return implAcquireConnection(m_readConMap, () -> m_fileItemDatabase.getReadConnectionWrapper());
        }

        /**
         *
         */
        @Override
        public void releaseReadOnly(Connection con) {
            implReleaseConnection(m_readConMap, con);
        }

        /**
         *
         */
        @Override
        public Connection acquireWritable() throws OXException {
            return implAcquireConnection(m_writeConMap, () -> m_fileItemDatabase.getWriteConnectionWrapper());
        }

        /**
         *
         */
        @Override
        public void releaseWritable(Connection con, boolean forReading) {
            implReleaseConnection(m_writeConMap, con);
        }

        // - Implementation ----------------------------------------------------

        /**
         * @param con
         * @param conMap
         */
        private synchronized Connection implAcquireConnection(
            @NonNull final Map<Connection, ConnectionWrapper> conMap,
            @NonNull final ConnectionWrappperCreator connectionWrappperCreator ) throws OXException {

            Connection ret = null;

            try {
                final ConnectionWrapper conWrapper = connectionWrappperCreator.create();

                if ((null != conWrapper) && (null != (ret = conWrapper.getConnection()))) {
                    // Connection map is synchronized
                    conMap.put(ret, conWrapper);
                }
            } catch (Exception e) {
                throw new OXException(e);
            }

            return ret;
        }

        /**
         * @param con
         * @param conMap
         */
        private void implReleaseConnection(@NonNull final Map<Connection, ConnectionWrapper> conMap, final Connection con) {
            if (null != con) {
                // Connection map is synchronized;
                // ConnectionWrapper is closed via try-with-resource after a commit on the connection;
                try (final ConnectionWrapper conWrapper = conMap.remove(con)) {
                    if (null != conWrapper) {
                        conWrapper.commit();
                    }
                } catch (Exception e) {
                    FileItemUtils.logExcp(e);
                }
            }
        }

        // - Members -----------------------------------------------------------

        final private Map<Connection, ConnectionWrapper> m_readConMap = Collections.synchronizedMap(new HashMap<>());

        final private Map<Connection, ConnectionWrapper> m_writeConMap = Collections.synchronizedMap(new HashMap<>());

        final private FileItemDatabase m_fileItemDatabase;
    }

    /**
     * {@link DBMigrationMonitorCustomizer}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.0
     */
    private class DBMigrationMonitorCustomizer implements ServiceTrackerCustomizer<DBMigrationMonitorService, DBMigrationMonitorService> {

        private final BundleContext m_context;

        /**
         * Initializes a new {@link DBMigrationMonitorCustomizer}.
         *
         * @param parent
         * @param context
         */
        DBMigrationMonitorCustomizer(final BundleContext context) {
            m_context = context;
        }

        /*
         * (non-Javadoc)
         *
         * @see org.osgi.util.tracker.ServiceTrackerCustomizer#addingService(org.osgi.framework.ServiceReference)
         */
        @Override
        public DBMigrationMonitorService addingService(ServiceReference<DBMigrationMonitorService> reference) {
            if (null != reference) {
                final DBMigrationMonitorService migrationMonitor = m_context.getService(reference);

                if (migrationMonitor != null) {
                    Executors.newSingleThreadExecutor().execute(() -> {
                        try {
                            boolean dbUpdateInProgress = !migrationMonitor.getScheduledFiles().isEmpty();
                            if (dbUpdateInProgress) {
                                long waitNanos = TimeUnit.SECONDS.toNanos(1L); // 1 second
                                do {
                                    LockSupport.parkNanos(waitNanos);
                                    dbUpdateInProgress = !migrationMonitor.getScheduledFiles().isEmpty();
                                } while (dbUpdateInProgress);
                            }
                        } catch (@SuppressWarnings("unused") Exception e) {
                            // ok
                        }

                        implRegisterService();
                    });

                    return migrationMonitor;
                }

                m_context.ungetService(reference);
            }

            return null;
        }

        /*
         * (non-Javadoc)
         *
         * @see org.osgi.util.tracker.ServiceTrackerCustomizer#modifiedService(org.osgi.framework.ServiceReference, java.lang.Object)
         */
        @Override
        public void modifiedService(ServiceReference<DBMigrationMonitorService> reference, DBMigrationMonitorService service) {
            //
        }

        /*
         * (non-Javadoc)
         *
         * @see org.osgi.util.tracker.ServiceTrackerCustomizer#removedService(org.osgi.framework.ServiceReference, java.lang.Object)
         */
        @Override
        public void removedService(ServiceReference<DBMigrationMonitorService> reference, DBMigrationMonitorService service) {
            m_context.ungetService(reference);
        }
    }
}
