/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.imap.command;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.mail.MessagingException;
import com.openexchange.imap.command.MailMessageFetchIMAPCommand.MailMessageFetchInterceptor;
import com.openexchange.mail.IndexRange;
import com.openexchange.mail.dataobjects.MailMessage;
import com.openexchange.mail.utils.MailMessageComparator;


/**
 * {@link RangeSortingInterceptor} - An interceptor that holds a sorted range.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.10.3
 */
public class RangeSortingInterceptor implements MailMessageFetchInterceptor {

    private final int total;
    private final List<MailMessage> mails;
    private final MailMessageComparator comparator;
    private final IndexRange indexRange;
    private boolean sorted;

    /**
     * Initializes a new {@link RangeSortingInterceptor}.
     */
    public RangeSortingInterceptor(IndexRange indexRange, MailMessageComparator comparator) {
        super();
        this.indexRange = indexRange;
        this.comparator = comparator;
        total = indexRange.getEnd();
        mails = new ArrayList<>(total + 1);
        sorted = false;
    }

    /**
     * Gets the comparator.
     *
     * @return The comparator
     */
    public MailMessageComparator getComparator() {
        return comparator;
    }

    /**
     * Gets the index range.
     *
     * @return The index range
     */
    public IndexRange getIndexRange() {
        return indexRange;
    }

    @Override
    public void intercept(MailMessage mail) throws MessagingException {
        if (mail != null && mails.add(mail) && (mails.size() > total)) {
            // Appended mail to the end of the list
            // Added more mails than requested. Need to sort & remove last.
            Collections.sort(mails, comparator);
            sorted = true;
            mails.remove(total);
        }
    }

    @Override
    public MailMessage[] getMails() {
        int size = mails.size();
        if (size <= 0) {
            return new MailMessage[0];
        }

        int fromIndex = indexRange.start;
        if ((fromIndex) > size) {
            // Return empty array if start is out of range
            return new MailMessage[0];
        }

        // Reset end index if out of range
        int toIndex = indexRange.end;
        if (toIndex > size) {
            toIndex = size;
        }
        MailMessage[] arr = new MailMessage[toIndex - fromIndex];
        if (!sorted) {
            Collections.sort(mails, comparator);
            sorted = true;
        }
        for (int k = fromIndex, i = 0; k < toIndex; k++, i++) {
            arr[i] = mails.get(k);
        }
        return arr;
    }

}
