/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.drive.events.fcm.osgi;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Properties;
import org.osgi.framework.ServiceReference;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.firebase.FirebaseOptions;
import com.openexchange.config.lean.LeanConfigurationService;
import com.openexchange.drive.events.DriveEventService;
import com.openexchange.drive.events.fcm.FCMKeyProvider;
import com.openexchange.drive.events.fcm.groupware.RenameGCM2FCMUpdateTask;
import com.openexchange.drive.events.fcm.internal.DriveEventsFCMProperty;
import com.openexchange.drive.events.fcm.internal.FCMDriveEventPublisher;
import com.openexchange.drive.events.subscribe.DriveSubscriptionStore;
import com.openexchange.fragment.properties.loader.FragmentPropertiesLoader;
import com.openexchange.groupware.update.DefaultUpdateTaskProviderService;
import com.openexchange.groupware.update.UpdateTaskProviderService;
import com.openexchange.java.Strings;
import com.openexchange.osgi.HousekeepingActivator;
import com.openexchange.osgi.SimpleRegistryListener;

/**
 * {@link FCMActivator}
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class FCMActivator extends HousekeepingActivator {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(FCMActivator.class);
    private static final String SCOPES = "https://www.googleapis.com/auth/firebase.messaging";

    /**
     * Initializes a new {@link FCMActivator}.
     */
    public FCMActivator() {
        super();
    }

    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { DriveEventService.class, DriveSubscriptionStore.class, LeanConfigurationService.class };
    }

    @Override
    protected Class<?>[] getOptionalServices() {
        return new Class<?>[] { FCMKeyProvider.class };
    }

    @Override
    protected void startBundle() throws Exception {
        LOG.info("Starting bundle: {}", context.getBundle().getSymbolicName());

        registerService(UpdateTaskProviderService.class, new DefaultUpdateTaskProviderService(new RenameGCM2FCMUpdateTask()));

        track(FragmentPropertiesLoader.class, new SimpleRegistryListener<FragmentPropertiesLoader>() {

            private FCMKeyProvider provider;

            @Override
            public synchronized void added(ServiceReference<FragmentPropertiesLoader> ref, FragmentPropertiesLoader service) {
                Properties properties = service.load(DriveEventsFCMProperty.FRAGMENT_FILE_NAME);
                if (properties == null) {
                    return;
                }
                String key = properties.getProperty(DriveEventsFCMProperty.KEYPATH.getFQPropertyName());
                if (Strings.isEmpty(key)) {
                    return;
                }
                final FirebaseOptions options;
                try (InputStream keyStream = service.loadResource(key)) {
                    GoogleCredentials googleCredentials = GoogleCredentials.fromStream(keyStream).createScoped(Arrays.asList(SCOPES));
                    options = FirebaseOptions.builder().setCredentials(googleCredentials).build();
                } catch (IOException e) {
                    return;
                }
                String name = properties.getProperty(DriveEventsFCMProperty.NAME.getFQPropertyName());
                if (Strings.isEmpty(name)) {
                    return;
                }
                provider = new FCMKeyProvider() {

                    @Override
                    public String getName() {
                        return name;
                    }

                    @Override
                    public FirebaseOptions getOptions() {
                        return options;
                    }

                };
                registerService(FCMKeyProvider.class, provider);
            }

            @Override
            public synchronized void removed(ServiceReference<FragmentPropertiesLoader> ref, FragmentPropertiesLoader service) {
                if (provider != null) {
                    unregisterService(provider);
                }
            }
        });
        openTrackers();

        //@formatter:off
        getServiceSafe(DriveEventService.class).registerPublisher(new FCMDriveEventPublisher(this,
                        getServiceSafe(DriveSubscriptionStore.class),
                        getServiceSafe(LeanConfigurationService.class)));
        //@formatter:on
    }

    @Override
    protected void stopBundle() throws Exception {
        LOG.info("Stopping bundle: {}", context.getBundle().getSymbolicName());
        super.stopBundle();
    }
}
