/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 Open-Xchange GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.git.conflict;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.git.MergeException;
import com.openexchange.git.data.Conflict;

/**
 * Implements the conflict solver for project set files. It currently can only solve the following merges:
 * release-somewhat -> develop, master
 * hotfix-<SWAMPID> -> develop
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class ProjectSet implements Solver {

    private static final Log LOG = LogFactory.getLog(ProjectSet.class);

    private static final String REGEX = "^(<project reference=\"1\\.0,https://git\\.open-xchange\\.com/git/.*?,)(.*?)(,[\\w\\-\\.]*\"/>)$";
    private static final Pattern PATTERN = Pattern.compile(REGEX);

    public ProjectSet() {
        super();
    }

    public boolean applies(Conflict conflict) {
//        <<<<<<< HEAD
//        <project reference="1.0,https://git.open-xchange.com/git/wd/backend,master,com.openexchange.jslob.test"/>
//        <project reference="1.0,https://git.open-xchange.com/git/wd/backend,master,com.openexchange.jslob.test"/>
//        =======
//        <project reference="1.0,https://git.open-xchange.com/git/wd/backend,release-7.0.2,com.openexchange.jslob.test"/>
//        <project reference="1.0,https://git.open-xchange.com/git/wd/backend,release-7.0.2,com.openexchange.jslob.test"/>
//        >>>>>>> origin/release-7.0.2

//        <<<<<<< HEAD
//        <project reference="1.0,https://git.open-xchange.com/git/wd/backend,develop,com.openexchange.jslob.test"/>
//        =======
//        <project reference="1.0,https://git.open-xchange.com/git/wd/backend,master-7.0.1,com.openexchange.jslob.test"/>
//        >>>>>>> hotfix-1377
        List<String> here = conflict.getHere();
        List<String> there = conflict.getThere();
        String hereBranch;
        try {
            hereBranch = parseHereBranch(conflict);
        } catch (MergeException e) {
            return false;
        }
        if (conflict.getThereName().contains("release-")) {
            if ("master".equals(hereBranch) || "develop".equals(hereBranch)) {
                return testMatch(here, there);
            }
        }
        if (conflict.getThereName().contains("hotfix-")) {
            if (conflict.getHere().size() == conflict.getThere().size()) {
                boolean allSame = true;
                for (int i = 0; i < here.size() && allSame; i++) {
                    Matcher matcherHere = PATTERN.matcher(here.get(i));
                    Matcher matcherThere = PATTERN.matcher(there.get(i));
                    if (!matcherHere.matches() || !matcherThere.matches()) {
                        return false;
                    }
                    allSame = matcherHere.group(3).equals(matcherThere.group(3));
                }
                return allSame;
            }
        }
        return false;
    }

    private static boolean testMatch(List<String> here, List<String> there) {
        for (String line : here) {
            if (!PATTERN.matcher(line).matches()) {
                return false;
            }
        }
        for (String line : there) {
            if (!PATTERN.matcher(line).matches()) {
                return false;
            }
        }
        return true;
    }

    public List<String> solve(Conflict conflict) throws MergeException {
        final List<String> retval;
        if (conflict.getThereName().contains("release-")) {
            retval = mergeReleaseBranch(conflict);
        } else if (conflict.getThereName().contains("hotfix-")) {
            retval = mergeHotfixBranch(conflict);
        } else {
            throw new MergeException();
        }
        return retval;
    }

    private static List<String> mergeReleaseBranch(Conflict conflict) throws MergeException {
//      <<<<<<< HEAD
//      <project reference="1.0,https://git.open-xchange.com/git/wd/backend,master,com.openexchange.jslob.test"/>
//      <project reference="1.0,https://git.open-xchange.com/git/wd/backend,master,com.openexchange.jslob.test"/>
//      =======
//      <project reference="1.0,https://git.open-xchange.com/git/wd/backend,release-7.0.2,com.openexchange.jslob.test"/>
//      <project reference="1.0,https://git.open-xchange.com/git/wd/backend,release-7.0.2,com.openexchange.jslob.test"/>
//      >>>>>>> origin/release-7.0.2
        String hereBranch = parseHereBranch(conflict);
        List<String> lines = conflict.getThere();
        List<String> retval = new ArrayList<String>(lines.size());
        for (String line : lines) {
            Matcher matcher = PATTERN.matcher(line);
            matcher.matches();
            retval.add(matcher.group(1) + hereBranch + matcher.group(3));
        }
        return retval;
    }

    private static List<String> mergeHotfixBranch(Conflict conflict) throws MergeException {
//      <<<<<<< HEAD
//      <project reference="1.0,https://git.open-xchange.com/git/wd/backend,develop,com.openexchange.jslob.test"/>
//      =======
//      <project reference="1.0,https://git.open-xchange.com/git/wd/backend,master-7.0.1,com.openexchange.jslob.test"/>
//      >>>>>>> hotfix-1377
        final List<String> retval;
        String hereBranch = parseHereBranch(conflict);
        List<String> lines = conflict.getThere();
        retval = new ArrayList<String>(lines.size());
        for (String line : lines) {
            Matcher matcher = PATTERN.matcher(line);
            matcher.matches();
            retval.add(matcher.group(1) + hereBranch + matcher.group(3));
        }
        return retval;
    }

    private static String parseHereBranch(Conflict conflict) throws MergeException {
        List<String> lines = conflict.getHere();
        if (0 == lines.size()) {
            throw new MergeException("Can not determine name of branch to retain.");
        }
        Matcher matcher = PATTERN.matcher(lines.get(0));
        if (matcher.matches()) {
            return matcher.group(2);
        }
        throw new MergeException("Pattern did not match.");
    }
}
