/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.drive.events.fcm.internal;

import static com.openexchange.java.Autoboxing.I;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.IntStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.firebase.FirebaseApp;
import com.google.firebase.FirebaseOptions;
import com.google.firebase.messaging.AndroidConfig;
import com.google.firebase.messaging.FirebaseMessaging;
import com.google.firebase.messaging.FirebaseMessagingException;
import com.google.firebase.messaging.Message;
import com.openexchange.config.lean.LeanConfigurationService;
import com.openexchange.drive.events.DriveContentChange;
import com.openexchange.drive.events.DriveEvent;
import com.openexchange.drive.events.DriveEventPublisher;
import com.openexchange.drive.events.fcm.FCMKeyProvider;
import com.openexchange.drive.events.subscribe.DriveSubscriptionStore;
import com.openexchange.drive.events.subscribe.Subscription;
import com.openexchange.drive.events.subscribe.SubscriptionMode;
import com.openexchange.drive.events.subscribe.SubscriptionTransport;
import com.openexchange.exception.OXException;
import com.openexchange.java.Strings;
import com.openexchange.server.ServiceLookup;

/**
 * {@link FCMDriveEventPublisher}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class FCMDriveEventPublisher implements DriveEventPublisher {

    private static final Logger LOG = LoggerFactory.getLogger(FCMDriveEventPublisher.class);

    private static final String SERVICE_ID = SubscriptionTransport.FCM.name().toLowerCase();
    private static final String[] SERVICE_IDS = new String[] { SERVICE_ID };
    private static final String SCOPES = "https://www.googleapis.com/auth/firebase.messaging";

    private final ServiceLookup services;
    private final DriveSubscriptionStore subscriptionStore;
    private final LeanConfigurationService configService;

    /**
     * Initializes a new {@link FCMDriveEventPublisher}.
     */
    public FCMDriveEventPublisher(ServiceLookup services, DriveSubscriptionStore subscriptionStore, LeanConfigurationService configService) {
        super();
        this.services = services;
        this.subscriptionStore = subscriptionStore;
        this.configService = configService;
    }

    @Override
    public boolean isLocalOnly() {
        return true;
    }

    @Override
    public void publish(DriveEvent event) {
        for (Subscription subscription : getSubscriptions(event)) {
            processSubscription(subscription, event);
        }
    }

    /**
     * Retrieves all subscriptions for the specified event
     * 
     * @param event The event
     * @return A list with all subscriptions for the event or an empty list if none found
     */
    private List<Subscription> getSubscriptions(DriveEvent event) {
        try {
            return subscriptionStore.getSubscriptions(event.getContextID(), SERVICE_IDS, event.getFolderIDs());
        } catch (OXException e) {
            LOG.error("unable to get subscriptions for service {}", SERVICE_ID, e);
            return Collections.emptyList();
        }
    }

    /**
     * Processes the specified subscription
     * 
     * @param subscription The subscription
     * @param event The event
     */
    private void processSubscription(Subscription subscription, DriveEvent event) {
        String pushTokenReference = event.getPushTokenReference();
        if (null != pushTokenReference && subscription.matches(pushTokenReference)) {
            LOG.debug("Skipping push notification for subscription: {}", subscription);
            return;
        }
        try {
            FirebaseMessaging client = createOrGet(subscription.getContextID(), subscription.getUserID());
            if (client == null) {
                LOG.debug("No API key available for push via FCM for subscription {}, skipping notification.", subscription);
                return;
            }
            Message message = createMessage(event, subscription);
            String messageId = client.send(message);
            LOG.debug("Successfully sent push message with id: '{}'", messageId != null ? messageId : "");
        } catch (FirebaseMessagingException e) {
            LOG.warn("Error publishing drive event", e);
            handleException(subscriptionStore, event.getContextID(), subscription.getToken(), e);
        }
    }

    /**
     * Removes the registration with the specified registration id
     * 
     * @param subscriptionStore The subscription store
     * @param contextID the context identifier
     * @param registrationID the registration identifier
     */
    private static void removeRegistrations(DriveSubscriptionStore subscriptionStore, int contextID, String registrationID) {
        try {
            if (0 < subscriptionStore.removeSubscriptions(contextID, SERVICE_ID, registrationID)) {
                LOG.info("Successfully removed registration ID '{}'.", registrationID);
            } else {
                LOG.warn("Registration ID '{}' not removed.", registrationID);
            }
        } catch (OXException e) {
            LOG.error("Error removing registrations", e);
        }
    }

    /**
     * Creates the message to send
     * 
     * @param event The drive event
     * @param subscription The subscription
     * @return The message
     */
    private static Message createMessage(DriveEvent event, Subscription subscription) {
        AndroidConfig androidConfig = AndroidConfig.builder().setCollapseKey("TRIGGER_SYNC").build();
        //@formatter:off
        Message.Builder builder = Message.builder().setToken(subscription.getToken()).
                                    setAndroidConfig(androidConfig).
                                    putData("root", subscription.getRootFolderID()).
                                    putData("action", "sync");
        //@formatter:on

        if (!event.isContentChangesOnly() || !SubscriptionMode.SEPARATE.equals(subscription.getMode())) {
            return builder.build();
        }

        IntStream.range(0, event.getContentChanges().size()).filter(index -> {
            DriveContentChange contentChange = event.getContentChanges().get(index);
            return contentChange.isSubfolderOf(subscription.getRootFolderID());
        }).forEach(index -> {
            DriveContentChange contentChange = event.getContentChanges().get(index);
            builder.putData("path_" + index, contentChange.getPath(subscription.getRootFolderID()));
        });
        return builder.build();
    }

    /**
     * Creates or gets the firebase messaging instance
     * 
     * @param contextId The context identifier
     * @param userId The user identifier
     * @return The firebase messaging instance
     * @throws OXException if the configuration is invalid for the specified user in the specified context
     */
    private FirebaseMessaging createOrGet(int contextId, int userId) {
        if (!configService.getBooleanProperty(userId, contextId, DriveEventsFCMProperty.ENABLED)) {
            LOG.debug("Push via FCM is disabled for user {} in context {}.", I(userId), I(contextId));
            return null;
        }
        String name = configService.getProperty(userId, contextId, DriveEventsFCMProperty.NAME);
        String keyPath = configService.getProperty(userId, contextId, DriveEventsFCMProperty.KEYPATH);
        if (Strings.isNotEmpty(name) && Strings.isNotEmpty(keyPath)) {
            return createOrGetApp(name, keyPath, userId, contextId);
        }

        FCMKeyProvider keyProvider = services.getOptionalService(FCMKeyProvider.class);
        if (null == keyProvider) {
            LOG.debug("No API key available for push via FCM for user {} in context {}.", I(userId), I(contextId));
            return null;
        }
        LOG.debug("Using fallback API key for push via FCM for user {} in context {}.", I(userId), I(contextId));
        return createOrGetApp(keyProvider.getOptions(), keyProvider.getName());
    }

    /**
     * Creates or gets the app with the specified name and for the specified key path
     * 
     * @param name The app's name
     * @param keyPath The key path
     * @param userId the user identifier
     * @param contextId The context identifier
     * @return
     */
    private FirebaseMessaging createOrGetApp(String name, String keyPath, int userId, int contextId) {
        try (FileInputStream keyStream = new FileInputStream(keyPath)) {
            GoogleCredentials googleCredentials = GoogleCredentials.fromStream(keyStream).createScoped(Arrays.asList(SCOPES));
            FirebaseOptions options = FirebaseOptions.builder().setCredentials(googleCredentials).build();
            return createOrGetApp(options, name);
        } catch (IOException e) {
            LOG.debug("Unable to load FCM key for app {} with path {} for user {} in context {}", name, keyPath, I(userId), I(contextId), e);
            return null;
        }
    }

    /**
     * Creates the messaging app. If another app with the same name already exists,
     * it will be deleted and re-initialised
     * 
     * @param options The firebase options
     * @param name The app's name
     * @return The firebase messaging app
     */
    private FirebaseMessaging createOrGetApp(FirebaseOptions options, String name) {
        try {
            FirebaseApp firebaseApp = FirebaseApp.initializeApp(options, name);
            return FirebaseMessaging.getInstance(firebaseApp);
        } catch (IllegalStateException e) {
            LOG.debug("", e);
            FirebaseApp.getInstance(name).delete();
            FirebaseApp app = FirebaseApp.initializeApp(options, name);
            return FirebaseMessaging.getInstance(app);
        }
    }

    /**
     * Handles the specified {@link FirebaseMessagingException}
     * 
     * @param subscriptionStore The subscription store
     * @param contextID The context identifier
     * @param token The registration id, i.e. the token
     * @param e the exception
     */
    private void handleException(DriveSubscriptionStore subscriptionStore, int contextID, String token, FirebaseMessagingException e) {
        switch (e.getErrorCode()) {
            case ABORTED:
                LOG.warn("Push message could not be sent because the request was aborted.");
                return;
            case ALREADY_EXISTS:
                LOG.warn("Push message could not be sent because the resource the client tried to created already exists.");
                return;
            case CANCELLED:
                LOG.warn("Push message could not be sent because the request was cancelled.");
                return;
            case CONFLICT:
                LOG.warn("Push message could not be sent because there was a concurrency conflict.");
                return;
            case DATA_LOSS:
                LOG.warn("Push message could not be sent because there was an unrecoverable data loss or data corruption.");
                return;
            case DEADLINE_EXCEEDED:
                LOG.warn("Push message could not be sent because the request's deadline was exceeded.");
                return;
            case FAILED_PRECONDITION:
                LOG.warn("Push message could not be sent because a precondition failed.");
                return;
            case INTERNAL:
                LOG.warn("Push message could not be sent because an internal error was occurred in the FCM servers.");
                return;
            case INVALID_ARGUMENT:
                LOG.warn("Push message could not be sent because an invalid argument was provided.");
                return;
            case NOT_FOUND:
                LOG.warn("Received error '{}' when sending push message to '{}', removing registration ID.", e.getErrorCode(), token);
                removeRegistrations(subscriptionStore, contextID, token);
                return;
            case OUT_OF_RANGE:
                LOG.warn("Push message could not be sent because an invalid range was specified.");
                return;
            case PERMISSION_DENIED:
                LOG.warn("Push message could not be sent due to insufficient permissions. This can happen because the OAuth token does not have the right scopes, the client doesn't have permission, or the API has not been enabled for the client project.");
                return;
            case RESOURCE_EXHAUSTED:
                LOG.warn("Push message could not be sent due to the resource either being out of quota or rate limited.");
                return;
            case UNAUTHENTICATED:
                LOG.warn("Push message could not be sent because the request was not authenticated due to missing, invalid or expired OAuth token.");
                return;
            case UNAVAILABLE:
                LOG.warn("Push message could not be sent because the FCM servers were not available.");
                return;
            case UNKNOWN:
                LOG.warn("Push message could not be sent because an unknown error was occurred in the FCM servers.");
                return;
            default:
                LOG.warn("", e);
        }
    }

}
