/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.gdpr.dataexport.impl.utils;

import static com.openexchange.groupware.upload.impl.UploadUtility.getSize;
import static com.openexchange.java.Autoboxing.I;
import static com.openexchange.java.Autoboxing.isNot;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.LockSupport;
import org.xml.sax.SAXParseException;
import com.openexchange.ajax.container.TmpFileFileHolder;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.exception.OXException;
import com.openexchange.filestore.FileStorage;
import com.openexchange.java.Streams;

/**
 * {@link AppendingFileStorageOutputStream} - The output stream appending subsequent data to a file storage destination.
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since v7.10.3
 */
public class AppendingFileStorageOutputStream extends OutputStream {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(AppendingFileStorageOutputStream.class);

    /** The default in-memory threshold of approximately 5 MB (left some head room for transferred chunk size of storages). */
    public static final int DEFAULT_IN_MEMORY_THRESHOLD = 5242880 - 1024; // ~5 MB

    private final FileStorage fileStorage;
    private final AtomicReference<File> tmpFileReference;
    private final long fileStorageChunkSize;
    private final boolean useTmpFile;
    private final byte[] buf;
    private String fileStorageLocation;
    private int bufpos;
    private long bytesWritten;

    /**
     * Initializes a new instance with {@link #DEFAULT_IN_MEMORY_THRESHOLD default buffer size}.
     *
     * @param fileStorage The file storage to write to
     * @see #AppendingFileStorageOutputStream(int, FileStorage)
     */
    public AppendingFileStorageOutputStream(FileStorage fileStorage) {
        this(DEFAULT_IN_MEMORY_THRESHOLD, fileStorage);
    }

    /**
     * Initializes a new instance.
     *
     * @param size The size in bytes for the in-memory buffer
     * @param fileStorage The file storage to write to
     */
    public AppendingFileStorageOutputStream(int size, FileStorage fileStorage) {
        super();
        if (size <= 0) {
            throw new IllegalArgumentException("Buffer size <= 0");
        }
        this.fileStorage = fileStorage;
        tmpFileReference = new AtomicReference<File>();
        long chunkSize = fileStorage.getPreferredChunkSize();
        fileStorageChunkSize = chunkSize > 0L && chunkSize > size ? chunkSize : 0L;
        useTmpFile = fileStorageChunkSize > 0;
        buf = new byte[size];
        bufpos = 0;
        bytesWritten = 0L;
        fileStorageLocation = null;
    }

    /**
     * Gets (or creates if absent) the temporary file to use for buffering data for file storage.
     *
     * @return The temporary file
     * @throws IOException If creating the temporary file fails
     */
    private File getOrCreateTmpFile() throws IOException {
        File tmpFile = tmpFileReference.get();
        if (tmpFile == null) {
            try {
                File newTmpFile = TmpFileFileHolder.newTempFile("open-xchange-tmpexportbuffer-", false);
                tmpFile = newTmpFile;
                tmpFileReference.set(tmpFile);
            } catch (OXException e) {
                IOException ioe = ExceptionUtils.extractFrom(e, IOException.class);
                if (ioe != null) {
                    throw ioe;
                }
                throw new IOException(e);
            }
        }
        return tmpFile;
    }

    /**
     * Flushes the internal buffer to either file storage or temporary file.
     *
     * @throws IOException If flushing buffer fails
     */
    private void flushBuffer() throws IOException {
        if (bufpos <= 0) {
            // No bytes in buffer
            return;
        }

        if (isNot(useTmpFile)) {
            // Use no temporary file. Flush buffer's content to file storage directly.
            transferToFileStorage(() -> Streams.newByteArrayInputStream(buf, 0, bufpos), bufpos);
            bufpos = 0;
            return;
        }

        // Use temporary file to buffer data prior to transferring to file storage
        File tmpFile = getOrCreateTmpFile();
        long fileLength = tmpFile.length();
        if (fileLength + bufpos > fileStorageChunkSize) {
            // Appending current buffer content to file would exceed file storage's chunk size, therefore:
            // Transfer temporary file to file storage
            if (fileLength > 0) { // NOSONARLINT
                transferToFileStorage(() -> new FileInputStream(tmpFile), fileLength);
                clearFileContent(tmpFile);
            }
        }

        // Append buffer content to temporary file
        FileOutputStream fos = new FileOutputStream(tmpFile, true);
        try {
            fos.write(buf, 0, bufpos);
            fos.flush();
            bufpos = 0;
        } finally {
            Streams.close(fos);
        }
    }

    /**
     * Clears the content of given file.
     *
     * @param file The file to clear
     * @throws IOException Of clearing file's content fails
     */
    private static void clearFileContent(File file) throws IOException {
        FileOutputStream fos = new FileOutputStream(file);
        try {
            fos.write(new byte[0], 0, 0);
            fos.flush();
        } finally {
            Streams.close(fos);
        }
    }

    /**
     * Transfers given stream's data to file storage location
     *
     * @param provider The provider for the stream to transfer to file storage
     * @param numberOfBytes The number of bytes provided by given stream
     * @throws IOException If an I/O error occurs
     */
    private void transferToFileStorage(InputStreamProvider provider, long numberOfBytes) throws IOException {
        if (numberOfBytes <= 0) {
            // Nothing to write
            return;
        }

        boolean retry = true;
        int retryCount = 0;
        int maxRetries = 5;
        do {
            InputStream in = provider.newInputStream();
            try {
                boolean newFile = fileStorageLocation == null;
                if (newFile) {
                    fileStorageLocation = fileStorage.saveNewFile(in);
                    bytesWritten = numberOfBytes;
                    LOG.debug("Flushed buffer ({}) to new file {} during data export after {} attempts.", getSize(numberOfBytes, 2, false, true), fileStorageLocation, I(retryCount + 1));
                } else {
                    fileStorage.appendToFile(in, fileStorageLocation, bytesWritten);
                    bytesWritten += numberOfBytes;
                    LOG.debug("Flushed buffer ({}) to existent file {} during data export after {} attempts.", getSize(numberOfBytes, 2, false, true), fileStorageLocation, I(retryCount + 1));
                }
                retry = false;
            } catch (Exception e) {
                Streams.close(in);
                in = null;

                if (retryCount++ >= maxRetries || ExceptionUtils.isNoneOf(e, IOException.class, SAXParseException.class)) {
                    if (fileStorageLocation == null) {
                        LOG.error("Failed flushing buffer ({}) to new file during data export after {} attempts.", getSize(numberOfBytes, 2, false, true), I(retryCount), e);
                    } else {
                        LOG.error("Failed flushing buffer ({}) to existent file {} during data export after {} attempts.", getSize(numberOfBytes, 2, false, true), fileStorageLocation, I(retryCount), e);
                    }
                    IOException ioe = ExceptionUtils.extractFrom(e, IOException.class);
                    throw ioe != null ? ioe : new IOException(e);
                }

                // A timeout while connecting to an HTTP server or waiting for an available connection from an HttpConnectionManager
                if (fileStorageLocation == null) {
                    LOG.info("Could not flush buffer ({}) to new file during data export on {}. attempt due to a retry-able error (\"{}\"). Retrying...", getSize(numberOfBytes, 2, false, true), I(retryCount), e.getMessage());
                } else {
                    LOG.info("Could not flush buffer ({}) to existent file {} during data export on {}. attempt due to a retry-able error (\"{}\"). Retrying...", getSize(numberOfBytes, 2, false, true), fileStorageLocation, I(retryCount), e.getMessage());
                }

                // Retry using exponential back-off...
                exponentialBackoffWait(retryCount, 1000L);
            } finally {
                Streams.close(in);
            }
        } while (retry);
    }

    /**
     * Performs a wait according to exponential back-off strategy.
     * <pre>
     * (retry-count * base-millis) + random-millis
     * </pre>
     *
     * @param retryCount The current number of retries
     * @param baseMillis The base milliseconds
     */
    private static void exponentialBackoffWait(int retryCount, long baseMillis) {
        long nanosToWait = TimeUnit.NANOSECONDS.convert((retryCount * baseMillis) + ((long) (Math.random() * baseMillis)), TimeUnit.MILLISECONDS);
        LockSupport.parkNanos(nanosToWait);
    }

    /**
     * Gets the file storage location
     *
     * @return The file storage location
     */
    public synchronized Optional<String> getFileStorageLocation() {
        return Optional.ofNullable(fileStorageLocation);
    }

    @Override
    public synchronized void write(int b) throws IOException {
        if (bufpos >= buf.length) {
            flushBuffer();
        }
        buf[bufpos++] = (byte)b;
    }

    @Override
    public synchronized void write(byte[] b, int off, int len) throws IOException {
        int toTransfer = len;
        int offset = off;
        while (toTransfer > 0) {
            int remaining = buf.length - bufpos;
            if (remaining <= 0) {
                // No space left in buffer
                flushBuffer();
            } else {
                if (remaining <= toTransfer) {
                    // Fill complete buffer & flush it
                    transferBytes2Buffer(b, offset, remaining);
                    offset += remaining;
                    toTransfer -= remaining;
                    flushBuffer();
                } else {
                    // Fill partial buffer
                    transferBytes2Buffer(b, offset, toTransfer);
                    toTransfer = 0;
                }
            }
        }
    }

    private void transferBytes2Buffer(byte[] b, int off, int len) {
        System.arraycopy(b, off, buf, bufpos, len);
        bufpos += len;
    }

    @Override
    public synchronized void flush() throws IOException {
        flushBuffer();
        flushAndDeleteTmpFile();
    }

    @Override
    public synchronized void close() throws IOException {
        flushBuffer();
        flushAndDeleteTmpFile();
        super.close();
    }

    /**
     * Flushes the content of the temporary file (if any) to file storage.
     *
     * @throws IOException If an I/O error occurs
     */
    private void flushAndDeleteTmpFile() throws IOException {
        File tmpFile = tmpFileReference.getAndSet(null);
        if (tmpFile != null) {
            // Transfer temporary file to file storage
            long numberOfBytes = tmpFile.length();
            if (numberOfBytes > 0) {
                transferToFileStorage(() -> new FileInputStream(tmpFile), numberOfBytes);
            }

            // Delete temporary file
            deleteFile(tmpFile);
        }
    }

    /**
     * (safely) deletes given file from disk.
     *
     * @param file The file to delete
     */
    private static void deleteFile(File file) {
        try {
            Files.delete(file.toPath());
        } catch (java.nio.file.NoSuchFileException e) {
            LOG.debug("Temporary file {} already deleted", file, e);
        } catch (Exception e) {
            LOG.error("Failed to delete temporary file {}", file, e);
        }
    }

    // -------------------------------------------------------------------------------------------------------------------------------------

    /**
     * Provides an input stream.
     */
    @FunctionalInterface
    private static interface InputStreamProvider {

        /**
         * Creates a new input stream.
         *
         * @return The input stream
         * @throws IOException If input stream cannot be created due to an I/O error
         */
        InputStream newInputStream() throws IOException;
    }

}
