/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.activator.impl.RegistrationTools.registerIntoContext;
import static com.openexchange.util.activator.impl.RegistrationTools.serviceListings;
import static com.openexchange.util.activator.impl.RegistrationTools.serviceSets;
import static com.openexchange.util.activator.impl.RegistrationTools.trackAndOpen;
import static com.openexchange.util.activator.impl.RegistrationTools.whatIsProvidedBy;
import static com.openexchange.util.custom.base.NullUtil.emptyList;
import static com.openexchange.util.custom.base.NullUtil.emptyMap;
import static com.openexchange.util.custom.base.NullUtil.notNull;
import java.util.Dictionary;
import javax.annotation.concurrent.Immutable;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.util.activator.ServiceDependencyResolver;
import com.openexchange.util.activator.ServiceRegistrationHandler;
import com.openexchange.util.custom.base.NullUtil;

/**
 * {@link ServiceWithDependenciesTemplate}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.2
 * @param <S>
 * @param <T>
 */
@Immutable
public abstract class ServiceWithDependenciesTemplate<S, T extends S> implements ServiceWithDependencies<T> {
    
    protected final Class<T> serviceInstanceClass;
    protected final Optional<Class<S>> optServiceClass;
    protected final ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies;
    protected final ImmutableSet<OptionalServiceDependency<?>> optionalDependencies;
    protected final ImmutableSet<Class<?>> serviceSets;
    protected final ImmutableSet<Class<?>> serviceListings;

    protected ServiceWithDependenciesTemplate(final Class<T> serviceInstanceClass,
        @Nullable Class<S> serviceClass,
        ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies,
        ImmutableSet<OptionalServiceDependency<?>> optionalDependencies,
        ImmutableSet<Class<?>> serviceSets,
        ImmutableSet<Class<?>> serviceListings) {
        this.serviceInstanceClass = serviceInstanceClass;
        {
            this.optServiceClass = NullUtil.optionalFromNullable(serviceClass);
        }
        this.mandatoryDependencies = mandatoryDependencies;
        this.optionalDependencies = optionalDependencies;
        this.serviceSets = serviceSets;
        this.serviceListings = serviceListings;
    }
    
    @Override
    public final void registerService(final BundleContext bundleContext, final ActivatorRegistry registry,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        
        if (mandatoryDependencies.isEmpty() && optionalDependencies.isEmpty()) {
            final ServiceDependencyResolver resolver;
            {
                final ImmutableMap<Class<?>, ServiceSet<?>> serviceSetMap = serviceSets(bundleContext, registry, serviceSets);
                final ImmutableMap<Class<?>, ServiceListing<?>> serviceListingMap = serviceListings(bundleContext, registry, serviceListings);
                resolver = new ServiceDependencyResolverImpl(
                    emptyMap(),
                    emptyList(),
                    serviceSetMap,
                    serviceListingMap,
                    properties
                );
            }
            final T serviceInstance = createInstance(resolver);
            final ImmutableSet<Class<?>> serviceClasses = whatIsProvidedBy(serviceInstance.getClass(), optServiceClass.orNull());
            
            final ServiceRegistration<S> r = registerIntoContext(bundleContext, serviceClasses, serviceInstance, properties);
            registry.add(new SimpleServiceRegistration(serviceInstance, r));
            if (serviceRegistrationHandler != null) {
                serviceRegistrationHandler.onServiceRegistered(serviceInstance);
            }
        } else {
            final ImmutableSet<Class<?>> serviceClasses = whatIsProvidedBy(serviceInstanceClass, optServiceClass.orNull());
            final ServiceTrackerCustomizer<S, T> registerer = createRegisterer(
                bundleContext,
                registry,
                serviceClasses,
                properties,
                serviceRegistrationHandler
            );
            final ServiceTracker<S, T> serviceTracker = trackAndOpen(
                bundleContext,
                mandatoryDependencies,
                optionalDependencies,
                registerer,
                notNull(serviceInstanceClass.getName())
            );
            
            registry.add(new DependentServiceRegistration(serviceTracker, notNull(serviceInstanceClass.getName())));
        }
    }
    
    protected ServiceTrackerCustomizer<S, T> createRegisterer(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final ImmutableSet<Class<?>> serviceClasses,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        
        return new ServiceWithDependenciesServiceRegisterer<>(bundleContext,
            registry,
            this,
            serviceClasses,
            notNull(serviceInstanceClass.getName()),
            properties,
            serviceRegistrationHandler
        );
    }

    @Override
    public final ImmutableSet<MandatoryServiceDependency<?>> mandatoryServiceDependencies() {
        return this.mandatoryDependencies;
    }
    
    @Override
    public ImmutableSet<OptionalServiceDependency<?>> optionalServiceDependencies() {
        return this.optionalDependencies;
    }

    @Override
    public final ImmutableSet<Class<?>> serviceSetClasses() {
        return this.serviceSets;
    }

    @Override
    public final ImmutableSet<Class<?>> serviceListingClasses() {
        return this.serviceListings;
    }
    
    @Override
    public final boolean isEmpty() {
        return this.mandatoryDependencies.isEmpty() && this.optionalDependencies.isEmpty()
            && this.serviceSets.isEmpty() && this.serviceListings.isEmpty();
    }

}