/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.activator.impl;

import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.util.activator.ServiceNotAvailableException;
import com.openexchange.util.activator.ServiceRef;

public final class AtomicServiceRef<E> implements ServiceRef<E>, UpdatableServiceRef<E> {
    
    public static <X> AtomicServiceRef<X> forClass(final Class<X> theClass) {
        return new AtomicServiceRef<X>(theClass);
    }

    private final AtomicReference<E> ref;
    private final Class<E> refClass;
    
    public AtomicServiceRef(final Class<E> refClass) {
        this.ref = new AtomicReference<>(null);
        this.refClass = refClass;
    }

    @Override
    public boolean isPresent() {
        final E e = ref.get();
        return e != null;
    }

    @Override
    public E get() throws ServiceNotAvailableException {
        final E e = ref.get();
        if (e != null) {
            return e;
        } else {
            throw new ServiceNotAvailableException(refClass);
        }
    }

    @Override
    public @Nullable E tryGet() {
        return ref.get();
    }

    @Override
    public Set<E> asSet() {
        final E e = ref.get();
        if (e != null) {
            @SuppressWarnings("null")
            final @NonNull Set<E> result = ImmutableSet.<E>of(e);
            return result;
        } else {
            @SuppressWarnings("null")
            final @NonNull Set<E> result = ImmutableSet.<E>of();
            return result;
        }
    }
    
    @Override
    public Optional<E> asOptional() {
        @SuppressWarnings("null")
        final @NonNull Optional<E> result = Optional.fromNullable(ref.get());
        return result;
    }

    @Override
    public void set(final E service) {
        ref.set(service);
    }

    @Override
    public void unset() {
        ref.set(null);
    }
    
    @Override
    public Class<E> serviceRefClass() {
        return refClass;
    }

}
