/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.database.ox;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.context.ContextService;
import com.openexchange.database.DatabaseService;
import com.openexchange.exception.OXException;
import com.openexchange.usm.api.database.DatabaseAccessWithNoTimeoutSupport;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.util.SQLToolkit;
import com.openexchange.usm.util.Toolkit;

public class OXDBService implements DatabaseAccessWithNoTimeoutSupport {

    private static final String UNHEX = "UNHEX";

    private static final String HEX = "HEX";

    private final DatabaseService _delegate;

    private static final Log LOG = LogFactory.getLog(OXDBService.class);

    private final ContextService _contextService;

    /**
     * Initializes a new {@link OXDBService}.
     *
     * @param contextService
     * @param databaseService
     */
    public OXDBService(ContextService contextService, DatabaseService databaseService) {
        _contextService = contextService;
        _delegate = databaseService;
        LOG.info("USM Connection to OX DB service activated");
    }

    @Override
    public void backReadOnly(int contextID, Connection connection) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Releasing read only JDBC Connection for " + contextID);
        }
        _delegate.backReadOnly(contextID, connection);
    }

    @Override
    public void backWritable(int contextID, Connection connection) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Releasing read/write JDBC Connection for " + contextID);
        }
        _delegate.backWritable(contextID, connection);
    }

    @Override
    public void backNoTimeout(int contextID, Connection connection) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Releasing no-timeout JDBC Connection for " + contextID);
        }
        _delegate.backForUpdateTask(contextID, connection);
    }

    @Override
    public EncapsulatedConnection getReadOnly(int contextID) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrieving read only JDBC Connection for " + contextID);
        }
        try {
            return new EncapsulatedConnection(this, _delegate.getReadOnly(contextID), contextID, true);
        } catch (OXException e) {
            String errorMessage = "Couldn't get read only connection for " + contextID;
            LOG.error(errorMessage, e);
            throw new StorageAccessException(USMDatabaseOXErrorCodes.GET_READ_ONLY_ERROR, errorMessage, e);
        }
    }

    @Override
    public EncapsulatedConnection getWritable(int contextID) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrieving read/write JDBC Connection for " + contextID);
        }
        try {
            return new EncapsulatedConnection(this, _delegate.getWritable(contextID), contextID, false);
        } catch (OXException e) {
            String errorMessage = "Couldn't get read/write connection for " + contextID;
            LOG.error(errorMessage, e);
            throw new StorageAccessException(USMDatabaseOXErrorCodes.GET_READ_WRITE_ERROR, errorMessage, e);
        }
    }

    @Override
    public EncapsulatedConnection getNoTimeout(int contextID) throws StorageAccessException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrieving no-timeout JDBC Connection for " + contextID);
        }
        try {
            return new EncapsulatedConnection(this, _delegate.getForUpdateTask(contextID), contextID, EncapsulatedConnection.Nature.NO_TIMEOUT);
        } catch (OXException e) {
            String errorMessage = "Couldn't get no-timeout connection for " + contextID;
            LOG.error(errorMessage, e);
            throw new StorageAccessException(USMDatabaseOXErrorCodes.GET_READ_WRITE_ERROR, errorMessage, e);
        }
    }

    @Override
    public void generateSequence(int contextID, String sequenceName) throws USMStorageException, StorageAccessException {
        EncapsulatedConnection con = getWritable(contextID);
        Statement stmt = null;
        try {
            try {
                stmt = con.createStatement();
                con.setAutoCommit(false);
                stmt.execute(SQLToolkit.createSequenceStatement(sequenceName));
                stmt.execute(SQLToolkit.createSequenceProcedure(sequenceName));
                con.commit();
                LOG.info("Creation of sequence " + sequenceName + " successfully performed");
            } catch (SQLException e) {
                con.rollback();
                throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR, e.getMessage(), e);
            } finally {
                con.setAutoCommit(true);
                Toolkit.close(stmt);
                Toolkit.close(con);
            }
        } catch (SQLException e) {
            throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR_AUTOCOMMIT_SETTINGS, e.getMessage(), e);
        }

    }

    @Override
    public int getNextIdFromSequence(int contextID, String sequenceName) throws USMStorageException, StorageAccessException {
        EncapsulatedConnection con = getWritable(contextID);
        try {
            return SQLToolkit.getNextIdFromSequence(con, sequenceName, contextID);
        } catch (SQLException e) {
            throw new USMStorageException(USMDatabaseOXErrorCodes.GET_NEXT_FROM_SEQUENCE_ERROR, e.getMessage(), e);
        } finally {
            Toolkit.close(con);
        }
    }

    @Override
    public String retrieveHexFunctionName() {
        return HEX;
    }

    @Override
    public String retrieveUnHexFunctionName() {
        return UNHEX;
    }

    @Override
    public List<Integer> getAllContextIds() throws StorageAccessException {
        try {
            return _contextService.getAllContextIds();
        } catch (OXException e) {
            throw new StorageAccessException(USMDatabaseOXErrorCodes.GET_ALL_CONTEXT_IDS_ERROR, "Error while retrieving all context ids", e);
        }
    }

    @Override
    public int[] getContextsInSameSchema(int contextId) throws StorageAccessException {
        try {
            return _delegate.getContextsInSameSchema(contextId);
        } catch (OXException e) {
            throw new StorageAccessException(
                USMDatabaseOXErrorCodes.GET_CONTEXTS_IN_SAME_SCHEMA_ERROR,
                "Error while retrieving all context ids in same schema as " + contextId,
                e);
        }
    }

    @Override
    public boolean supportsOnDuplicateKeyUpdateStatement() {
        return true;
    }

    @Override
    public boolean supportsInsertIgnoreStatement() {
        return true;
    }
}
