/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.contenttypes.resources.impl;

import java.util.BitSet;
import java.util.HashMap;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.folder.OXFolderContent;
import com.openexchange.usm.api.contenttypes.resource.ResourcesConstants;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.UnsupportedContentOperationException;
import com.openexchange.usm.api.ox.json.JSONResult;
import com.openexchange.usm.api.ox.json.JSONResultType;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.contenttypes.util.AbstractTransferHandler;
import com.openexchange.usm.contenttypes.util.USMContentTypesUtilErrorCodes;

public class ResourcesContentTypeTransferHandler extends AbstractTransferHandler {

    public ResourcesContentTypeTransferHandler(ContentType contentType, OXJSONAccess ajaxAccess) {
        super(contentType, ajaxAccess);
    }

    @Override
    protected String getOXAjaxAccessPath() {
        return ResourcesConstants.RESOURCES_PATH;
    }

    @Override
    public void writeNewDataObject(DataObject object) throws USMException {
        if (LOG.isErrorEnabled())
            LOG.error(object.getSession() + " Creating a resource is not supported");
        throw new UnsupportedContentOperationException(
            USMContentTypesResourcesErrorCodes.CREATE_ERROR,
            "Creating a resource is not supported");
    }

    @Override
    public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
        if (LOG.isErrorEnabled())
            LOG.error(object.getSession() + " Updating a resource is not supported");
        throw new UnsupportedContentOperationException(
            USMContentTypesResourcesErrorCodes.UPDATE_ERROR,
            "Updating a resource is not supported");
    }

    @Override
    public void writeDeletedDataObject(DataObject object) throws USMException {
        if (LOG.isErrorEnabled())
            LOG.error(object.getSession() + " Deleting a resource is not supported");
        throw new UnsupportedContentOperationException(
            USMContentTypesResourcesErrorCodes.DELETE_ERROR,
            "Deleting a resource is not supported");
    }

    @Override
    public DataObject[] readUpdatedFolderContent(Folder folder, BitSet requestedFields, long timestamp) throws USMException {
        if (LOG.isErrorEnabled())
            LOG.error(folder.getSession() + " Updating a folder content is not supported for resources");
        throw new UnsupportedContentOperationException(
            USMContentTypesResourcesErrorCodes.UPDATE_FOLDER_ERROR,
            "Updating a folder content is not supported for resources");
    }

    @Override
    public OXFolderContent readFolderContent(Folder folder, int limit, DataObjectFilter filter) throws USMException {
        return new OXFolderContent(true, readFolderContent(folder, folder.getSession().getFieldFilter(folder.getElementsContentType())));
    }

    @Override
    public DataObject[] readFolderContent(Folder folder, BitSet requestedFields) throws USMException {
        Session session = folder.getSession();
        if (LOG.isDebugEnabled())
            LOG.debug(session + " Read all " + _contentType.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), CommonConstants.ACTION_ALL, session, parameters);
        checkResult(result, JSONResultType.JSONObject);
        JSONArray identifiersArray = extractArrayResult(result);
        JSONArray requestBody = new JSONArray();
        int length = identifiersArray.length();
        for (int i = 0; i < length; i++) {
            JSONObject obj = new JSONObject();
            try {
                obj.put(CommonConstants.ID, identifiersArray.getString(i));
            } catch (JSONException e) {
                throw new InternalUSMException(USMContentTypesResourcesErrorCodes.CREATE_REQUEST_ERROR, "Error on creating request body", e);
            }
            requestBody.put(obj);
        }
        JSONResult result2 = _ajaxAccess.doPut(getOXAjaxAccessPath(), CommonConstants.ACTION_LIST, session, parameters, requestBody);
        checkResult(result2, JSONResultType.JSONObject);
        JSONArray resourcesArray = extractArrayResult(result2);
        long timestamp = result.getJSONObject().optLong(CommonConstants.TIMESTAMP, System.currentTimeMillis());
        int length2 = resourcesArray.length();
        DataObject[] resultArray = new DataObject[length2];
        try {
            for (int i = 0; i < length2; i++) {
                JSONObject jsonObj = resourcesArray.getJSONObject(i);
                resultArray[i] = _contentType.newDataObject(session);
                resultArray[i].setTimestamp(timestamp);
                ContentTypeField[] fields = _contentType.getFields();
                for (int j = 0; j < fields.length; j++) {
                    ContentTypeField field = fields[j];
                    String fieldName = field.getFieldName();
                    resultArray[i].setFieldContent(fieldName, field.getFieldType().extractFromJSONObject(session, jsonObj, fieldName));
                }
            }
        } catch (JSONException e) {
            throw new InternalUSMException(USMContentTypesResourcesErrorCodes.READ_RESULT_ERROR, "Error on reading result", e);
        }
        return resultArray;
    }

    public DataObject[] getAllResources(Session session, BitSet requestedFields) throws AuthenticationFailedException, OXCommunicationException, UnsupportedContentOperationException, InternalUSMException {
        if (LOG.isDebugEnabled())
            LOG.debug(session + " Read all resources ");
        Map<String, String> parameters = new HashMap<String, String>();
        //parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
        JSONResult result = _ajaxAccess.doGet(ResourcesConstants.RESOURCES_PATH, CommonConstants.ACTION_ALL, session, parameters);
        checkResult(result, JSONResultType.JSONObject);
        JSONArray jsonar = null;
        JSONObject jsonObject = result.getJSONObject();
        try {
            jsonar = jsonObject.getJSONArray(CommonConstants.RESULT_DATA);
            JSONArray resources = new JSONArray();
            int length = jsonar.length();
            for (int j = 0; j < length; j++) {
                JSONObject resourceId = new JSONObject();
                resourceId.put(CommonConstants.ID, jsonar.get(j));
                resources.put(resourceId);
            }
            result = _ajaxAccess.doPut(ResourcesConstants.RESOURCES_PATH, CommonConstants.ACTION_LIST, session, parameters, resources);
            return getListOfResults(session, requestedFields, result);
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER7,
                null,
                "getListOfResults: data array not present",
                e);
        }
    }

    private DataObject[] getListOfResults(Session session, BitSet requestedFields, JSONResult result) throws OXCommunicationException, InternalUSMException {
        JSONArray jsonar = null;
        long timestamp = 0L;
        checkResult(result, JSONResultType.JSONObject);
        JSONObject jsonObject = result.getJSONObject();
        try {
            jsonar = jsonObject.getJSONArray(CommonConstants.RESULT_DATA);
        } catch (JSONException e) {
            throw generateException(
                USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER7,
                null,
                "getListOfResults: data array not present",
                e);
        }
        int length = jsonar.length();
        if (length == 0)
            return SyncResult.EMPTY_DATA_OBJECT_ARRAY;
        try {
            if (jsonObject.has(CommonConstants.RESULT_TIMESTAMP))
                timestamp = jsonObject.getLong(CommonConstants.RESULT_TIMESTAMP);
        } catch (JSONException e) {
            throw generateException(USMContentTypesUtilErrorCodes.INVALID_TIMESTAMP_NUMBER3, null, "getListOfResults: invalid timestamp", e);
        }
        DataObject[] dataObjects = new DataObject[length];
        for (int j = 0; j < length; j++) {
            // get array which contains the information specified by the corresponding identifiers in the columns parameter
            try {
                JSONObject resourceObject = (JSONObject) jsonar.get(j);
                DataObject newDataObject = createDataObject(convertObject2Array(resourceObject, requestedFields), session, requestedFields, 0);
                dataObjects[j] = newDataObject;
                newDataObject.setTimestamp(timestamp);
            } catch (JSONException je) {
                throw generateException(
                    USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER6,
                    null,
                    "getListOfResults: error while reading index " + j + " of resources ",
                    je);
            }
        }
        return dataObjects;
    }

    private JSONArray convertObject2Array(JSONObject resourceObject, BitSet requestedFields) throws OXCommunicationException {
        JSONArray columnsArray = new JSONArray();
        ContentTypeField[] fields = _contentType.getFields();
        try {
            for (int i=0; i<fields.length; i++) {
                if (requestedFields.get(i)) {
                    columnsArray.put(resourceObject.getString(fields[i].getFieldName()));
                }
            }
        } catch (JSONException e) {
            throw generateException(USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER6, null, "read resources: value not found", e);
        }
        return columnsArray;
    }
    
    private DataObject createDataObject(JSONArray columnsArray, Session session, BitSet requestedFields, int idIndex) throws InternalUSMException, OXCommunicationException {
        String objId;
        try {
            objId = columnsArray.getString(idIndex);
        } catch (JSONException e) {
            throw generateException(USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER6, null, "read resources: no id for new object", e);
        }
        DataObject dataObj = _contentType.newDataObject(session);
        dataObj.setID(objId);
        updateDataObjectFromJSONArray(dataObj, requestedFields, columnsArray);
        return dataObj;
    }

    public void getResource(DataObject object, BitSet requestedFields) throws AuthenticationFailedException, OXCommunicationException, UnsupportedContentOperationException, InternalUSMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Read resource with id " + object.getID());
        checkTypeToWrite(object, _contentType.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.ID, object.getID());
        JSONResult result = _ajaxAccess.doGet(ResourcesConstants.RESOURCES_PATH, CommonConstants.ACTION_GET, object.getSession(), parameters);
        checkResult(result, JSONResultType.JSONObject);
        updateDataObjectFromJSONResult(object, requestedFields, result);
    }
}
