/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */



package com.openexchange.office.spellcheck.impl.client;

import java.util.concurrent.atomic.AtomicBoolean;
import com.openexchange.office.tools.annotation.NonNull;
import com.openexchange.office.tools.annotation.Nullable;

/**
 * {@link VersionValidator}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.4
 */
public class VersionValidator {

    final public  static int VERSION_INVALID = -1;

    /**
     * {@link GetVersionFunc}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.4
     */
    public interface GetVersionHandler {
        int getVersion();
    }

    /**
     * VersionChangedHandler
     */
    public interface VersionChangedHandler {
        void versionChanged(int oldVersion, int newVersion);
    }

    /**
     * Initializes a new {@link VersionValidator}.
     */
    public VersionValidator(
        @NonNull final GetVersionHandler getVersionHandler,
        @Nullable final VersionChangedHandler gotValidVersionHandler,
        @Nullable final VersionChangedHandler lostValidVersionHandler) {

        super();

        m_getVersionHandler = getVersionHandler;
        m_gotValidVersionHandler = gotValidVersionHandler;
        m_lostValidVersionHandler = lostValidVersionHandler;
    }

    /**
     * @param enabled
     * @return
     */
    public void setEnabled(final boolean enabled) {
        m_enabled.set(enabled);
    }

    /**
     * @param enabled
     * @return
     */
    public boolean isEnabled() {
        return m_enabled.get();
    }

    /**
     * @return
     */
    public int getVersion() {
        if (!isEnabled()) {
            return VERSION_INVALID;
        }

        synchronized (this) {
            final int oldVersion = m_version;

            if (VERSION_INVALID == oldVersion) {
                final int currentVersion = Math.max(m_getVersionHandler.getVersion(), VERSION_INVALID);

                if (currentVersion > VERSION_INVALID) {
                    m_version = currentVersion;

                    if (null != m_gotValidVersionHandler) {
                        m_gotValidVersionHandler.versionChanged(oldVersion, currentVersion);
                    }
                }
            }

            return m_version;
        }
    }

    /**
     *
     */
    synchronized public void setVersionInvalid(boolean notifyLostValidVersion) {
        final int oldVersion = m_version;

        if (oldVersion > VERSION_INVALID) {
            m_version = VERSION_INVALID;

            if (notifyLostValidVersion && (null != m_lostValidVersionHandler)) {
                m_lostValidVersionHandler.versionChanged(oldVersion, VERSION_INVALID);
            }
        }
    }

    /**
     *
     */
    public void validate() {
        // just call getVersion to initiate validation process
        getVersion();
    }

    /**
     * @return
     */
    public boolean isValid() {
        return (getVersion() > VERSION_INVALID);
    }

    // - Members ---------------------------------------------------------------

    final private AtomicBoolean m_enabled = new AtomicBoolean(true);

    final private GetVersionHandler m_getVersionHandler;

    final private VersionChangedHandler m_gotValidVersionHandler;

    final private VersionChangedHandler m_lostValidVersionHandler;

    private int m_version = VERSION_INVALID;


}
