/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.office.rt2.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import org.springframework.stereotype.Service;

import com.google.common.collect.EvictingQueue;
import com.openexchange.office.rt2.core.proxy.RT2DocProxyLogInfo;
import com.openexchange.office.rt2.core.proxy.RT2DocProxyLogInfo.Direction;
import com.openexchange.office.rt2.protocol.RT2Message;
import com.openexchange.office.rt2.protocol.value.RT2CliendUidType;
import com.openexchange.office.rt2.protocol.value.RT2DocUidType;

@Service
public class DocProxyMessageLoggerService {
	
	private static final int DEFAULT_COUNT_MESSAGES_STORED = 20;
	
	private ConcurrentHashMap<RT2CliendUidType, Collection<RT2DocProxyLogInfo>> msgs = new ConcurrentHashMap<>();
	
	//-------------------------------------------------------------------------
    public void addMessageForQueueToLog(Direction direction, final RT2Message msg) {
    	if (msg != null) {
    		getMsgsOfDocProxy(msg.getClientUID()).add(new RT2DocProxyLogInfo(direction, msg));
    	}
    }

    //-------------------------------------------------------------------------	
	public void resetQueueSizeForLogInfo(RT2CliendUidType clientUid, int count) {
		if (msgs.computeIfAbsent(clientUid, k -> {return Collections.synchronizedCollection(EvictingQueue.create(count));}) != null) {
			msgs.computeIfPresent(clientUid, (k, v) -> {
				final Collection<RT2DocProxyLogInfo> newQueue = Collections.synchronizedCollection(EvictingQueue.create(count));
				newQueue.addAll(v);
				return newQueue;
			});
		}
	}
	
    //-------------------------------------------------------------------------
    public List<String> formatMsgsLogInfo(RT2CliendUidType clientUid) {
    	List<RT2DocProxyLogInfo> res = getMsgsOfDocProxyAsList(clientUid);
    	return res.stream().map(m -> m.toString()).collect(Collectors.toList());
    }

    //-------------------------------------------------------------------------
    public List<RT2LogInfo> getMsgsAsList(RT2CliendUidType clientUid) {
    	List<RT2LogInfo> res = new ArrayList<>();
    	Collection<RT2DocProxyLogInfo> logList = getMsgsOfDocProxy(clientUid);
    	if (logList != null) {
    		res.addAll(logList);
    	}
    	return res;
    }

    //-------------------------------------------------------------------------
    public void add(RT2DocUidType docUid, Direction direction, String msgType, RT2CliendUidType clientUid) {
    	RT2DocProxyLogInfo docProxyLogInfo = new RT2DocProxyLogInfo(direction, msgType, clientUid, docUid);
    	getMsgsOfDocProxy(clientUid).add(docProxyLogInfo);
    }
        
    //-------------------------------------------------------------------------    
    public void remove(RT2CliendUidType clientUid) {
    	msgs.remove(clientUid);
    }
    
    //-------------------------------------------------------------------------
	private List<RT2DocProxyLogInfo> getMsgsOfDocProxyAsList(RT2CliendUidType clientUid) {
		List<RT2DocProxyLogInfo> res = new ArrayList<>();
    	Collection<RT2DocProxyLogInfo> logList = getMsgsOfDocProxy(clientUid);
    	if (logList != null) {
    		res.addAll(logList);
    	}
    	Collections.sort(res);
		return res;
	}    

	//-------------------------------------------------------------------------
	private Collection<RT2DocProxyLogInfo> getMsgsOfDocProxy(RT2CliendUidType cliendUid) {
		Collection<RT2DocProxyLogInfo> currCol = msgs.get(cliendUid);
		if (currCol == null) {
			Collection<RT2DocProxyLogInfo> newCol = Collections.synchronizedCollection(EvictingQueue.create(DEFAULT_COUNT_MESSAGES_STORED));
			Collection<RT2DocProxyLogInfo> oldCol = msgs.putIfAbsent(cliendUid, newCol);
			if (oldCol == null) {
				return newCol;
			}
		}
		return currCol;
	}

}
