/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */



package com.openexchange.office.htmldoc;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang3.StringEscapeUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class GenDocHelper
{

    private final static String EMPTYSPAN = "<span nonjquerydata=\"{&quot;isempty&quot;:true}\" style=\"font-family: Arial, Helvetica, sans-serif; font-weight: normal; font-style: normal; text-decoration: none; color: rgb(0, 0, 0); font-size: 11pt; line-height: 20px; background-color: transparent;\"></span>";
    private final static String EMPTYSPAN_NOSTYLE = "<span nonjquerydata=\"{&quot;isempty&quot;:true}\"></span>";

    static JSONArray shiftedCopy(
        JSONArray input)
    {
        final List<Object> list = input.asList();
        list.remove(0);
        return new JSONArray(list);
    }

    public static void appendAttributes(
        JSONObject attrs,
        StringBuilder document)
            throws JSONException
    {

        appendAttributes(attrs, document, false);

    }

    public static JSONObject getJQueryData(
        JSONObject attrs)
            throws JSONException
    {
        return getJQueryData(attrs, false);
    }

    public static void appendAttributes(
        JSONObject attrs,
        StringBuilder document,
        boolean implicit)
            throws JSONException
    {
        final JSONObject adapter = getJQueryData(attrs, implicit);
        appendJQueryData(document, adapter);
    }

    public static void appendJQueryData(
        StringBuilder document,
        JSONObject jquerydata)
            throws JSONException
    {

        if (jquerydata.length() > 0)
        {
            document.append(" jquerydata='");
            String str = escapeUIString(jquerydata.toString());
            document.append(str);
            document.append("' ");
        }
    }

    public static final String escapeUIString(
        final String text)
    {
        String str = StringEscapeUtils.escapeHtml4(text);
        str = str.replaceAll("'", "&#39;"); // escaping all single quotes
        return str;
    }

    public static JSONObject getJQueryData(
        JSONObject attrs,
        boolean implicit)
            throws JSONException
    {
        final JSONObject adapter = new JSONObject();
        if (attrs != null)
        {
            adapter.put("attributes", attrs);
        }
        if (implicit)
        {
            adapter.put("implicit", true);
        }
        return adapter;
    }

    public static boolean addEmptySpanForNecessaryNode(
        StringBuilder doc,
        INode node,
        boolean emptySpanNeeded)
    {
        boolean nodeTypeNeedsEmptySpan = node.needsEmptySpan();

        if (nodeTypeNeedsEmptySpan && emptySpanNeeded)
        {
            doc.append(EMPTYSPAN);
        }

        return nodeTypeNeedsEmptySpan;
    }

    public static void addEmptySpan(
        StringBuilder doc, boolean style)
    {
        int lengthPos = 0;
        int lastIndex = 0;

        if (style) {
            lastIndex = doc.lastIndexOf(EMPTYSPAN); // check, if the document already ends with such an empty span (55174)
            lengthPos = doc.length() - EMPTYSPAN.length();
            if (lastIndex != lengthPos) {
                doc.append(EMPTYSPAN);
            }
        } else {
            lastIndex = doc.lastIndexOf(EMPTYSPAN_NOSTYLE); // check, if the document already ends with such an empty span (55174)
            lengthPos = doc.length() - EMPTYSPAN_NOSTYLE.length();
            if (lastIndex != lengthPos) {
                doc.append(EMPTYSPAN_NOSTYLE);
            }
        }
    }

    private static final Map<String, Float> FACTORS;

    static
    {
        FACTORS = new HashMap<String, Float>();
        FACTORS.put("px", 1F);
        FACTORS.put("pc", 1F / 9F);
        FACTORS.put("pt", 4F / 3F);
        FACTORS.put("in", 96F);
        FACTORS.put("cm", 96F / 2.54F);
        FACTORS.put("mm", 96F / 25.4F);
    }

    public static float convertLength(
        float value,
        String fromUnit,
        String toUnit)
    {
        value *= FACTORS.get(fromUnit) / FACTORS.get(toUnit);
        return value;
    }

    public static float convertHmmToLength(
        float value,
        String toUnit)
    {
        return convertLength(value / 100, "mm", toUnit);
    };

    public static float convertHmmToPx(
        float value)
    {
        return convertHmmToLength(value, "px");
    };

}
