/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */



/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.odt.dom;

import java.util.Stack;
import org.xml.sax.Attributes;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.ElementNSWriter;
import com.openexchange.office.filter.odf.IContentDom;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.SaxContextHandler;
import com.openexchange.office.filter.odf.UnknownContentHandler;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawFrameHandler;
import com.openexchange.office.filter.odf.draw.Shape;
import com.openexchange.office.filter.odf.draw.ShapeHelper;
import com.openexchange.office.filter.odf.odt.dom.components.ParagraphComponent;
import com.openexchange.office.filter.odf.odt.dom.components.RootComponent;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleFamily;

public class BodyHandler extends SaxContextHandler {

    final Body body;
	final boolean contentStyle;
	final private DLList<Object> pageAnchoredComponents = new DLList<Object>();
	private Stack<SectionStartMarker> sectionStack = null;

    public BodyHandler(SaxContextHandler parentContext, Body body) {
    	super(parentContext);

    	this.body = body;
    	contentStyle = getFileDom() instanceof IContentDom;
    }

    @Override
    public SaxContextHandler startElement(Attributes attributes, String uri, String localName, String qName) {
    	final SaxContextHandler textHandler = TextContentHelper.startElement(this, body, attributes, uri, localName, qName);
    	if(textHandler!=null) {
    		return textHandler;
    	}
    	if(qName.equals("draw:frame")) {
    		final DrawFrame drawFrame = new DrawFrame(new AttributesImpl(attributes), null, true, contentStyle);
    		pageAnchoredComponents.add(drawFrame);
    		return new DrawFrameHandler(this, drawFrame);
    	}
    	final Shape shape = ShapeHelper.getShape(attributes, uri, localName, qName, null, true, contentStyle);
    	if(shape!=null) {
    	    pageAnchoredComponents.add(shape);
    	    return shape.getContextHandler(this);
    	}
    	if(qName.equals("text:section")) {
    		if(sectionStack==null) {
    			sectionStack = new Stack<SectionStartMarker>();
    		}
    		final SectionStartMarker sectionStartMarker = new SectionStartMarker(attributes);
    		sectionStack.push(sectionStartMarker);
    		body.getContent().add(sectionStartMarker);
    		return this;
    	}
		final ElementNSWriter element = new ElementNSWriter(getFileDom(), attributes, uri, qName);
		body.getContent().add(element);
		return new UnknownContentHandler(this, element);
    }

	@Override
	public void endElement(String localName, String qName) {
		super.endElement(localName, qName);

		if(qName.equals("text:section")) {
			final SectionStartMarker sectionStartMarker = sectionStack.pop();
			final SectionEndMarker sectionEndMarker = new SectionEndMarker(sectionStartMarker);
			body.getContent().add(sectionEndMarker);
			sectionStartMarker.setEndMarker(sectionEndMarker);
		}
	}

	@Override
	public void endContext(String qName, String characters) {
		super.endContext(qName, characters);

		if(qName.equals("office:text")) {
		    // TODO: operationDocument needs to be supplied
            final Paragraph paragraph = ParagraphComponent.getFirstParagraph(new RootComponent((OdfOperationDoc)null, body, true), true);
            final String firstParaStyle = paragraph.getStyleName();
            if(firstParaStyle!=null&&!firstParaStyle.isEmpty()) {
                // try to determine the master page style... otherwise we will use "Standard"
                final StyleBase paraStyle = ((TextContent)body.getOwnerDocument()).getDocument().getStyleManager().getStyle(firstParaStyle, StyleFamily.PARAGRAPH, true);
                if(paraStyle!=null) {
                    final String masterPageName = paraStyle.getAttribute("style:master-page-name");
                    if(masterPageName!=null) {
                        body.setMasterPageName(masterPageName);
                    }
                }
            }
			if(!pageAnchoredComponents.isEmpty()) {
				// move these objects into the first paragraph...
				final TextSpan textSpan = new TextSpan();
				paragraph.getContent().addFirst(textSpan);
				textSpan.setContent(pageAnchoredComponents);
			}
		}
	}
}
