/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */



package com.openexchange.office.spellcheck.impl.hunspell;

import java.io.File;
import java.util.ArrayList;
import java.util.Timer;
import java.util.regex.Pattern;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.openexchange.config.ConfigurationService;
import com.openexchange.office.spellcheck.api.ISpellCheck;
import com.openexchange.office.spellcheck.api.IValidator;
import com.openexchange.office.tools.common.string.UnicodeValidator;

/**
 * {@link SpellCheckHunspell}
 *
 * @author <a href="mailto:oliver.specht@open-xchange.com">Oliver Specht</a>
 */
@Service
public class SpellCheckHunspell implements InitializingBean, IValidator, ISpellCheck {
    final private static Logger LOG = LoggerFactory.getLogger(SpellCheckHunspell.class);
    final private static String SEPARATOR_PATTERN = "[\t\n\u00a0\f\r \\.,;\\?!\":(){}\\[\\] ]";
    final private static String THE_DOT = ".";
    final private static char THE_BLANK = ' ';
    final private static int INIT_HUNSPELL_TIMER_PERIOD_MILLIS = 300000;

    final private static String KEY_SPELL_RESULT = "spellResult";
    final private static String KEY_SUPPORTED_LOCALES = "SupportedLocales";

    /**
     * {@link HunspellStatus}
     *
     * @author <a href="mailto:oliver.specht@open-xchange.com">Oliver Specht</a>
     * @since v7.8.3
     */
    private enum HunspellStatus {
        ENABLED,
        DISABLED,
        INITIALIZE
    }

    /**
     * Initializes a new {@link SpellCheckHunspell}.
     */
    public SpellCheckHunspell() {
        super();
    }

    // - InitializingBean ------------------------------------------------------

    @Override
    public void afterPropertiesSet() throws Exception {
        m_hunspellStatus = implInitHunspell();
    }

    // - IValidator ------------------------------------------------------------

    @Override
    public boolean isValid() {
        return (HunspellStatus.ENABLED == m_hunspellStatus);
    }

    // - ISpellCheck ---------------------------------------------------------

    @Override
    public JSONObject checkSpelling(final JSONObject jsonParams) throws JSONException {
        final JSONArray jsonArrayResult = new JSONArray();
        final String text = jsonParams.getString("text");
        final String locale = jsonParams.getString("locale");
        final int offset = jsonParams.getInt("offset");
        final Hunspell.Dictionary dict = ((null != m_dictionaries) ? m_dictionaries.getDictionary(locale) : null);

        if (dict != null) {
            final String textToUse = text + ' ';
            int startPos = 0;
            int len = 0;

            for (int pos = 0; pos < textToUse.length(); ++pos) {
                final String substring = textToUse.substring(pos, pos + 1);

                if (Pattern.matches(SEPARATOR_PATTERN, substring)) {
                    if (len > 0) {
                        // #40631# A dot might also be used for abbreviations.
                        final int extra = substring.equals(THE_DOT) ? 1 : 0;
                        final String word = textToUse.substring(startPos, startPos + len + extra);

                        // #65073# workaround for possible hunspell crash with
                        // composed unicode characters
                        // we just replace high/low-surrogate characters with a
                        // replacement character.
                        final String safeWord = UnicodeValidator.replaceComposedUnicodeCharacters(word, THE_BLANK);

                        if (dict.misspelled(safeWord)) {
                            final JSONObject error = new JSONObject();

                            try {
                                error.put("start", startPos + offset);
                                error.put("length", len);
                                error.put("locale", dict.getLocale());
                                error.put("word", word);
                                error.put("replacements", new JSONArray(dict.suggest(safeWord)));
                            } catch (@SuppressWarnings("unused") final JSONException e) {
                                // OK
                            }

                            jsonArrayResult.put(error);
                        }

                        len = 0;
                    }

                    startPos = pos + 1;
                } else {
                    ++len;
                }
            }
        }

        return new JSONObject().put(KEY_SPELL_RESULT, jsonArrayResult);
    }

    @Override
    public JSONObject checkParagraphSpelling(final JSONObject jsonParams) throws JSONException {
        final JSONArray paragraph = jsonParams.getJSONArray("paragraph");
        final JSONArray jsonArraySpellResult = new JSONArray();
        final StringBuilder text = new StringBuilder(256);
        int pos = 0;
        int offset = 0;
        String locale = null;

        while (pos < paragraph.length()) {
            final JSONObject element = paragraph.getJSONObject(pos);

            if ((locale != null && locale.compareTo(element.getString("locale")) != 0)) {
                final JSONObject jsonCurParams = new JSONObject();

                jsonCurParams.put("text", text.toString());
                jsonCurParams.put("locale", locale);
                jsonCurParams.put("offset", offset);

                final JSONArray jsonArrayCheck = checkSpelling(jsonCurParams).getJSONArray(KEY_SPELL_RESULT);

                for (int i = 0; i < jsonArrayCheck.length(); ++i) {
                    jsonArraySpellResult.put(jsonArrayCheck.get(i));
                }

                offset += text.length();
                text.setLength(0);
                locale = null;
            }

            locale = element.getString("locale");
            text.append(element.getString("word"));
            ++pos;
        }
        if (text.length() > 0) {
            final JSONObject jsonCurParams = new JSONObject();

            jsonCurParams.put("text", text.toString());
            jsonCurParams.put("locale", locale);
            jsonCurParams.put("offset", offset);

            final JSONArray check = checkSpelling(jsonCurParams).getJSONArray(KEY_SPELL_RESULT);

            for (int i = 0; i < check.length(); ++i) {
                jsonArraySpellResult.put(check.get(i));
            }
        }

        return new JSONObject().put(KEY_SPELL_RESULT, jsonArraySpellResult);
    }

    @Override
    public JSONObject suggestReplacements(final JSONObject jsonParams) throws JSONException {
        final String word = jsonParams.getString("word");
        final String locale = jsonParams.getString("locale");
        final String safeWord = UnicodeValidator.replaceComposedUnicodeCharacters(word, THE_BLANK);
        final Hunspell.Dictionary dict = (null != m_dictionaries) ? m_dictionaries.getDictionary(locale) : null;

        return new JSONObject().put(KEY_SPELL_RESULT, new JSONArray((null != dict) ? dict.suggest(safeWord) : new ArrayList<>()));
    }

    @Override
    public JSONObject isMisspelled(final JSONObject jsonParams) throws JSONException {
        final String word = jsonParams.getString("word");
        final String locale = jsonParams.getString("locale");
        final String safeWord = UnicodeValidator.replaceComposedUnicodeCharacters(word, THE_BLANK);
        final Hunspell.Dictionary dict = ((null != m_dictionaries) ? m_dictionaries.getDictionary(locale) : null);

        return new JSONObject().put(KEY_SPELL_RESULT, ((null != dict) && dict.misspelled(safeWord)));
    }

    @Override
    public JSONObject getSupportedLocales() throws JSONException {
        return new JSONObject().put(KEY_SUPPORTED_LOCALES, new JSONArray((null != m_dictionaries) ? m_dictionaries.getLocales() : new ArrayList<>()));
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return Either {@link HunspellStatus.ENABLED} or
     * {@link HunspellStatus.INITIALIZE} if Hunspell still needs to be
     * initialized
     */
    protected synchronized HunspellStatus implInitHunspell() {
        try {
            String hunspellLibrary =
                m_configService.getProperty("com.openexchange.office.spellchecker.hunspell.library", null);
            if (hunspellLibrary == null) {
                hunspellLibrary = m_configService.getProperty("com.openexchange.spellchecker.hunspell.library", null);
            }
            if (hunspellLibrary == null) {
                for (String _hunspellLibrary : hunspellLibraryList) {
                    if (new File(_hunspellLibrary).canRead()) {
                        hunspellLibrary = _hunspellLibrary;
                        break;
                    }
                }
            }
            String dictionaryPath =
                m_configService.getProperty("com.openexchange.office.spellchecker.hunspell.dictionaries", null);
            if (dictionaryPath == null) {
                dictionaryPath = m_configService.getProperty("com.openexchange.spellchecker.hunspell.dictionaries", null);
            }
            if (dictionaryPath == null) {
                for (String _dictionaryPath : dictionaryPathList) {
                    if (new File(_dictionaryPath).canRead()) {
                        dictionaryPath = _dictionaryPath;
                        break;
                    }
                }
            }
            if (hunspellLibrary != null && dictionaryPath != null) {
                try {
                    final boolean hunspellLibExists = (new File(hunspellLibrary)).canRead();
                    final boolean dictionaryPathExists = (new File(dictionaryPath)).canRead();

                    if (hunspellLibExists && dictionaryPathExists) {
                        m_hunspell = Hunspell.getInstance(hunspellLibrary);
                        m_dictionaries = new DictionaryContainer(m_hunspell,
                            dictionaryPath);
                    }
                } catch (final UnsatisfiedLinkError e) {
                    if (LOG.isErrorEnabled()) {
                        LOG.error("Error initializing Hunspell - Unsatisfied Link Error", e);
                    }
                } catch (final Throwable e) {
                    if (LOG.isErrorEnabled()) {
                        LOG.error("Error initializing Hunspell", e);
                    }
                }
            }

            if (null != m_hunspell && null != m_dictionaries) {
                LOG.info("SpellCheck sucessfully initialized Hunspell library");
                return HunspellStatus.ENABLED;
            }

            if ((HunspellStatus.DISABLED == m_hunspellStatus)) {
                if (null == m_hunspell) {
                    LOG.warn(
                        "SpellCheck failed to initialize Hunspell library. Hunspell library cannot be found or executed: "
                            + hunspellLibrary);
                }

                if (null == m_dictionaries) {
                    LOG.warn(
                        "SpellCheck failed to initialize Hunspell library. Hunspell dictionary path cannot be found or read: "
                            + dictionaryPath);
                }

                LOG.info(new StringBuilder(128)
                    .append("SpellCheck automatically retries to initialize Hunspell library with a period of ")
                    .append(INIT_HUNSPELL_TIMER_PERIOD_MILLIS / 1000)
                    .append("s...")
                    .toString());
            }
            return HunspellStatus.INITIALIZE;
        } catch (java.security.AccessControlException e) {
            LOG.warn("AccessControlException: Spellcheck failed to initialize Hunspell library, please configure com.openexchange.spellchecker.hunspell.library and com.openexchange.spellchecker.hunspell.dictionaries.");
            return HunspellStatus.DISABLED;
        }
    }

    // - Static members --------------------------------------------------------

	final private static String[] hunspellLibraryList = {
		"/usr/lib/x86_64-linux-gnu/libhunspell-1.7.so.0",
		"/usr/lib64/libhunspell-1.7.so.0", "/usr/lib/x86_64-linux-gnu/libhunspell-1.4.so.0",
		"/usr/lib64/libhunspell-1.4.so.0", "/usr/lib/x86_64-linux-gnu/libhunspell-1.3.so.0",
		"/usr/lib64/libhunspell-1.3.so.0", "/usr/lib/x86_64-linux-gnu/libhunspell-1.2.so.0",
		"/usr/lib64/libhunspell-1.2.so.0" };

	final private static String[] dictionaryPathList = {
		"/usr/share/hunspell",
		"/usr/share/myspell" };

    // - Members ---------------------------------------------------------------

    @Autowired
    private ConfigurationService m_configService;

    private HunspellStatus m_hunspellStatus;

    private Hunspell m_hunspell = null;

    private DictionaryContainer m_dictionaries = null;

    protected Timer m_initHunspellTimer = null;
}
