/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */



package com.openexchange.office.spellcheck.impl.hunspell;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * {@link DictionaryContainer}
 *
 * @author <a href="mailto:oliver.specht@open-xchange.com">Firstname Lastname</a>
 */
public class DictionaryContainer {
    final private static Logger LOG = LoggerFactory.getLogger(DictionaryContainer.class);

    private final List<Hunspell.Dictionary> dictionaries = new ArrayList<>();
    private final List<String> locales = new ArrayList<>();
    /**
     * Initializes a new {@link DictionaryContainer}.
     * @param hunspell instance of Hunspell
     * @param dictPath path to the Hunspell dictionaries
     */
    public DictionaryContainer( Hunspell hunspell, String dictPath ){
        FileFilter fileFilter = new FileFilter() {
            @Override
            public boolean accept(File file) {
                if( file.isDirectory() ) {
                    return false;
                }
                String fileName = file.toURI().getPath();
                return fileName.endsWith(".dic");
            }
        };
        File dictDirectory = new File( dictPath );
        // get all .dic and .aff files from the folder
        File files[] = dictDirectory.listFiles( fileFilter );
        for( int dictIndex = 0; dictIndex < files.length; ++ dictIndex ){
            File dictFile = files[dictIndex];
            String fileName = dictFile.getName();
            int localeLen = fileName.lastIndexOf('.', fileName.length() );
            if( localeLen > 0){
                String locale = fileName.substring(0, localeLen);
                File affFile = new File(dictPath + '/' + locale + ".aff");
                if( affFile.exists() ){
                    try{
                        Hunspell.Dictionary dict = hunspell.getDictionary( dictPath + '/' + locale);
                        dictionaries.add(dict);
                        locales.add( locale );
                    }catch( UnsupportedEncodingException e){
                        LOG.error(e.getMessage());
                    }catch( FileNotFoundException e){
                        LOG.error(e.getMessage());
                  }
                }
            }
        }

    }
    /**
     * @param locale requested locale
     * @return null if locale is not supported, a dictionary otherwise
     */
    public Hunspell.Dictionary getDictionary( String locale ){
        Hunspell.Dictionary ret = null;
        if( locale.length() > 1 ){
            Iterator<Hunspell.Dictionary> it = dictionaries.iterator();
            while( it.hasNext() ){
                Hunspell.Dictionary dict = it.next();
                String lc = dict.getLocale().getCountry();
                String ls1 = locale.length() >= 5 ? locale.substring(3, 5) : new String();
                String ll = dict.getLocale().getLanguage();
                String ls2 = locale.substring(0, 2);
                if( lc.equals( ls1 ) &&
                    ll.equals( ls2 ) ){
                    ret = dict;
                    break;
                }
            }
        }
        return ret;
    }
    /**
     * @return list of dictionary locales
     */
    public List<String> getLocales(){
        return locales;
    }
}
