/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */



package com.openexchange.office.rt2.rest.json.fields;

import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import com.openexchange.ajax.customizer.file.AdditionalFileField;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.file.storage.DelegatingFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.UserizedFile;
import com.openexchange.java.Strings;
import com.openexchange.office.tools.annotation.RegisteredService;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link ResourceIDField}
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @since v7.8.0
 */
@Service
@RegisteredService(registeredClass = AdditionalFileField.class)
public class ResourceIDField implements AdditionalFileField {

    private static final Logger LOG                = LoggerFactory.getLogger(ResourceIDField.class);
    private static final String HIDRIVE_SCHEMA     = "hidrive://";
    private static final int    HIDRIVE_PATH_PARTS = 2;
    private static final String PATH_SEPARATOR     = "/";

    // ---------------------------------------------------------------
    /**
     * Initializes a new {@link ResourceIDField}.
     */
    public ResourceIDField() {
        super();
    }

    // ---------------------------------------------------------------
    @Override
    public Field[] getRequiredFields() {
        return new Field[] { Field.FOLDER_ID, Field.ID };
    }

    // ---------------------------------------------------------------
    @Override
    public int getColumnID() {
        return 7020;
    }

    // ---------------------------------------------------------------
    @Override
    public String getColumnName() {
        return "com.openexchange.realtime.resourceID";
    }

    // ---------------------------------------------------------------
    @Override
    public Object getValue(File file, ServerSession session) {
        return getRealtimeID(file);
    }

    // ---------------------------------------------------------------
    @Override
    public List<Object> getValues(List<File> files, ServerSession session) {
        if (null == files) {
            return null;
        }
        List<Object> values = new ArrayList<Object>(files.size());
        for (File file : files) {
            values.add(getRealtimeID(file));
        }
        return values;
    }

    // ---------------------------------------------------------------
    @Override
    public Object renderJSON(AJAXRequestData requestData, Object value) {
        return value;
    }

    // ---------------------------------------------------------------
    /**
     * Gets the realtime file identifier for a specific file. By default, this is the file's (fully qualified) identifier, while for files
     * from the <code>infostore</code> account, only the (relative) file identifier is used in order to support scenarios where the same
     * file appears in different folders for different users.
     *
     * @param file The file to get the realtime identifier for
     * @return The realtime identifier, or <code>null</code> if no identifier can be extracted from the passed file reference
     */
    private static String getRealtimeID(File file) {
        if (null != file) {
            final String fileId = file.getId();
            if (StringUtils.isNotEmpty(fileId) && fileId.startsWith(HIDRIVE_SCHEMA)) {
                return extractRealtimeIDFromHiDriveFile(file);
            }
            File accessFile = file;
            if (DelegatingFile.class.isInstance(accessFile)) {
                // Mandatory for Guard support - see DOCS-2723
                // Guard wraps file with a MetaDataAddFile which is not in the
                // class hierarchy regarding UserizedFile. Therefore we need to
                // extract the delegate file to have access to the original file
                // instance.
                accessFile = ((DelegatingFile) accessFile).getDelegate();
            }
            if (UserizedFile.class.isInstance(accessFile)) {
                String originalID = ((UserizedFile) accessFile).getOriginalId();
                return Strings.isEmpty(originalID) ? accessFile.getId() : originalID;
            }
        }
        return null;
    }

    // ---------------------------------------------------------------
    /**
     * Extracts the unique part from a HiDrive file.
     *
     * @param file instance from a hidrive file
     * @return the unique part from the hidrive file to be used as
     *         unique identifier.
     */
    private static String extractRealtimeIDFromHiDriveFile(File file) {
        // Extract static unique part from the hidrive url.
        // ATTENTION: This is highly implementation specific and will
        // break as soon as someone changes the hidrive url structure.
        // We currently assume the following hidrive file-id "synatx"
        //
        // "hidrive://"<account-id>"/"<folderid>"/"<fileId>
        //
        final String hiDriveFileId = file.getId();
        if (hiDriveFileId.length() > HIDRIVE_SCHEMA.length()) {
            final String pathPart = hiDriveFileId.substring(HIDRIVE_SCHEMA.length());
            final int numPathParts = (StringUtils.countMatches(pathPart, PATH_SEPARATOR));
            if (numPathParts == HIDRIVE_PATH_PARTS) {
                // We cut-out the user/account-specific part from the path part
                // to have a global unique id for the file.
                final int offsetFirstSep = pathPart.indexOf(PATH_SEPARATOR);
                String realtimeId = pathPart.substring(offsetFirstSep + 1);
                return realtimeId;
            }
        }

        LOG.warn("Malformed hidrive url detected while trying to parse it {}", hiDriveFileId);
        return file.getId();
    }

}
