/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */



package com.openexchange.office.document.api;

import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.tools.common.error.ErrorCode;

/**
 * Provides information about an optional advisory
 * lock set on a document file.
 *
 * @author Carsten Driesner {carsten.driesner@open-xchange.com}
 * @since 7.10.5
 *
 */
public class AdvisoryLockInfo {
    public static final String ADVISORYLOCK_KEY = "oxDocAdvisoryLock";

    private static final String PROP_HOSTNAME = "hostName";
    private static final String PROP_CONTEXT_ID = "cid";
    private static final String PROP_USER_ID = "uid";
    private static final String PROP_SINCE = "since";

    private final String name;
    private final int contextId;
    private final int userId;
    private final String sinceDateTime;
    private ErrorCode errorCode = ErrorCode.NO_ERROR;
    private boolean advisoryLockWritten = false;

    public AdvisoryLockInfo(ErrorCode errorCode) {
        name = "";
        contextId = -1;
        userId = -1;
        sinceDateTime = "";
        this.errorCode = errorCode;
    }

	public AdvisoryLockInfo(String hostName, int contextId, int userId, String sinceDateTime) {
		this.name = hostName;
		this.contextId = contextId;
		this.userId = userId;
		this.sinceDateTime = sinceDateTime;
	}

	public AdvisoryLockInfo(JSONObject advisoryLockInfo) throws JSONException {
		name = advisoryLockInfo.getString(PROP_HOSTNAME);
		contextId = advisoryLockInfo.getInt(PROP_CONTEXT_ID);
		userId = advisoryLockInfo.getInt(PROP_USER_ID);
		sinceDateTime = advisoryLockInfo.getString(PROP_SINCE);
	}

	public JSONObject asJSONObject() throws JSONException {
		final JSONObject json = new JSONObject();
		json.put(PROP_HOSTNAME, this.name);
		json.put(PROP_CONTEXT_ID, this.contextId);
		json.put(PROP_USER_ID, this.userId);
		json.put(PROP_SINCE, sinceDateTime);
		return json;
	}

	public String getName() {
		return this.name;
	}

    public int getContextId() {
        return this.contextId;
    }

    public int getUserId() {
        return this.userId;
    }

    public String getSinceDateTime() {
        return this.sinceDateTime;
    }

    public ErrorCode getErrorCode() {
        return this.errorCode;
    }

    public AdvisoryLockInfo setErrorCode(ErrorCode errorCode) {
        this.errorCode = errorCode;
        return this;
    }

    public boolean getAdvisoryLockWritten() {
        return this.advisoryLockWritten;
    }

    public void setAdvisoryLockWritten(boolean written) {
        this.advisoryLockWritten = written;
    }

}
