/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */



package com.openexchange.imageconverter.impl;

import static com.openexchange.imageconverter.impl.ImageConverterUtils.LOG;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItemService;
import com.openexchange.imageconverter.api.ISubGroup;
import com.openexchange.imageconverter.api.ImageConverterException;

/**
 * {@link ImageConverterCache}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10
 */
public class ImageConverterCache extends Thread {

    /**
     * Initializes a new {@link ImageConverterCache}.
     */
    public ImageConverterCache(@NonNull final ImageConverter imageConverter) {
        super("ImageConverter Cache");

        m_fileItemService = (m_imageConverter = imageConverter).getFileItemService();

        m_maxCacheSize = ImageConverterConfig.IMAGECONVERTER_CACHE_MAX_SIZE;
        m_maxCacheKeyCount = ImageConverterConfig.IMAGECONVERTER_CACHE_MAX_KEY_COUNT;
        m_cacheKeyTimeoutMillis = ImageConverterConfig.IMAGECONVERTER_CACHE_KEY_TIMEOUT_MILLIS;
        m_cacheCleanupTimeoutMillis = ImageConverterConfig.IMAGECONVERTER_CACHE_CLEANUP_TIMEOUT_MILLIS;
    }

    /* (non-Javadoc)
     * @see java.lang.Thread#run()
     */
    @Override
    public void run() {
        LOG.trace("IC ImageConverterCache thread worker started");

        while (isRunning()) {
            try {
                sleep(m_cacheCleanupTimeoutMillis);
                implPerformCleanup();
            } catch (@SuppressWarnings("unused") InterruptedException e) {
                LOG.trace("IC ImageConverterCache thread worker received interrupt");
            }
        }

        LOG.trace("IC ImageConverterCache thread worker finished");
    }

    /**
    *
    */
   public void shutdown() {
       if (m_running.compareAndSet(true, false)) {
           final boolean info = LOG.isInfoEnabled();
           long shutdownStartTimeMillis = 0;

           if (info) {
               LOG.info("IC ImageConverterCache starting shutdown");
               shutdownStartTimeMillis = System.currentTimeMillis();
           }

           interrupt();

           if (info) {
               LOG.info(ImageConverterUtils.IC_STR_BUILDER().
                   append("ImageConverterCache shutdown finished in ").
                   append(System.currentTimeMillis() - shutdownStartTimeMillis).append("ms").toString());
           }
       }
   }

    /**
     * isRunning
     */
    public boolean isRunning() {
        return m_running.get();
    }

    // - Implementation --------------------------------------------------------

    /**
     *
     */
    private void implPerformCleanup() {
        final long startTimeMillis = System.currentTimeMillis();
        long endTimeMillis = 0;
        long countDurationMillis = 0;
        long timeDurationMillis = 0;
        long sizeDurationMillis = 0;
        int countRemoved = 0;
        int timeRemoved = 0;
        int sizeRemoved = 0;

        try {
            countRemoved = implRemoveObsoleteCountBasedKeys();
        } catch (ImageConverterException | FileItemException e) {
            LOG.error("IC ImageConverterCache received exception during detection of removable count based keys", Throwables.getRootCause(e));
        } finally {
            countDurationMillis = (timeDurationMillis = System.currentTimeMillis()) - startTimeMillis;
        }

        try {
            timeRemoved = implRemoveObsoleteTimeBasedKeys();
        } catch (ImageConverterException | FileItemException e) {
            LOG.error("IC ImageConverterCache received exception during detection of removable time based keys", Throwables.getRootCause(e));
        } finally {
            timeDurationMillis = (sizeDurationMillis = System.currentTimeMillis()) - timeDurationMillis;
        }

        try {
            sizeRemoved = implRemoveObsoleteSizeBasedKeys();
        } catch (ImageConverterException | FileItemException e) {
            LOG.error("IC ImageConverterCache received exception during detection of removable size based keys", Throwables.getRootCause(e));
        } finally {
            sizeDurationMillis = (endTimeMillis = System.currentTimeMillis()) - sizeDurationMillis;
        }

        implLogCacheStatistics(countRemoved, countDurationMillis, timeRemoved, timeDurationMillis, sizeRemoved, sizeDurationMillis, endTimeMillis - startTimeMillis);
    }

    /**
     * @return
     * @throws ImageConverterException
     */
    final private static String SQL_COUNT_BASED_QUERY = ImageConverterUtils.STR_BUILDER().
        append("SELECT fc.FileStoreNumber, fc.FileStoreId, fc.SubGroupId, fc.FileId, MIN(fp.ModificationDate) AS KeyOldest").
        append(" FROM Content_OX_IC fc INNER JOIN Properties_OX_IC fp ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId").
        append(" GROUP BY fc.SubGroupId, fc.FileId").
        append(" ORDER BY KeyOldest ASC").toString();

    private int implRemoveObsoleteCountBasedKeys() throws ImageConverterException, FileItemException {
        int removedKeys = 0;

        if (m_maxCacheKeyCount > 0) {
            final int limitCount = 4096;
            final Set<String> subGroupIdSet = new LinkedHashSet<>(limitCount);
            long curCacheKeyCount = m_imageConverter.getKeyCount();

            while (curCacheKeyCount > m_maxCacheKeyCount) {
                final String limitClause = Long.toString(Math.min(curCacheKeyCount - m_maxCacheKeyCount, limitCount));

                subGroupIdSet.clear();
                m_fileItemService.getSubGroupsBy(ImageConverterUtils.IMAGECONVERTER_GROUPID, null, limitClause, new Consumer<ISubGroup>() {

                    /* (non-Javadoc)
                     * @see java.util.function.Consumer#accept(java.lang.Object)
                     */
                    @Override
                    public void accept(@NonNull final ISubGroup curElement) {
                        subGroupIdSet.add(curElement.getSubGroupId());
                    }
                });

                final int curKeyCountToRemove = subGroupIdSet.size();

                if (curKeyCountToRemove > 0) {
                    m_imageConverter.remove(subGroupIdSet);

                    removedKeys += curKeyCountToRemove;
                    curCacheKeyCount -= curKeyCountToRemove;
                } else {
                    // leave in case, we got no items at all
                    break;
                }
            }
        }

        return removedKeys;
    }

    /**
     * @return
     * @throws ImageConverterException
     */
    final private static String SQL_TIME_BASED_QUERY = ImageConverterUtils.STR_BUILDER().
        append("SELECT fc.FileStoreNumber, fc.FileStoreId, fc.SubGroupId, fc.FileId, MIN(fp.ModificationDate) AS KeyOldest").
        append(" FROM Content_OX_IC fc INNER JOIN Properties_OX_IC fp ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId").
        append(" WHERE fp.ModificationDate<=").toString();

    private int implRemoveObsoleteTimeBasedKeys() throws ImageConverterException, FileItemException {
        int removedKeys = 0;

        if (m_cacheKeyTimeoutMillis > 0) {
            final long maxKeyTimeMillis = System.currentTimeMillis() - m_cacheKeyTimeoutMillis;
            final int limitCount = 4096;
            final Set<String> subGroupIdSet = new LinkedHashSet<>(limitCount);
            final String whereClause = "fp.ModificationDate <= " + maxKeyTimeMillis;
            final String limitClause = Integer.toString(limitCount);

            // no removal, if keys to remove < 0
            while (true) {
                subGroupIdSet.clear();
                m_fileItemService.getSubGroupsBy(ImageConverterUtils.IMAGECONVERTER_GROUPID, whereClause, limitClause, new Consumer<ISubGroup>() {

                    /* (non-Javadoc)
                     * @see java.util.function.Consumer#accept(java.lang.Object)
                     */
                    @Override
                    public void accept(final ISubGroup curElement) {
                        subGroupIdSet.add(curElement.getSubGroupId());
                    }
                });

                final int curKeyCountToRemove = subGroupIdSet.size();

                if (curKeyCountToRemove > 0) {
                    m_imageConverter.remove(subGroupIdSet);
                    removedKeys += curKeyCountToRemove;
                } else {
                    // leave in case, we got no items at all
                    break;
                }
            }
        }

        return removedKeys;
    }

    /**
     * @return
     * @throws ImageConverterException
     */
    final private static String SQL_SIZE_BASED_QUERY = ImageConverterUtils.STR_BUILDER().
        append("SELECT fc.FileStoreNumber, fc.FileStoreId, fc.SubGroupId, fc.FileId, SUM(fp.Length) AS KeyLength, MIN(fp.ModificationDate) AS KeyOldest").
        append(" FROM Content_OX_IC fc INNER JOIN Properties_OX_IC fp ON fc.FileStoreNumber=fp.FileStoreNumber AND fc.FileStoreId=fp.FileStoreId").
        append(" GROUP BY fc.SubGroupId, fc.FileId").
        append(" ORDER BY KeyOldest ASC").toString();

    private int implRemoveObsoleteSizeBasedKeys() throws ImageConverterException, FileItemException {
        int removedKeys = 0;

        if (m_maxCacheSize > 0) {
            final long sizeToRemove = m_imageConverter.getTotalImagesSize() - m_maxCacheSize;

            if (sizeToRemove > 0) {
                final int limitCount = 4096;
                final Set<String> subGroupIdSet = new LinkedHashSet<>(limitCount);
                final String limitClause = Integer.toString(limitCount);
                final AtomicLong removedSize = new AtomicLong(0);

                while (removedSize.get() < sizeToRemove) {
                    subGroupIdSet.clear();
                    m_fileItemService.getSubGroupsBy(ImageConverterUtils.IMAGECONVERTER_GROUPID, null, limitClause, new Consumer<ISubGroup>() {

                        /* (non-Javadoc)
                         * @see java.util.function.Consumer#accept(java.lang.Object)
                         */
                        @Override
                        public void accept(final ISubGroup curElement) {
                            if ((null != curElement) && ((removedSize.get() < sizeToRemove))) {
                                subGroupIdSet.add(curElement.getSubGroupId());
                                removedSize.addAndGet(curElement.getLength());
                            }
                        }
                    });

                    final int curKeyCountToRemove = subGroupIdSet.size();

                    if (curKeyCountToRemove > 0) {
                        m_imageConverter.remove(subGroupIdSet);
                        removedKeys += curKeyCountToRemove;
                    } else {
                        // leave in case, we got no items at all
                        break;
                    }
                }
            }
        }

        return removedKeys;
    }

    /**
     * @param removableKeySet
     * @param cleanupStartTimeMillis
     */
    private static void implLogCacheStatistics(final int countRemoved, final long countDuration,
        final int timeRemoved, final long timeDuration,
        final int sizeRemoved, final long sizeDuration,
        final long totalDuration) {

        LOG.trace(ImageConverterUtils.IC_STR_BUILDER().
            append("cache detected ").
            append(countRemoved).append(" (").append(countDuration).append("ms) count based, ").
            append(timeRemoved).append(" (").append(timeDuration).append("ms) time based, ").
            append(sizeRemoved).append(" (").append(sizeDuration).append("ms) size based distinct keys that have been removed").
            append(" (").append(totalDuration).append("ms)").toString());
    }

    // - Members ---------------------------------------------------------------

    final private AtomicBoolean m_running = new AtomicBoolean(true);

    final private ImageConverter m_imageConverter;

    final private IFileItemService m_fileItemService;

    private long m_maxCacheSize = -1L;

    private long m_maxCacheKeyCount = -1L;

    private long m_cacheKeyTimeoutMillis = 30L * 24L * 60L * 60L * 1000L;

    private long m_cacheCleanupTimeoutMillis = 5L * 60L * 1000L;

}
