/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */



package com.openexchange.documentconverter.client.impl;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import com.openexchange.documentconverter.DocumentConverterManager;
import com.openexchange.documentconverter.DocumentConverterUtil;
import com.openexchange.documentconverter.HttpHelper;
import com.openexchange.documentconverter.LogData;
import com.openexchange.documentconverter.NonNull;

/**
 * {@link RemoteDocumentConverterCheck}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
class RemoteValidator {

    /**
     * ENABLE_LOG_TIMEOUT
     */
    final private static long LOG_TIMEOUT = 60000;

    protected class RemoteValidatorRunnable implements Runnable {

        /**
         *
         */
        public RemoteValidatorRunnable() {
            super();
        }

        /**
         * @return
         * @throws Exception
         */
        @Override
        public void run() {
            final StringBuilder dcServerId = new StringBuilder(32);
            final AtomicInteger remoteApiVersion = new AtomicInteger(-1);

            // check remote connection API; may be checked repeatedly, if a valid remote URL is set
            if (null != m_remoteUrl) {
                try {
                    HttpHelper.get().executeGet(
                        m_remoteUrl,
                        (int responseCode, InputStream responseInputStream, String resultCookie) -> {
                            try (final BufferedReader inputReader = new BufferedReader(new InputStreamReader(responseInputStream))) {
                                for (String readLine = null; (readLine = inputReader.readLine()) != null;) {
                                    readLine = readLine.toLowerCase();

                                    // check, if we really have a DC server connection first
                                    if (remoteApiVersion.get() < 0) {
                                        // newer versions have the serverId set => try to extract this first
                                        if (readLine.contains("id:")) {
                                            final Matcher matcher = dcServerIDPattern.matcher(readLine);

                                            if (matcher.find()) {
                                                dcServerId.append(matcher.group(1));
                                            }
                                        }

                                        // look for valid server id or old style 'open xchange documentconverter' string parts
                                        if ((DocumentConverterUtil.DOCUMENTCONVERTER_SERVER_ID.equals(dcServerId.toString()) ||
                                            (readLine.contains("open") && readLine.contains("xchange") && (readLine.contains("documentconverter"))))) {

                                            remoteApiVersion.set(1);
                                        }
                                    }

                                    if (readLine.contains("api:")) {
                                        final Matcher matcher = apiVersionPattern.matcher(readLine);

                                        if (matcher.find()) {
                                            remoteApiVersion.set(Integer.parseInt(matcher.group(1)));
                                        }
                                    }
                                }
                            } catch (@SuppressWarnings("unused") Exception e) {
                                // #52867: no action intended here
                            }
                        });
                } catch (@SuppressWarnings("unused") Exception e) {
                    // #52867: no action intended here
                }

                if (remoteApiVersion.get() < 0) {
                    // no connection established until now
                    final long curLogTime = System.currentTimeMillis();

                    if ((0 == m_lastLogTime) || ((curLogTime - m_lastLogTime) >= LOG_TIMEOUT)) {
                        DocumentConverterManager.logError("DC client remote connection could not be established => please check remote converter setup",
                            new LogData("remoteurl", m_remoteUrl.toString()),
                            new LogData("async_request", "true"));
                    }

                    m_lastLogTime = curLogTime;
                } else {
                    // connection established
                    DocumentConverterManager.logInfo("DC client established remote connection",
                        new LogData("remoteurl", m_remoteUrl.toString()),
                        new LogData("async_request", "true"));
                }
            }

            m_remoteAPIVersion.set(remoteApiVersion.get());
            m_isRunning.compareAndSet(true, false);
        }
    }

    /**
     * Initializes a new {@link RemoteValidator}.
     * @param serverType
     * @param remoteURL
     */
    RemoteValidator(@NonNull final URL remoteURL) {
        super();

        implSetRemoteURL(remoteURL);
    }

    /**
     * @param remoteURL
     */
    void updateRemoteURL(@NonNull final URL remoteURL) {
        implSetRemoteURL(remoteURL);
    }

    /**
     * @return
     */
    boolean isRemoteServerSupported() {
        return (null != m_remoteUrl);
    }

    /**
     * @return
     */
    int getRemoteAPIVersion() {
        return m_remoteAPIVersion.get();
    }

    /**
     * @return
     */
    synchronized void trigger() {
        if ((null != m_runnable) && m_isRunning.compareAndSet(false, true)) {
            Executors.newSingleThreadExecutor().execute(m_runnable);
        }
    }

    /**
     *
     */
    synchronized void resetConnection() {
        if ((m_remoteAPIVersion.get() >= 1) && !m_isRunning.get()) {
            setConnectionLost();
        }
    }

    /**
     *
     */
    void setConnectionLost() {
        m_remoteAPIVersion.set(-1);

        if (DocumentConverterManager.isLogError()) {
            DocumentConverterManager.logError("DC client lost remote connection:", new LogData("remoteurl", m_remoteUrl.toString()));
        }
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param remoteUrl
     */
    void implSetRemoteURL(@NonNull final URL remoteUrl) {
        if ((null == m_remoteUrl) || !remoteUrl.equals(m_remoteUrl)) {
            m_remoteAPIVersion.set(-1);
            m_isRunning.set(false);
            m_remoteUrl = remoteUrl;

            // trigger new connection validation
            trigger();
        }
    }

    // - Static members --------------------------------------------------------

    protected static Pattern apiVersionPattern = Pattern.compile("api\\:\\s*v([0-9]+)");
    protected static Pattern dcServerIDPattern = Pattern.compile("id\\:\\s*(([0-9a-f]+\\-){4}[0-9a-f]+)");

    // - Members ---------------------------------------------------------------

    protected RemoteValidatorRunnable m_runnable = new RemoteValidatorRunnable();

    protected AtomicInteger m_remoteAPIVersion = new AtomicInteger(-1);

    protected AtomicBoolean m_isRunning = new AtomicBoolean(false);

    protected URL m_remoteUrl = null;

    protected long m_lastLogTime = 0;
}
