/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.custom.base;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableTable;
import com.openexchange.annotation.Nullable;

/**
 * Utility functions for dealing with @Null and @NonNull values.
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.2.0
 */
public enum NullUtil {
    ;

    /**
     * Get an SLF4J {@link Logger} without having to ignore the null warning.
     * <p>
     * From its implementation, it is known that {@link LoggerFactory} does not
     * return null values.
     * 
     * @param c the {@link Class} for which to get a {@link Logger}
     * @return a {@link Logger} for the specified {@link Class}
     */
    @SuppressWarnings("null")
    public static Logger logger(final Class<?> c) {
        return LoggerFactory.getLogger(c);
    }
    
    /**
     * Check a value as being non-null.
     * <p>
     * Note that it does perform a null check and that it throws an {@link IllegalArgumentException}
     * whenever the value actually is {@code null}, using the specified {@code name} parameter
     * in the exception message.
     * 
     * @param value the value
     * @param name the name of the parameter, or value
     * @return the non-{@code null} value
     * @throws IllegalArgumentException thrown if the value is null
     */
    public static <X> X notNull(final @Nullable X value, final String name) throws IllegalArgumentException {
        if (value == null) {
            throw new IllegalArgumentException(name + " is null");
        }
        return value;
    }

    /**
     * "Cast" a non-null value as such, when you know for sure it isn't {@code null}.
     * <p>
     * Note that it does perform a null check and that it throws an {@link IllegalArgumentException}
     * whenever the value actually is {@code null}.
     * 
     * @param value the value
     * @return the non-{@code null} value
     * @throws IllegalArgumentException thrown if the value is null
     */    
    public static <X> X notNull(final @Nullable X value) throws IllegalArgumentException {
        if (value == null) {
            throw new IllegalArgumentException("unexpected null value");
        }
        return value;
    }
    
    /**
     * Perform a {@code String.format()} without having to null-check the result.
     * <p>
     * {@link String#format} is known for never returning {@code null} yet Eclipse doesn't know
     * about that and in order to avoid having to null-check the result of that frequently used
     * method, use this one instead.
     * 
     * @param format
     * @param args
     * @return
     */
    public static String f(final String format, final @Nullable Object... args) {
        final String result = String.format(format, args);
        if (result == null) {
            throw new IllegalStateException("result of String.format is null");
        }
        return result;
    }

    /**
     * Return the class name of a {@link Class} or an {@link Object}.
     * @param classOrObject can be a {@link Class} or an {@link Object}
     * @return the class name of the {@link Class} or {@link Object}
     */
    @SuppressWarnings("null")
    public static String className(final Object classOrObject) {
        if (classOrObject.getClass().equals(Class.class)) {
            return ((Class<?>) classOrObject).getName();
        } else {
            return classOrObject.getClass().getName();
        }
    }

    @SuppressWarnings("null")
    public static <X> Optional<X> absent() {
        return Optional.<X>absent();
    }

    @SuppressWarnings("null")
    public static <X> Optional<X> optional(final X value) {
        return Optional.of(value);
    }

    @SuppressWarnings("null")
    public static <X> Optional<X> optionalFromNullable(final @Nullable X value) {
        return Optional.fromNullable(value);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> emptySet() {
        return ImmutableSet.<X>of();
    }

    @SuppressWarnings("null")
    public static <X> ImmutableList<X> emptyList() {
        return ImmutableList.<X>of();
    }
    
    @SuppressWarnings("null")
    public static <K, V> ImmutableMap<K, V> emptyMap() {
        return ImmutableMap.<K, V>of();
    }

    @SuppressWarnings("null")
    public static <R, C, V> ImmutableTable<R, C, V> emptyTable() {
        return ImmutableTable.<R, C, V>of();
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSet(final X element) {
        return ImmutableSet.<X>of(element);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSet(final X e1, final X e2) {
        return ImmutableSet.<X>of(e1, e2);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSet(final X e1, final X e2, final X e3) {
        return ImmutableSet.<X>of(e1, e2, e3);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSet(final X e1, final X e2, final X e3, final X e4) {
        return ImmutableSet.<X>of(e1, e2, e3, e4);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSet(final X e1, final X e2, final X e3, final X e4, final X e5) {
        return ImmutableSet.<X>of(e1, e2, e3, e4, e5);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSet(final X e1, final X e2, final X e3, final X e4, final X e5, final X e6) {
        return ImmutableSet.<X>of(e1, e2, e3, e4, e5, e6);
    }
    
    @SafeVarargs
    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSet(final X e1, final X e2, final X e3, final X e4, final X e5, final X e6, final X... others) {
        return ImmutableSet.<X>of(e1, e2, e3, e4, e5, e6, others);
    }
    
    @SuppressWarnings("null")
    public static <K, V> ImmutableMap<K, V> immutableMap(final K key, final V value) {
        return ImmutableMap.<K, V>of(key, value);
    }

    @SuppressWarnings("null")
    public static <K, V> ImmutableMap<K, V> immutableMap(final K k1, final V v1, final K k2, final V v2) {
        return ImmutableMap.<K, V>of(k1, v1, k2, v2);
    }

    @SuppressWarnings("null")
    public static <K, V> ImmutableMap<K, V> immutableMap(final K k1, final V v1, final K k2, final V v2, final K k3, final V v3) {
        return ImmutableMap.<K, V>of(k1, v1, k2, v2, k3, v3);
    }

    @SuppressWarnings("null")
    public static <K, V> ImmutableMap<K, V> immutableMap(final K k1, final V v1, final K k2, final V v2, final K k3, final V v3, final K k4, final V v4) {
        return ImmutableMap.<K, V>of(k1, v1, k2, v2, k3, v3, k4, v4);
    }

    @SuppressWarnings("null")
    public static <K, V> ImmutableMap<K, V> immutableMap(final K k1, final V v1, final K k2, final V v2, final K k3, final V v3, final K k4, final V v4, final K k5, final V v5) {
        return ImmutableMap.<K, V>of(k1, v1, k2, v2, k3, v3, k4, v4, k5, v5);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X element) {
        return ImmutableList.<X>of(element);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2) {
        return ImmutableList.<X>of(e1, e2);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2, final X e3) {
        return ImmutableList.<X>of(e1, e2, e3);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2, final X e3, final X e4) {
        return ImmutableList.<X>of(e1, e2, e3, e4);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2, final X e3, final X e4, final X e5) {
        return ImmutableList.<X>of(e1, e2, e3, e4, e5);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2, final X e3, final X e4, final X e5, final X e6) {
        return ImmutableList.<X>of(e1, e2, e3, e4, e5, e6);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2, final X e3, final X e4, final X e5, final X e6, final X e7) {
        return ImmutableList.<X>of(e1, e2, e3, e4, e5, e6, e7);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2, final X e3, final X e4, final X e5, final X e6, final X e7, final X e8) {
        return ImmutableList.<X>of(e1, e2, e3, e4, e5, e6, e7, e8);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2, final X e3, final X e4, final X e5, final X e6, final X e7, final X e8, final X e9) {
        return ImmutableList.<X>of(e1, e2, e3, e4, e5, e6, e7, e8, e9);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2, final X e3, final X e4, final X e5, final X e6, final X e7, final X e8, final X e9, final X e10) {
        return ImmutableList.<X>of(e1, e2, e3, e4, e5, e6, e7, e8, e9, e10);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2, final X e3, final X e4, final X e5, final X e6, final X e7, final X e8, final X e9, final X e10, final X e11) {
        return ImmutableList.<X>of(e1, e2, e3, e4, e5, e6, e7, e8, e9, e10, e11);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableList(final X e1, final X e2, final X e3, final X e4, final X e5, final X e6, final X e7, final X e8, final X e9, final X e10, final X e11, final X e12) {
        return ImmutableList.<X>of(e1, e2, e3, e4, e5, e6, e7, e8, e9, e10, e11, e12);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableCopyOf(final Set<X> set) {
        return ImmutableSet.copyOf(set);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSetCopyOf(final Set<X> set) {
        return ImmutableSet.copyOf(set);
    }

    @SafeVarargs
    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSetCopyOf(final X... elements) {
        return ImmutableSet.copyOf(elements);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSetCopyOf(final Iterable<X> iterable) {
        return ImmutableSet.copyOf(iterable);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSetCopyOf(final Iterator<X> iterator) {
        return ImmutableSet.copyOf(iterator);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> immutableSetCopyOf(final Collection<X> collection) {
        return ImmutableSet.copyOf(collection);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableCopyOf(final Collection<X> list) {
        return ImmutableList.copyOf(list);
    }

    @SafeVarargs
    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableCopyOf(final X... elements) {
        return ImmutableList.copyOf(elements);
    }

    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableCopyOf(final Iterator<X> iterator) {
        return ImmutableList.copyOf(iterator);
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableList<X> immutableCopyOf(final Iterable<X> iterable) {
        return ImmutableList.copyOf(iterable);
    }
    
    @SuppressWarnings("null")
    public static <K, V> ImmutableMap<K, V> immutableCopyOf(final Map<K, V> map) {
        return ImmutableMap.copyOf(map);
    }
    
    @SuppressWarnings("null")
    public static <K, V> ImmutableMap<K, V> build(ImmutableMap.Builder<K, V> builder) {
        return builder.build();
    }

    @SuppressWarnings("null")
    public static <X> ImmutableSet<X> build(ImmutableSet.Builder<X> builder) {
        return builder.build();
    }

    @SuppressWarnings("null")
    public static <X> ImmutableList<X> build(ImmutableList.Builder<X> builder) {
        return builder.build();
    }

    @SuppressWarnings("null")
    public static <R, C, V> ImmutableTable<R, C, V> build(ImmutableTable.Builder<R, C, V> builder) {
        return builder.build();
    }
    
    @SuppressWarnings("null")
    public static <X> ImmutableList.Builder<X> listBuilder() {
        return ImmutableList.<X>builder();
    }

    @SuppressWarnings("null")
    public static <X> ImmutableSet.Builder<X> setBuilder() {
        return ImmutableSet.<X>builder();
    }

    @SuppressWarnings("null")
    public static <K, V> ImmutableMap.Builder<K, V> mapBuilder() {
        return ImmutableMap.<K, V>builder();
    }

    @SuppressWarnings("null")
    public static <R, C, V> ImmutableTable.Builder<R, C, V> tableBuilder() {
        return ImmutableTable.<R, C, V>builder();
    }
    
}
