/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.util.custom.config;

import static com.openexchange.util.custom.base.NullUtil.absent;
import static com.openexchange.util.custom.base.NullUtil.emptyList;
import static com.openexchange.util.custom.base.NullUtil.emptySet;
import static com.openexchange.util.custom.base.NullUtil.immutableCopyOf;
import static com.openexchange.util.custom.base.NullUtil.immutableSetCopyOf;
import static com.openexchange.util.custom.base.NullUtil.optional;
import static com.openexchange.util.custom.base.NullUtil.optionalFromNullable;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import javax.mail.internet.idn.IDNA;
import com.google.common.base.CharMatcher;
import com.google.common.base.Charsets;
import com.google.common.base.Optional;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterators;
import com.google.common.io.Closeables;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.ConfigurationService;
import com.openexchange.configuration.ConfigurationExceptionCodes;
import com.openexchange.exception.OXException;
import com.openexchange.util.custom.base.EmptyPropertiesSupplier;


/**
 * A convenient API to retrieve properties from a {@link ConfigurationService}.
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.2.0
 */
public class ConfigReader {
    
    /**
     * Create a {@link ConfigReader}..
     * 
     * @param configurationService the {@link ConfigurationService} instance to use to query the properties
     * @return a {@link ConfigReader} instance
     * @since 1.2.0
     */
    public static ConfigReader of(final @Nullable ConfigurationService configurationService) {
        return new ConfigReader(null, configurationService, null);
    }

    /**
     * Create a {@link ConfigReader} that prefixes all the property names that will be queried.
     * 
     * @param prefix a prefix to prepend to all property names that will be queried using the resulting {@link ConfigReader}
     * @param configurationService the {@link ConfigurationService} instance to use to query the properties
     * @return a {@link ConfigReader} instance
     * @since 1.2.0
     */
    public static ConfigReader of(final String prefix, final @Nullable ConfigurationService configurationService) {
        return new ConfigReader(prefix, configurationService, null);
    }

    /**
     * Create a {@link ConfigReader} that keeps track of the property names that are queried.
     * 
     * @param configurationService the {@link ConfigurationService} instance to use to query the properties
     * @param propertyNameTracker a {@link ConfigTracker} to which all the property names and files that will be queried using
     *  the resulting {@link ConfigReader} will be {@linkplain Set#add added to}
     * @return a {@link ConfigReader} instance
     * @since 1.2.0
     */
    public static ConfigReader of(final @Nullable ConfigurationService configurationService, final ConfigTracker propertyNameTracker) {
        return new ConfigReader(null, configurationService, propertyNameTracker);
    }

    /**
     * Create a {@link ConfigReader} that keeps track of the property names that are queried.
     * 
     * @param configurationService the {@link ConfigurationService} instance to use to query the properties
     * @param propertyNameTracker a {@link Set} to which all the property names that will be queried using the resulting
     *  {@link ConfigReader} will be {@linkplain Set#add added to}
     * @return a {@link ConfigReader} instance
     * @since 1.2.0
     * @deprecated use a {@link ConfigTracker} instead to also keep track of file names
     */
    @Deprecated
    public static ConfigReader of(final @Nullable ConfigurationService configurationService, final Set<String> propertyNameTracker) {
        return new ConfigReader(null, configurationService, new ConfigTracker(propertyNameTracker));
    }
    
    /**
     * Create a {@link ConfigReader} that keeps track of the property names that are queried and prefixes all the
     * propety names that will be queried
     * 
     * @param prefix a prefix to prepend to all property names that will be queried using the resulting {@link ConfigReader}
     * @param configurationService the {@link ConfigurationService} instance to use to query the properties
     * @param propertyNameTracker a {@link ConfigTracker} to which all the property names and files that will be queried using
     *  the resulting {@link ConfigReader} will be {@linkplain Set#add added to}
     * @return a {@link ConfigReader} instance
     * @since 1.2.0
     */
    public static ConfigReader of(final String prefix, final @Nullable ConfigurationService configurationService, final ConfigTracker propertyNameTracker) {
        return new ConfigReader(prefix, configurationService, propertyNameTracker);
    }

    /**
     * Create a {@link ConfigReader} that keeps track of the property names that are queried.
     * 
     * @param prefix a prefix to prepend to all property names that will be queried using the resulting {@link ConfigReader}
     * @param configurationService the {@link ConfigurationService} instance to use to query the properties
     * @param propertyNameTracker a {@link Set} to which all the property names that will be queried using the resulting
     *  {@link ConfigReader} will be {@linkplain Set#add added to}
     * @return a {@link ConfigReader} instance
     * @since 1.2.0
     * @deprecated use a {@link ConfigTracker} instead to also keep track of file names
     */
    @Deprecated
    public static ConfigReader of(final String prefix, final @Nullable ConfigurationService configurationService, final Set<String> propertyNameTracker) {
        return new ConfigReader(prefix, configurationService, new ConfigTracker(propertyNameTracker));
    }
    
    /**
     * Create a {@link ConfigReader} with a new or extended property name prefix.
     * <p>
     * If the {@value newPrefix} starts with {@value "."}, then it will be appended to the
     * existing prefix, if there is one.
     * <p>
     * If {@value newPrefix} does not start with {@value "."} then it will be used as the prefix,
     * regardless of the existing one.
     * <p>
     * If {@value newPrefix} is an empty String, then the resulting {@link ConfigReader} will not
     * use any prefix at all.
     * 
     * @param newPrefix the new or extended prefix to use to look up property names
     * @return a {@link ConfigReader} that shares the {@link ConfigTracker} and {@link ConfigurationService}
     *  but with a different property name prefix
     * @since 1.2.0
     */
    public ConfigReader withPrefix(final String newPrefix) {
        if (newPrefix.startsWith(".")) {
            final String p = prefix; // to make Eclipse null analysis happy, despite the final on prefix
            if (p != null) {
                if (newPrefix.equals(".")) {
                    /*
                     *      new prefix = "."
                     * existing prefix != null
                     * => just use the existing prefix
                     */
                    return new ConfigReader(p, configService, track);
                } else {
                    if (p.endsWith(".")) {
                        /*
                         *      new prefix = ".xyz"
                         * existing prefix = "abc."
                         * => drop the "." from one of them before concatenating
                         */
                        return new ConfigReader(p + newPrefix.substring(1), configService, track);
                    } else {
                        /*
                         *      new prefix = ".xyz"
                         * existing prefix = "abc"
                         * => concatenate
                         */
                        return new ConfigReader(p + newPrefix, configService, track);
                    }
                }
            } else {
                /*
                 *      new prefix = ".xyz"
                 * existing prefix = null
                 * => use the new prefix and ignore the previous one, as the old one is not set,
                 *    but drop the "." at the beginning as it results in an absolute one
                 */
                return new ConfigReader(newPrefix.substring(1), configService, track);
            }
        } else {
            /*
             *      new prefix = "xyz" or "xyz."
             * => use the new prefix and ignore the previous one, as the new one is absolute
             */
            return new ConfigReader(newPrefix, configService, track);
        }
    }
    
    private final @Nullable ConfigurationService configService;
    private final @Nullable String prefix;
    private final @Nullable ConfigTracker track;
    
    private ConfigReader(final @Nullable String prefix, final @Nullable ConfigurationService configurationService, final @Nullable ConfigTracker track) {
        this.configService = configurationService;
        if (prefix == null || prefix.isEmpty()) {
            this.prefix = null;
        } else {
            this.prefix = prefix;
        }
        this.track = track;
    }
    
    private final String n(final String name) {
        final String result = prefix(prefix, name);
        trackProperty(result);
        return result;
    }
    
    private static final class ConfigPropertyPrefixFunction implements Function<String, String> {
        private final @Nullable String prefix;
        public ConfigPropertyPrefixFunction(final @Nullable String prefix) {
            this.prefix = prefix;
        }
        @Override
        @Nullable
        public String apply(@Nullable String input) {
            return input != null ? prefix(prefix, input) : null;
        }
        @Override
        public boolean equals(@Nullable Object obj) {
            return super.equals(obj);
        }
    }
    
    public static final Function<String, String> toPrefixFunction(final @Nullable String prefix) {
        return new ConfigPropertyPrefixFunction(prefix);
    }
    
    public final String prefix(final String name) {
        return prefix(this.prefix, name);
    }
    
    @SuppressWarnings("null")
    public static final String prefix(final @Nullable String prefix, final String name) {
        if (prefix == null) {
            return name;
        }
        if (name.startsWith(".") && prefix.endsWith(".")) {
            return new StringBuilder(prefix.length() + name.length() - 1)
                .append(prefix)
                .append(name.substring(1))
                .toString();
        }
        if (! (name.startsWith(".") || prefix.endsWith("."))) {
            return new StringBuilder(prefix.length() + name.length() + 1)
                .append(prefix)
                .append(".")
                .append(name)
                .toString();
        }
        return new StringBuilder(prefix.length() + name.length())
            .append(prefix)
            .append(name)
            .toString();
    }
    
    private ConfigReader trackProperty(final @Nullable String propertyName) {
        if (propertyName != null && track != null) {
            track.trackProperty(propertyName);
        }
        return this;
    }

    private ConfigReader trackFile(final @Nullable String fileName) {
        if (fileName != null && track != null) {
            track.trackConfigFileName(fileName);
        }
        return this;
    }
    
    public final class MandatoryValueReader {
        public String str(final String name) throws OXException {
            final String n = n(name);
            final ConfigurationService cs = configService;
            if (cs == null) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create("no ConfigurationService");
            }
            String str = cs.getProperty(n);
            if (str == null) {
                throw ConfigurationExceptionCodes.PROPERTY_MISSING.create(n);
            }
            str = str.replaceAll("^\\s*", "").replaceAll("\\s*$", "");
            if (str.isEmpty()) {
                throw ConfigurationExceptionCodes.PROPERTY_MISSING.create(n);
            }
            return str;
        }
        
        public String str(final String name, final String defaultValue) {
            final String n = n(name);
            final ConfigurationService cs = configService;
            if (cs == null) {
                return defaultValue;
            }
            String str = cs.getProperty(n);
            if (str == null) {
                return defaultValue;
            }
            str = str.replaceAll("^\\s*", "").replaceAll("\\s*$", "");
            if (str.isEmpty()) {
                return defaultValue;
            }
            return str;
        }
        
        public long longNum(final String name, final long defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (! str.isPresent()) {
                return defaultValue;
            }
            try {
                return Long.parseLong(str.get());
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }

        public long longNum(final String name) throws OXException {
            final String str = mandatory.str(name);
            try {
                return Long.parseLong(str);
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }
        
        public int intNum(final String name, final int defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (! str.isPresent()) {
                return defaultValue;
            }
            try {
                return Integer.parseInt(str.get());
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }

        public int intNum(final String name) throws OXException {
            final String str = mandatory.str(name);
            try {
                return Integer.parseInt(str);
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }

        public int positiveIntNum(final String name) throws OXException {
            final int i = mandatory.intNum(name);
            if (i < 0) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s must be a positive numeric value", n(name)));
            } else {
                return i;
            }
        }
        
        public boolean bool(final String name, final boolean defaultValue) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return defaultValue;
            }
            final String str = o.get();
            if (str.equalsIgnoreCase("true")) {
                return true;
            }
            if (str.equalsIgnoreCase("false")) {
                return false;
            }
            throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a boolean (true/false)", n(name)));
        }

        public boolean bool(final String name) throws OXException {
            final String str = mandatory.str(name);
            if (str.equalsIgnoreCase("true")) {
                return true;
            }
            if (str.equalsIgnoreCase("false")) {
                return false;
            }
            throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a boolean (true/false)", n(name)));
        }
        
        public String hostname(final String name, final String defaultValue) {
            final String str = IDNA.toASCII(mandatory.str(name, defaultValue));
            if (str != null) {
                return str;
            } else {
                return defaultValue;
            }
        }

        public String hostname(final String name) throws OXException {
            final String str = IDNA.toASCII(mandatory.str(name));
            if (str != null) {
                return str;
            } else {
                throw ConfigurationExceptionCodes.PROPERTY_MISSING.create(n(name));
            }
        }
        
        public int port(final String name) throws OXException {
            final int value = mandatory.intNum(name);
            if (value <= 0 || value > 65535) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s does not contain a valid port number", n(name)));
            } else {
                return value;
            }
        }

        public int port(final String name, final int defaultValue) throws OXException {
            final Optional<Integer> o = optional.intNum(name);
            if (! o.isPresent()) {
                return defaultValue;
            }
            final int value = o.get();
            if (value <= 0 || value > 65535) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s does not contain a valid port number", n(name)));
            } else {
                return value;
            }
        }
        
        public Charset charset(final String name, final Charset defaultValue) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return defaultValue;
            } else {
                try {
                    final Charset cs = Charset.forName(o.get());
                    if (cs != null) {
                        return cs;
                    } else {
                        return defaultValue;
                    }
                } catch (final Exception e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("%s does not contain a valid charset: %s",
                        n(name), e.getMessage()));
                }
            }
        }

        public Charset charset(final String name) throws OXException {
            final String str = mandatory.str(name);
            final @Nullable Charset result;
            try {
                result = Charset.forName(str);
            } catch (final Exception e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("%s does not contain a valid charset: %s",
                    n(name), e.getMessage()));
            }
            if (result != null) {
                return result;
            } else {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s does not contain a valid charset", n(name)));
            }
        }
        
        public ImmutableList<String> strList(final String name, final CharMatcher split) throws OXException {
            final String str = mandatory.str(name, "");
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(str);
            return immutableCopyOf(iter);
        }

        public ImmutableList<String> strList(final String name) throws OXException {
            return mandatory.strList(name, DEFAULT_SPLIT_PATTERN);
        }
        
        public ImmutableSet<String> strSet(final String name, final CharMatcher split) throws OXException {
            final String str = mandatory.str(name, "");
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(str);
            return immutableSetCopyOf(iter);
        }

        public ImmutableSet<String> strSet(final String name) throws OXException {
            return mandatory.strSet(name, DEFAULT_SPLIT_PATTERN);
        }

        public URI uri(String name) throws OXException {
            final String str = mandatory.str(name);
            try {
                return new URI(str);
            } catch (final URISyntaxException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid URI", n(name)));
            }
        }

        public URI uri(String name, URI defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                try {
                    return new URI(str.get());
                } catch (final URISyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid URI", n(name)));
                }
            } else {
                return defaultValue;
            }
        }
        
        public URI uri(String name, String defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                try {
                    return new URI(str.get());
                } catch (final URISyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid URI", n(name)));
                }
            } else {
                try {
                    return new URI(defaultValue);
                } catch (final URISyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the default value of %s is not a valid URI", n(name)));
                }
            }
        }
        
        public Pattern pattern(String name) throws OXException {
            final String str = mandatory.str(name);
            final Pattern result;
            try {
                result = Pattern.compile(str);
            } catch (PatternSyntaxException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
            }
            if (result != null) {
                return result;
            } else {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
            }
        }
        
        public Pattern pattern(String name, Pattern defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get());
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                return defaultValue;
            }
        }

        public Pattern pattern(String name, int flags, Pattern defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get(), flags);
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                return defaultValue;
            }
        }
        
        public Pattern pattern(String name, String defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get());
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                final Pattern result;
                try {
                    result = Pattern.compile(defaultValue);
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the default value for %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the default value for %s is not a valid regular expression (null result)", n(name)));
                }
            }
        }
        
        public Pattern pattern(String name, int flags, String defaultValue) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get(), flags);
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                final Pattern result;
                try {
                    result = Pattern.compile(defaultValue, flags);
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the default value for %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return result;
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the default value for %s is not a valid regular expression (null result)", n(name)));
                }
            }
        }
        
        public Pattern pattern(String name, int flags) throws OXException {
            final String str = mandatory.str(name);
            final Pattern result;
            try {
                result = Pattern.compile(str, flags);
            } catch (PatternSyntaxException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
            }
            if (result != null) {
                return result;
            } else {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
            }
        }
        
        public final class MandatoryPropertiesLoader {
            @SuppressWarnings("null")
            public Properties file(final String filename) throws OXException {
                return file(filename, Charsets.ISO_8859_1); // ISO_8859_1 is the default for .properties file as of the Java specification
            }

            public Properties file(final String filename, final Charset charset) throws OXException {
                File file = new File(filename);
                if (! file.isAbsolute()) {
                    final ConfigurationService cs = configService;
                    if (cs == null) {
                        throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create("no ConfigurationService");
                    }
                    file = cs.getFileByName(filename);
                    trackFile(filename);
                    if (file == null) {
                        throw ConfigurationExceptionCodes.FILE_NOT_FOUND.create(filename);
                    }
                }
                if (! file.exists()) {
                    throw ConfigurationExceptionCodes.FILE_NOT_FOUND.create(filename);
                }
                final Properties result = new Properties();
                {
                    if (file.getName().endsWith(".xml")) {
                        InputStream is = null;
                        try {
                            is = new FileInputStream(file);
                            result.loadFromXML(is);
                        } catch (final Exception e) {
                            throw ConfigurationExceptionCodes.IO_ERROR.create(e, String.format("unable to load properties from XML file '%s': %s",
                                file.getAbsolutePath(), e.getMessage()));
                        } finally {
                            if (is != null) {
                                Closeables.closeQuietly(is);
                            }
                        }
                    } else {
                        Reader r = null;
                        try {
                            r = new InputStreamReader(new FileInputStream(file), charset);
                            result.load(r);
                        } catch (final Exception e) {
                            throw ConfigurationExceptionCodes.IO_ERROR.create(e, String.format("unable to load properties from %s properties file '%s': %s",
                                charset.displayName(), file.getAbsolutePath(), e.getMessage()));
                        } finally {
                            if (r != null) {
                                Closeables.closeQuietly(r);
                            }
                        }
                    }
                }
                return result;
            }
            
            @SuppressWarnings("null")
            public Properties file(final String filename, final Map<String, ?> defaults) throws OXException {
                return file(filename, Charsets.ISO_8859_1, defaults); // ISO_8859_1 is the default for .properties file as of the Java specification
            }
            
            public Properties file(final String filename, final Charset charset, final Map<String, ?> defaults) throws OXException {
                final Properties defaultsProps = new Properties();
                defaultsProps.putAll(defaults);
                return file(filename, charset, defaultsProps);
            }
            
            @SuppressWarnings("null")
            public Properties file(final String filename, final Properties defaults) throws OXException {
                return file(filename, Charsets.ISO_8859_1, defaults); // ISO_8859_1 is the default for .properties file as of the Java specification
            }
            
            public Properties file(final String filename, final Charset charset, final Properties defaults) throws OXException {
                File file = new File(filename);
                if (! file.isAbsolute()) {
                    final ConfigurationService cs = configService;
                    if (cs == null) {
                        return defaults;
                    }
                    file = cs.getFileByName(filename);
                    trackFile(filename);
                    if (file == null) {
                        return defaults;
                    }
                }
                if (! file.exists()) {
                    return defaults;
                }
                final Properties result = new Properties();
                {
                    if (file.getName().endsWith(".xml")) {
                        InputStream is = null;
                        try {
                            is = new FileInputStream(file);
                            result.loadFromXML(is);
                        } catch (final Exception e) {
                            throw ConfigurationExceptionCodes.IO_ERROR.create(e, String.format("unable to load properties from XML file '%s': %s",
                                file.getAbsolutePath(), e.getMessage()));
                        } finally {
                            if (is != null) {
                                Closeables.closeQuietly(is);
                            }
                        }
                    } else {
                        Reader r = null;
                        try {
                            r = new InputStreamReader(new FileInputStream(file), charset);
                            result.load(r);
                        } catch (final Exception e) {
                            throw ConfigurationExceptionCodes.IO_ERROR.create(e, String.format("unable to load properties from %s properties file '%s': %s",
                                charset.displayName(), file.getAbsolutePath(), e.getMessage()));
                        } finally {
                            if (r != null) {
                                Closeables.closeQuietly(r);
                            }
                        }
                    }
                }
                return result;
            }
        }
        public final MandatoryPropertiesLoader properties = new MandatoryPropertiesLoader();
        
    };
    public final MandatoryValueReader mandatory = new MandatoryValueReader();
    public final MandatoryValueReader m = mandatory;

    public final class OptionalValueReader {
        public Optional<String> str(final String name) {
            final String n = n(name);
            final ConfigurationService cs = configService;
            if (cs == null) {
                return absent();
            }
            String str = cs.getProperty(n);
            if (str == null) {
                return absent();
            }
            str = str.replaceAll("^\\s*", "").replaceAll("\\s*$", "");
            if (str.isEmpty()) {
                return absent();
            }
            return optional(str);
        }
        
        public Optional<Long> longNum(final String name) throws OXException {
            final Optional<String> str = optional.str(name);
            if (! str.isPresent()) {
                return absent();
            }
            try {
                return optionalFromNullable(Long.parseLong(str.get()));
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }

        public Optional<Integer> intNum(final String name) throws OXException {
            final Optional<String> str = optional.str(name);
            if (! str.isPresent()) {
                return absent();
            }
            try {
                return optionalFromNullable(Integer.parseInt(str.get()));
            } catch (final NumberFormatException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a numeric value", n(name)));
            }
        }

        public Optional<Integer> positiveIntNum(final String name) throws OXException {
            final Optional<Integer> o = optional.intNum(name);
            if (o.isPresent()) {
                final int value = o.get();
                if (value < 0) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s must be a positive numeric value", n(name)));
                }
            }
            return o;
        }
        
        public Optional<Boolean> bool(final String name) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return absent();
            }
            final String str = o.get();
            if (str.equalsIgnoreCase("true")) {
                return optional(true);
            }
            if (str.equalsIgnoreCase("false")) {
                return optional(false);
            }
            throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s is not a boolean (true/false)", n(name)));
        }
        
        public Optional<String> hostname(final String name) {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return absent();
            }
            return optionalFromNullable(IDNA.toASCII(o.get()));
        }
        
        public Optional<Integer> port(final String name) throws OXException {
            final Optional<Integer> i = optional.intNum(name);
            if (! i.isPresent()) {
                return i;
            }
            final int value = i.get();
            if (value <= 0 || value > 65535) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("%s does not contain a valid port number", n(name)));
            } else {
                return i;
            }
        }
        
        public Optional<Charset> charset(final String name) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return absent();
            } else {
                try {
                    return optionalFromNullable(Charset.forName(o.get()));
                } catch (final Exception e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("%s does not contain a valid charset: %s",
                        n(name), e.getMessage()));
                }
            }
        }
        
        public ImmutableList<String> strList(final String name, final CharMatcher split, final Iterator<String> defaultValues) {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return immutableCopyOf(defaultValues);
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableCopyOf(iter);
        }

        public ImmutableList<String> strList(final String name, final Iterator<String> defaultValues) {
            return optional.strList(name, DEFAULT_SPLIT_PATTERN, defaultValues);
        }
        
        @SuppressWarnings("null")
        public ImmutableList<String> strList(final String name, final CharMatcher split, final String... defaultValues) {
            return optional.strList(name, split, Iterators.forArray(defaultValues));
        }
        
        @SuppressWarnings("null")
        public ImmutableList<String> strList(final String name, final String... defaultValues) {
            return optional.strList(name, DEFAULT_SPLIT_PATTERN, Iterators.forArray(defaultValues));
        }
        
        public ImmutableList<String> strList(final String name, final CharMatcher split) {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return emptyList();
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableCopyOf(iter);
        }

        public ImmutableList<String> strList(final String name) {
            return optional.strList(name, DEFAULT_SPLIT_PATTERN);
        }
        
        public ImmutableList<String> strList(final String name, final CharMatcher split, final Iterable<String> defaultValues) {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return immutableCopyOf(defaultValues);
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableCopyOf(iter);
        }

        public ImmutableList<String> strList(final String name, final Iterable<String> defaultValues) {
            return optional.strList(name, DEFAULT_SPLIT_PATTERN, defaultValues);
        }
        
        public ImmutableSet<String> strSet(final String name, final CharMatcher split, final Iterator<String> defaultValues) {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return immutableSetCopyOf(defaultValues);
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableSetCopyOf(iter);
        }

        public ImmutableSet<String> strSet(final String name, final Iterator<String> defaultValues) {
            return optional.strSet(name, DEFAULT_SPLIT_PATTERN, defaultValues);
        }
        
        @SuppressWarnings("null")
        public ImmutableSet<String> strSet(final String name, final CharMatcher split, final String... defaultValues) {
            return optional.strSet(name, split, Iterators.forArray(defaultValues));
        }
        
        @SuppressWarnings("null")
        public ImmutableSet<String> strSet(final String name, final String... defaultValues) {
            return optional.strSet(name, DEFAULT_SPLIT_PATTERN, Iterators.forArray(defaultValues));
        }
        
        public ImmutableSet<String> strSet(final String name, final CharMatcher split) {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return emptySet();
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableSetCopyOf(iter);
        }

        public ImmutableSet<String> strSet(final String name) {
            return optional.strSet(name, DEFAULT_SPLIT_PATTERN);
        }
        
        public ImmutableSet<String> strSet(final String name, final CharMatcher split, final Iterable<String> defaultValues) {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return immutableSetCopyOf(defaultValues);
            }
            @SuppressWarnings("null")
            final @NonNull Iterable<String> iter = Splitter.on(split).omitEmptyStrings().split(o.get());
            return immutableSetCopyOf(iter);
        }

        public ImmutableSet<String> strSet(final String name, final Iterable<String> defaultValues) {
            return optional.strSet(name, DEFAULT_SPLIT_PATTERN, defaultValues);
        }
        
        public Optional<URI> uri(String name) throws OXException {
            final Optional<String> o = optional.str(name);
            if (! o.isPresent()) {
                return absent();
            }
            try {
                return optional(new URI(o.get()));
            } catch (final URISyntaxException e) {
                throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid URI", n(name)));
            }
        }
        
        public Optional<Pattern> pattern(String name) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get());
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return optional(result);
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                return absent();
            }
        }
        
        public Optional<Pattern> pattern(String name, int flags) throws OXException {
            final Optional<String> str = optional.str(name);
            if (str.isPresent()) {
                final Pattern result;
                try {
                    result = Pattern.compile(str.get(), flags);
                } catch (PatternSyntaxException e) {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(e, String.format("the value of %s is not a valid regular expression", n(name)));
                }
                if (result != null) {
                    return optional(result);
                } else {
                    throw ConfigurationExceptionCodes.INVALID_CONFIGURATION.create(String.format("the value of %s is not a valid regular expression (null result)", n(name)));
                }
            } else {
                return absent();
            }
        }
        
        public final class OptionalPropertiesLoader {
            @SuppressWarnings("null")
            public Optional<Properties> file(final String filename) throws OXException {
                return file(filename, Charsets.ISO_8859_1); // ISO_8859_1 is the default for .properties file as of the Java specification
            }

            public Optional<Properties> file(final String filename, final Charset charset) throws OXException {
                File file = new File(filename);
                if (! file.isAbsolute()) {
                    final ConfigurationService cs = configService;
                    if (cs == null) {
                        return absent();
                    }
                    file = cs.getFileByName(filename);
                    trackFile(filename);
                    if (file == null) {
                        return absent();
                    }
                }
                if (! file.exists()) {
                    return absent();
                }
                final Properties result = new Properties();
                {
                    if (file.getName().endsWith(".xml")) {
                        InputStream is = null;
                        try {
                            is = new FileInputStream(file);
                            result.loadFromXML(is);
                        } catch (final Exception e) {
                            throw ConfigurationExceptionCodes.IO_ERROR.create(e, String.format("unable to load properties from XML file '%s': %s",
                                file.getAbsolutePath(), e.getMessage()));
                        } finally {
                            if (is != null) {
                                Closeables.closeQuietly(is);
                            }
                        }
                    } else {
                        Reader r = null;
                        try {
                            r = new InputStreamReader(new FileInputStream(file), charset);
                            result.load(r);
                        } catch (final Exception e) {
                            throw ConfigurationExceptionCodes.IO_ERROR.create(e, String.format("unable to load properties from %s properties file '%s': %s",
                                charset.displayName(), file.getAbsolutePath(), e.getMessage()));
                        } finally {
                            if (r != null) {
                                Closeables.closeQuietly(r);
                            }
                        }
                    }
                }
                return optional(result);
            }
            
            @SuppressWarnings("null")
            public Properties fileOrEmpty(final String filename) throws OXException {
                return file(filename).or(EmptyPropertiesSupplier.INSTANCE);
            }
            
            @SuppressWarnings("null")
            public Properties fileOrEmpty(final String filename, final Charset charset) throws OXException {
                return file(filename, charset).or(EmptyPropertiesSupplier.INSTANCE);
            }
        }
        public final OptionalPropertiesLoader properties = new OptionalPropertiesLoader();
    };
    public final OptionalValueReader optional = new OptionalValueReader();
    public final OptionalValueReader o = optional;

    @SuppressWarnings("null")
    public static final CharMatcher DEFAULT_SPLIT_PATTERN = CharMatcher.anyOf("\r\n;, ");

}
