/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.spellchecker.impl.client;

import java.util.concurrent.atomic.AtomicBoolean;
import com.openexchange.office.tools.annotation.NonNull;
import com.openexchange.office.tools.annotation.Nullable;

/**
 * {@link VersionValidator}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.4
 */
public class VersionValidator {

    final public  static int VERSION_INVALID = -1;

    /**
     * {@link GetVersionFunc}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.4
     */
    public interface GetVersionHandler {
        int getVersion();
    }

    /**
     * VersionChangedHandler
     */
    public interface VersionChangedHandler {
        void versionChanged(int oldVersion, int newVersion);
    }

    /**
     * Initializes a new {@link VersionValidator}.
     */
    public VersionValidator(
        @NonNull final GetVersionHandler getVersionHandler,
        @Nullable final VersionChangedHandler gotValidVersionHandler,
        @Nullable final VersionChangedHandler lostValidVersionHandler) {

        super();

        m_getVersionHandler = getVersionHandler;
        m_gotValidVersionHandler = gotValidVersionHandler;
        m_lostValidVersionHandler = lostValidVersionHandler;
    }

    /**
     * @param enabled
     * @return
     */
    public void setEnabled(final boolean enabled) {
        m_enabled.set(enabled);
    }

    /**
     * @param enabled
     * @return
     */
    public boolean isEnabled() {
        return m_enabled.get();
    }

    /**
     * @return
     */
    public int getVersion() {
        if (!isEnabled()) {
            return VERSION_INVALID;
        }

        synchronized (this) {
            final int oldVersion = m_version;

            if (VERSION_INVALID == oldVersion) {
                final int currentVersion = Math.max(m_getVersionHandler.getVersion(), VERSION_INVALID);

                if (currentVersion > VERSION_INVALID) {
                    m_version = currentVersion;

                    if (null != m_gotValidVersionHandler) {
                        m_gotValidVersionHandler.versionChanged(oldVersion, currentVersion);
                    }
                }
            }

            return m_version;
        }
    }

    /**
     *
     */
    synchronized public void setVersionInvalid(boolean notifyLostValidVersion) {
        final int oldVersion = m_version;

        if (oldVersion > VERSION_INVALID) {
            m_version = VERSION_INVALID;

            if (notifyLostValidVersion && (null != m_lostValidVersionHandler)) {
                m_lostValidVersionHandler.versionChanged(oldVersion, VERSION_INVALID);
            }
        }
    }

    /**
     *
     */
    public void validate() {
        // just call getVersion to initiate validation process
        getVersion();
    }

    /**
     * @return
     */
    public boolean isValid() {
        return (getVersion() > VERSION_INVALID);
    }

    // - Members ---------------------------------------------------------------

    final private AtomicBoolean m_enabled = new AtomicBoolean(true);

    final private GetVersionHandler m_getVersionHandler;

    final private VersionChangedHandler m_gotValidVersionHandler;

    final private VersionChangedHandler m_lostValidVersionHandler;

    private int m_version = VERSION_INVALID;


}
