/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.spellchecker.impl;


import java.util.concurrent.atomic.AtomicBoolean;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.openexchange.config.ConfigurationService;
import com.openexchange.office.spellchecker.api.ISpellChecker;
import com.openexchange.office.spellchecker.api.IValidator;
import com.openexchange.office.spellchecker.impl.client.SpellCheckerClient;
import com.openexchange.office.spellchecker.impl.hunspell.SpellCheckerHunspell;
import com.openexchange.office.tools.annotation.NonNull;
import com.openexchange.office.tools.annotation.RegisteredService;

/**
 * {@link SpellChecker}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.4
 */
@Service
@RegisteredService(registeredClass = ISpellChecker.class)
public class SpellChecker implements InitializingBean, DisposableBean, IValidator, ISpellChecker {

    /**
     * Initializes a new {@link SpellCheckerClient}
     */
    public SpellChecker() {
        super();
    }

    // - InitializingBean ------------------------------------------------------

    /**
     *
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        m_enabled.set(m_configService.getBoolProperty("io.ox/office//module/spellingEnabled", false));

        LOG.info(new StringBuilder(128).append("SpellChecking is ").append(m_enabled.get() ? "enabled" : "disabled").append(" by configuration").toString());
    }

    // - DisposableBean --------------------------------------------------------

    /**
     *
     */
    @Override
    public void destroy() throws Exception {
        if (m_enabled.compareAndSet(true, false)) {
            // TODO (KA)
        }
    }

    // - IValidator ------------------------------------------------------------

    /**
    *
    */
    @Override
    public boolean isValid() {
        return (m_enabled.get() && (implGetValidSpellChecker() != null));
    }

    // - ISpellChecker ---------------------------------------------------------

    /**
     *
     */
    @Override
    public JSONObject checkSpelling(@NonNull final JSONObject jsonParams) throws Exception {
        final ISpellChecker spellChecker = implGetValidSpellChecker();

        return ((null != spellChecker) ? spellChecker.checkSpelling(jsonParams) : implGetErrorJsonObject());
    }

    /**
     *
     */
    @Override
    public JSONObject checkParagraphSpelling(@NonNull final JSONObject jsonParams) throws Exception {
        ISpellChecker spellChecker = implGetValidSpellChecker();

        return ((null != spellChecker) ? spellChecker.checkParagraphSpelling(jsonParams) : implGetErrorJsonObject());
    }

    /**
     *
     */
    @Override
    public JSONObject suggestReplacements(@NonNull final JSONObject jsonParams) throws Exception {
        final ISpellChecker spellChecker = implGetValidSpellChecker();

        return ((null != spellChecker) ? spellChecker.suggestReplacements(jsonParams) : implGetErrorJsonObject());
    }

    /**
     *
     */
    @Override
    public JSONObject isMisspelled(@NonNull final JSONObject jsonParams) throws Exception {
        final ISpellChecker spellChecker = implGetValidSpellChecker();

        return ((null != spellChecker) ? spellChecker.isMisspelled(jsonParams) : implGetErrorJsonObject());
    }

    /**
     *
     */
    @Override
    public JSONObject getSupportedLocales() throws Exception {
        final ISpellChecker spellChecker = implGetValidSpellChecker();

        return ((null != spellChecker) ? spellChecker.getSupportedLocales() : implGetErrorJsonObject());
    }

    // - Implementation --------------------------------------------------------

    /**
     * implGetValidSpellChecker
     *
     * @return a valid ISpellChecker instance or <code>null</code>
     */
    private ISpellChecker implGetValidSpellChecker() {
        return m_spellCheckerClient.isValid() ? m_spellCheckerClient : m_spellCheckerHunspell.isValid() ? m_spellCheckerHunspell : null;
    }

    /**
     * @return
     */
    private JSONObject implGetErrorJsonObject() {
        return new JSONObject();
    }

    // - Static Members --------------------------------------------------------

    final private static Logger LOG = LoggerFactory.getLogger(SpellChecker.class);

    // - Members ---------------------------------------------------------------

    final private AtomicBoolean m_enabled = new AtomicBoolean(false);

    @Autowired
    private ConfigurationService m_configService;

    @Autowired
    private SpellCheckerClient m_spellCheckerClient;

    @Autowired
    private SpellCheckerHunspell m_spellCheckerHunspell;
}
